/**********************************************************
 *
 * mp3splt-gtk -- utility based on mp3splt,
 *                for mp3/ogg splitting without decoding
 *
 * Copyright: (C) 2005-2006 Munteanu Alexandru
 * Contact: io_alex_2002@yahoo.fr
 *
 * http://mp3splt.sourceforge.net/
 *
 *********************************************************/

/**********************************************************
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,
 * USA.
 *
 *********************************************************/

/**********************************************************
 * Filename: tree_tab.c
 *
 * this file is used for the Splitpoints tab
 * (for the splitpoints table)
 *
 *********************************************************/

#include <gtk/gtk.h>
#include <libmp3splt/mp3splt.h>
#include <glib/gi18n.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>

#ifdef __WIN32__
#define usleep(x) Sleep(x*1000)
#endif

#include "util.h"
#include "player.h"
#include "tree_tab.h"
#include "main_win.h"
#include "utilities.h"
#include "player_tab.h"
#include "mp3splt-gtk.h"
#include "splitted_files.h"

//tree column enumeration
enum
  {
    /* the filename of the split */
    COL_DESCRIPTION,
    /* splitpoints, minutes, seconds,hundr */
    COL_MINUTES,
    COL_SECONDS,
    COL_HUNDR_SECS,
    /* length of the splitted song */
    COL_NUMBER,
    /* preview button */
    COL_PREVIEW,
    /* split preview button */
    COL_SPLIT_PREVIEW,
    NUM_COLUMNS = 7
  };

//splitpoints
GArray *splitpoints = NULL;
//minutes and seconds reflected by the spinners
gint spin_mins = 0;
gint spin_secs = 0;
gint spin_hundr_secs = 0;
//current description
gchar current_description[255] = "description here";

//splitpoints elements number
gint splitnumber = 0;

//buttons for adding and removing rows
GtkWidget *add_button;
GtkWidget *remove_all_button;
GtkWidget *remove_row_button;

//handle box for detaching window
GtkWidget *handle_box;

//our tree view, used global for 
//the update_*_from_spinner
GtkTreeView *tree_view;

//minutes and seconds spinner
GtkWidget *spinner_minutes;
GtkWidget *spinner_seconds;
GtkWidget *spinner_hundr_secs;

//if we have a preview, preview = TRUE
gboolean quick_preview = FALSE;
//the end of the preview
//preview_end_position = -1 means don't stop until the
//end of the song
gint quick_preview_end_splitpoint = -1;

//for the preview
gint this_row;
//the position transmitted to the player
gint preview_start_position;
gint preview_start_splitpoint = -1;

//if we add a new splitpoint at the left and we are currently
//previewing, we should increment quick_preview start and end
gboolean new_left_splitpoint_added = FALSE;

//the splitpoint selected, used when we move the splitpoint
//we have selected
gint first_splitpoint_selected = -1;

//options for splitting
extern GtkWidget *entry;
extern gint timer_active;
extern gint player_seconds, player_minutes,
  player_hundr_secs;
extern splt_state *the_state;
extern int selected_player;
//the percent progress bar
extern GtkWidget *percent_progress_bar;
extern gfloat current_time;


//updates add button, wether the spinners splitpoint is already
//in the table or not
void update_add_button()
{
  if (check_if_splitpoint_exists(tree_view,
				 spin_mins, 
				 spin_secs,
				 spin_hundr_secs,-1))
    {
      gtk_widget_set_sensitive(GTK_WIDGET(add_button), TRUE);
    }
  else
    {
      gtk_widget_set_sensitive(GTK_WIDGET(add_button), FALSE);
    }
}

//updates the minutes from the spinner
void update_minutes_from_spinner( GtkWidget *widget,
				  gpointer   data )
{
  spin_mins = 
    gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(spinner_minutes));
  update_add_button();
}

//updates the seconds from the spinner
void update_seconds_from_spinner( GtkWidget *widget,
				  gpointer   data )
{
  spin_secs = 
    gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(spinner_seconds));
  update_add_button();
}

//updates the hundredth of seconds for the spinner
void update_hundr_secs_from_spinner( GtkWidget *widget,
				     gpointer   data )
{
  spin_hundr_secs = 
    gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(spinner_hundr_secs));
  update_add_button();
}

//creates the model for the tree, gtkliststore
GtkTreeModel *create_model()
{
  GtkListStore *model;
  model = gtk_list_store_new (NUM_COLUMNS,
			      G_TYPE_STRING,
			      G_TYPE_INT,
			      G_TYPE_INT, 
			      G_TYPE_INT, 
			      G_TYPE_STRING,
			      GDK_TYPE_PIXBUF,
			      GDK_TYPE_PIXBUF);
  return GTK_TREE_MODEL (model);
}

//order the number column
void order_length_column(GtkTreeView *tree_view)
{
  //number to be put in the number column
  //we start at 0
  gint number = 0;
  GtkTreeIter iter;
  GtkTreeIter iter2;
  GtkTreePath *path = NULL,*path2 = NULL;
  GtkTreeModel *model;
  
  //data from the current line
  gint line_mins,line_secs,line_hundr;
  //data from the line+1
  gint line1_mins,line1_secs,line1_hundr;
  
  //final result
  gint result_mins = 0,
    result_secs = 0,result_hundr = 0;
  
  //the new string that we write
  gchar new_length_string[30];
  
  model = gtk_tree_view_get_model(tree_view);
  
  //for each column
  for(number = 0;number < splitnumber; number++)
    {
      path = 
	gtk_tree_path_new_from_indices (number ,-1);
      //get the iter correspondig to the path
      gtk_tree_model_get_iter(model, &iter, path);
      
      //if not the last
      if (number != splitnumber-1)
	{
	  path2 = 
	    gtk_tree_path_new_from_indices (number+1 ,-1);
	  //get the iter correspondig to the path
	  gtk_tree_model_get_iter(model, &iter2, path2);
	  
	  gtk_tree_model_get(GTK_TREE_MODEL(model), &iter,
			     COL_MINUTES, &line_mins,
			     COL_SECONDS, &line_secs,
			     COL_HUNDR_SECS, &line_hundr,
			     -1);
	  gtk_tree_model_get(GTK_TREE_MODEL(model), &iter2,
			     COL_MINUTES, &line1_mins,
			     COL_SECONDS, &line1_secs,
			     COL_HUNDR_SECS, &line1_hundr,
			     -1);
	  
	  //we put the result in result_*
	  result_mins = line1_mins - line_mins;
	  //if the seconds are less than 0
	  if ((result_secs = line1_secs - line_secs) < 0)
	    {
	      //we calculate the right seconds
	      result_secs = 60 - line_secs + line1_secs;
	      result_mins--;
	    }
	  //if the hundreths are less than 0
	  if ((result_hundr = line1_hundr - line_hundr) < 0)
	    {
	      result_hundr = 100 - line_hundr + line1_hundr;
	      result_secs--;
	      if (result_secs < 0)
		{
		  result_mins--;result_secs = 0;		    
		}
	    }
	  
	  //we write the new string
	  g_snprintf(new_length_string,30,
		     "%d:%02d:%02d",result_mins,
		     result_secs,result_hundr);
	  
	  //free memory
	  gtk_tree_path_free(path2);
	}
      else
	{
	  g_snprintf(new_length_string,30,"%s","-");
	}
      
      //free memory
      gtk_tree_path_free(path);
      
      //we put the string in the case
      gtk_list_store_set (GTK_LIST_STORE (model), 
			  &iter,
			  COL_NUMBER, new_length_string,
			  -1);
    }  
}

//checks if splitpoints exists in the table
//and different from current_split
gboolean check_if_splitpoint_exists(GtkTreeView *tree_view,
				    gint minutes, 
				    gint seconds,
				    gint hundr_secs,
				    gint current_split)
{
  GtkTreeModel *model;
  GtkTreeIter iter;
  //minutes and seconds from the column
  gint tree_minutes;
  gint tree_seconds;
  gint tree_hundr_secs;
  
  model = gtk_tree_view_get_model(tree_view);
  //for getting row number
  GtkTreePath *path = NULL;
  gint i;
  
  //if the table is not empty
  //get iter number
  if(gtk_tree_model_get_iter_first(model, &iter))
    {
      gtk_tree_model_get(GTK_TREE_MODEL(model), &iter,
			 COL_MINUTES, &tree_minutes,
			 COL_SECONDS, &tree_seconds,
			 COL_HUNDR_SECS, &tree_hundr_secs,
			 -1);
      
      //supposing we have a finite tree, so it will break somehow
      while(TRUE)
	{
	  gtk_tree_model_get(GTK_TREE_MODEL(model), &iter,
			     COL_MINUTES, &tree_minutes,
			     COL_SECONDS, &tree_seconds,
			     COL_HUNDR_SECS, &tree_hundr_secs,
			     -1);
	  
	  //we get the current line
	  path = gtk_tree_model_get_path(model, &iter);
	  i = gtk_tree_path_get_indices (path)[0];
	  
	  //if we already have the splitpoints in the table, return
	  //FALSE
	  if ((minutes == tree_minutes)
	      && (seconds == tree_seconds)
	      && (hundr_secs == tree_hundr_secs)
	      && (i != current_split))
	    {
	      //free memory
	      gtk_tree_path_free (path);
	      return FALSE;
	    }
	  
	  //free memory
	  gtk_tree_path_free (path);
	  
	  //go to next iter number(row)
	  if(!gtk_tree_model_iter_next(model, &iter))
	    break;
	}
    }
  
  //if everything is ok, 
  //and we have no row containing the splitpoint,
  return TRUE;
}

//checks if description exists
//we dont check the count = number
gboolean check_if_description_exists(gchar *descr,
				     gint number)
{
  GtkTreeModel *model;
  GtkTreeIter iter;
  
  model = gtk_tree_view_get_model(tree_view);
  
  gchar *description;
  //we count the rows
  gint count = 0;
  //if the table is not empty
  //get iter number
  if(gtk_tree_model_get_iter_first(model, &iter))
    {
      gtk_tree_model_get(GTK_TREE_MODEL(model), &iter,
			 COL_DESCRIPTION,&description,
			 -1);
      //freeing memory
      g_free(description);
      
      //supposing we have a finite tree, so it will break somehow
      while(TRUE)
	{
	  gtk_tree_model_get(GTK_TREE_MODEL(model), &iter,
			     COL_DESCRIPTION,&description,
			     -1);
	  
	  //if we already have the description in the table
	  //FALSE
	  if (description != NULL)
	    if (strcmp(descr, description) == 0)  
	      if (count != number)
		{
		  //freeing memory
		  g_free(description);
		  return FALSE;
		}
	  
	  //freeing memory
	  g_free(description);
	  
	  //go to next iter number(row)
	  if(!gtk_tree_model_iter_next(model, &iter))
	    break;
	  
	  count ++;
	}
    }
  
  return TRUE;
}

//returns the first splitpoint selected
gint get_first_splitpoint_selected()
{
  gint splitpoint_selected = -1;
  //we get the selection and change the selected splitpoint on
  //the player
  GtkTreeModel *model;
  model = gtk_tree_view_get_model(tree_view);
  GtkTreeSelection *selection;
  selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(tree_view));
  //our selected list
  GList *selected_list = NULL;
  selected_list = 
    gtk_tree_selection_get_selected_rows(GTK_TREE_SELECTION(selection),
					 &model);
  //if we have something in the selected list
  if (g_list_length(selected_list) > 0)
    {
      //get the first element
      GList *current_element = NULL;
      
      current_element = g_list_first(selected_list);
      //the path
      GtkTreePath *path;
      path = current_element->data;
      splitpoint_selected = gtk_tree_path_get_indices (path)[0];
      
      //we free the selected elements
      g_list_foreach (selected_list, 
		      (GFunc)gtk_tree_path_free, NULL);
      g_list_free (selected_list);
    }
  
  return splitpoint_selected;
}

//row selection event
void row_selection_event()
{
  if(!GTK_WIDGET_SENSITIVE(remove_row_button))
    gtk_widget_set_sensitive(GTK_WIDGET(remove_row_button), TRUE);  
}

//we dont check count = number
void update_current_description(gchar *descr, gint number)
{
  gint ll = 0;
  
  g_snprintf(current_description,255,"%s",descr);
  
  while (ll < splitnumber)
    {
      //if we already have the description
      if(!check_if_description_exists(current_description,
				      number))
	{
	  //we cut the part _* from the string and put
	  //it back
	  gchar *tmp;
	  tmp = strchr(current_description,'_');
	  if (tmp != NULL)
	    {
	      *tmp = '\0';
	    }
	  gchar *temp = g_strdup(current_description);
	  g_snprintf(current_description,255,
		     "%s_%d_",temp,ll);
	  //freeing memory
	  g_free(temp);
	}
      ll++;
    }
}

//returns secs, mins, hundr of secs from a time
//not used for now
void get_hundr_secs_mins_time(gint time_pos, gint *time_hundr,
			      gint *time_secs,gint *time_mins)
{
  *time_hundr = time_pos % 100;
  time_pos = time_pos / 100;
  *time_secs = time_pos % 60;
  time_pos = time_pos / 60;
  *time_mins = time_pos;
}

//selects a splitpoint
void select_splitpoint(gint index)
{
  GtkTreeModel *model;
  GtkTreePath *path;
  GtkTreeIter iter;
  GtkTreeSelection *selection;
  selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(tree_view));
  
  //we get the model
  model = gtk_tree_view_get_model(tree_view);
  //we get the path
  path = gtk_tree_path_new_from_indices (index ,-1);
  //we get iter
  gtk_tree_model_get_iter(model, &iter, path);
  gtk_tree_selection_unselect_all(selection);
  gtk_tree_selection_select_iter(selection,
				 &iter);
  //we free the path
  gtk_tree_path_free(path);
  
  remove_status_message();
}

//removes a splitpoint
//stop preview means we stop preview if necessary
void remove_splitpoint(gint index,gint stop_preview)
{
  //remove values from the splitpoint array
  g_array_remove_index (splitpoints, index);
  
  GtkTreeModel *model;
  GtkTreePath *path;
  GtkTreeIter iter;
  //we get the model
  model = gtk_tree_view_get_model(tree_view);
  //we get the path
  path = gtk_tree_path_new_from_indices (index ,-1);
  //we get iter
  gtk_tree_model_get_iter(model, &iter, path);
  
  //we cancel quick preview if necessary
  if (((index == preview_start_splitpoint) &&
      (stop_preview))||
      ((index == quick_preview_end_splitpoint) &&
      (quick_preview_end_splitpoint == (splitnumber-1))&&
       (stop_preview)))
    {
      cancel_quick_preview_all();
    }

  //remove from list
  gtk_list_store_remove (GTK_LIST_STORE (model), &iter);
  //we free the path
  gtk_tree_path_free(path);
  splitnumber--;
  
  //if we don't have the first selected, we hide remove button
  if (get_first_splitpoint_selected() == -1)
    {
      if(GTK_WIDGET_SENSITIVE(remove_row_button))
	gtk_widget_set_sensitive(GTK_WIDGET(remove_row_button), 
				 FALSE);  
    }
  
  //if we have no more splitpoints, disable buttons
  if (splitnumber == 0)
    {
      //disable remove all button
      if(GTK_WIDGET_SENSITIVE(remove_all_button))
	gtk_widget_set_sensitive(GTK_WIDGET(remove_all_button), FALSE);
    }
  
  remove_status_message();
  order_length_column(tree_view);
  remove_status_message();
  update_add_button();
  check_update_down_progress_bar();
  refresh_drawing_area();
}

//updates a splipoint
void update_splitpoint(gint index, Split_point new_point)
{
  if (check_if_splitpoint_exists(tree_view,
				 new_point.mins, 
				 new_point.secs,
				 new_point.hundr_secs,-1))
    {
      first_splitpoint_selected 
	= get_first_splitpoint_selected();
      //we remove the splitpoint, then we add it
      remove_splitpoint(index,FALSE);
      add_splitpoint(new_point,index);
    }
  else
    {      
      Split_point old_point = 
	g_array_index(splitpoints, Split_point, index);
      //don't put error if we move the same splitpoint
      //on the same place
      if ((new_point.mins == old_point.mins) &&
	  (new_point.secs == old_point.secs) &&
	  (new_point.hundr_secs == old_point.hundr_secs))
	{}
      else
	{
	  //if we already have a equal splitpoint
	  put_status_message((gchar *)_(" error : you already have"
					" the splitpoint in table "));
	}
    }
}

//updates a splitpoint
//i is the position in the GArray with splitpoints
void update_splitpoint_from_time(gint index, gdouble time)
{
  //if we have another splitpoint on the same place
  //we don't add it
  Split_point new_point;
  get_hundr_secs_mins_time((gint)time,
			   &new_point.hundr_secs,
			   &new_point.secs,
			   &new_point.mins);
  update_splitpoint(index, new_point);
}

//event for editing a cell
void cell_edited_event (GtkCellRendererText *cell,
			gchar               *path_string,
			gchar               *new_text,
			gpointer             data)
{
  GtkTreeView *tree_view = (GtkTreeView *)data;
  GtkTreeModel *model;
  GtkTreePath *path = 
    gtk_tree_path_new_from_string (path_string);
  GtkTreeIter iter;
  //indice
  gint i;
  //old splitpoint, and new one that will replace the old one
  Split_point old_point;
  Split_point new_point;
  
  model = gtk_tree_view_get_model(tree_view);
  
  //get the column number
  gint col = (gint)(g_object_get_data (G_OBJECT(cell), "col"));
  //get iter number
  gtk_tree_model_get_iter (model, &iter, path);
  //get the indice
  i = gtk_tree_path_get_indices (path)[0];
  old_point = g_array_index(splitpoints, Split_point, i);
  
  //check which column
  switch (col)
    {
    case COL_DESCRIPTION:
      update_current_description(new_text, i);
      
      //put the new content in the list
      gtk_list_store_set (GTK_LIST_STORE (model), 
			  &iter,
			  col, current_description,
			  -1);
      
      g_snprintf(current_description, 255, "%s",
		 (gchar *)_("description here"));      
      break;
      //seconds column
    case COL_SECONDS:
      new_point.mins = old_point.mins;
      new_point.secs = atoi(new_text);
      new_point.hundr_secs = old_point.hundr_secs;
      
      if (new_point.secs < 0)
	{
	  new_point.secs = 0;
	}
      if (new_point.secs > 59)
	{
	  new_point.secs = 59;
	}
      
      update_splitpoint(i,new_point);
      break;
      //minutes column
    case COL_MINUTES:
      new_point.mins = atoi(new_text);
      new_point.secs = old_point.secs;
      new_point.hundr_secs = old_point.hundr_secs;
      
      if (new_point.mins < 0)
	{
	  new_point.mins = 0;
	}
      if (new_point.mins > 2000)
	{
	  new_point.mins = 2000;
	}
	  
      update_splitpoint(i,new_point);
      break;
      //hundreth column
    case COL_HUNDR_SECS:
      new_point.mins = old_point.mins;
      new_point.secs = old_point.secs;
      new_point.hundr_secs = atoi(new_text);
      
      if (new_point.hundr_secs < 0)
	{
	  new_point.hundr_secs = 0;
	}
      if (new_point.hundr_secs > 99)
	{
	  new_point.hundr_secs = 99;
	}
	  
      update_splitpoint(i,new_point);
      break;
    default:
      break;
    }
  //free memory
  gtk_tree_path_free (path);
}

//adds a splitpoint from the player
void add_splitpoint_from_player(GtkWidget *widget, 
				gpointer data)
{
  if (timer_active)
    { 
      Split_point my_split_point;
      //get minutes and seconds
      my_split_point.mins = player_minutes;
      my_split_point.secs = player_seconds;
      my_split_point.hundr_secs = player_hundr_secs;
      add_splitpoint(my_split_point,-1);
    }
}

//adds a splitpoint
//old_index is used when we update a splitpoint
//to see where we had the play_preview point
void add_splitpoint(Split_point my_split_point,
		    gint old_index)
{
  GtkTreeIter iter;
  GtkTreeModel *model;
  
  if(check_if_splitpoint_exists(tree_view,
				my_split_point.mins,
				my_split_point.secs,
				my_split_point.hundr_secs,-1))
    {
      gchar *temp = g_strdup(current_description);
      update_current_description(temp, -1);
      free(temp);
      
      model = gtk_tree_view_get_model(tree_view);
      
      int k = 0;
      gint tree_minutes;
      gint tree_seconds;
      gint tree_hundr_secs;
      //if the table is not empty
      if(gtk_tree_model_get_iter_first(model, &iter))
	{
	  //for all the splitnumbers
	  while (k < splitnumber)
	    {
	      //we get the first
	      //get iter number
	      //get minutes and seconds for the first row
	      gtk_tree_model_get(GTK_TREE_MODEL(model), &iter,
				 COL_MINUTES, &tree_minutes,
				 COL_SECONDS, &tree_seconds,
				 COL_HUNDR_SECS, &tree_hundr_secs,
				 -1);
	      
	      //if first row minutes are superior to next row minutes
	      if (my_split_point.mins < tree_minutes)
		{
		  break;
		}
	      else
		//if minutes equal, check the seconds
		if (my_split_point.mins == tree_minutes)
		  {
		    //if seconds equal, check hundreth
		    if (my_split_point.secs < tree_seconds)
		      {
			break;
		      }
		    else
		      //if seconds equal, check hundreth
		      if (my_split_point.secs == tree_seconds)
			{
			  //we check the hundred of seconds
			  if (my_split_point.hundr_secs < tree_hundr_secs)
			    {
			      break;
			    }
			}
		  }
	      
	      //put the next row in iter
	      gtk_tree_model_iter_next(model, &iter);
	      k++;
	    }
	  	  
	  //insert line
	  gtk_list_store_insert(GTK_LIST_STORE(model),
				&iter,k--);
	  //put the values to the splitpoint array
	  g_array_insert_val(splitpoints,k+1,my_split_point);	  
	}
      else
	{
	  gtk_list_store_append (GTK_LIST_STORE (model),
				 &iter);
	  g_array_append_val(splitpoints,my_split_point);
	}
      splitnumber++;
      
      //we keep the selection on the previous splipoint
      if ((first_splitpoint_selected == old_index)
	  && (old_index != -1))
	{
	  GtkTreePath *path;
	  path = gtk_tree_model_get_path(model, &iter);
	  gtk_tree_view_set_cursor (tree_view,path,NULL,FALSE);
	  gtk_tree_path_free(path);
	}

      if (quick_preview)
	{
	  //if we move the current start preview splitpoint
	  //at the right of the current time, we cancel preview
	  if (old_index == preview_start_splitpoint)
	    {
	      if (current_time < 
		  get_splitpoint_time(preview_start_splitpoint)/10)
		{
		  cancel_quick_preview();
		}
	    }
	}
      
      //we manage the play preview here
      if (old_index != -1)
	{
	  //if we have a split preview on going
	  //if we move the point from the left to the right of the
	  //the start preview splitpoint
	  if ((old_index < preview_start_splitpoint))
	    {
	      if ((k+1) >= preview_start_splitpoint)
		{
		  preview_start_splitpoint--;
		  quick_preview_end_splitpoint = 
		    preview_start_splitpoint+1;
		}
	    }
	  else
	    {
	      //if we move from the right of the split preview
	      //to his left
	      if ((old_index > preview_start_splitpoint))
		{
		  if ((k+1) <= preview_start_splitpoint)
		    {
		      preview_start_splitpoint++;
		      quick_preview_end_splitpoint = 
			preview_start_splitpoint+1;
		    }
		}
	      else
		{
		  //if we move the start splitpoint on the right of
		  //the end splitpoint
		  if (old_index == preview_start_splitpoint)
		    {
		      if ((k+1) > preview_start_splitpoint)
			{
			  //we add how many splitpoints 
			  //we passed on
			  preview_start_splitpoint +=
			    (k+1)-preview_start_splitpoint;
			  quick_preview_end_splitpoint = 
			    preview_start_splitpoint+1;
			}
		      else
			{
			  //if we move the start splitpoint at the left
			  if ((k+1) < preview_start_splitpoint)
			    {
			      //we remove how many splitpoints 
			      //we passed on
			      preview_start_splitpoint -=
				preview_start_splitpoint-(k+1);
			      quick_preview_end_splitpoint = 
				preview_start_splitpoint+1;
			    }
			}
		    }
		}
	    }
	  
	  if (preview_start_splitpoint == (splitnumber-1))
	    {
	      cancel_quick_preview_all();
	    }
	}
      else
	{
	  //if we add a splitpoint at the left of the quick
	  //preview start, add 1
	  if ((k+1) <= preview_start_splitpoint)
	    {
	      preview_start_splitpoint ++;
	      quick_preview_end_splitpoint = 
		preview_start_splitpoint+1;
	    }
	}
            
      //put values in the line
      //sets text in the minute, second and milisecond column
      gtk_list_store_set (GTK_LIST_STORE (model), 
			  &iter,
			  COL_DESCRIPTION,current_description,
			  COL_MINUTES,my_split_point.mins,
			  COL_SECONDS,my_split_point.secs,
			  COL_HUNDR_SECS,my_split_point.hundr_secs,
			  -1);
	        
      //enable remove all rows button if needed
      if(!GTK_WIDGET_SENSITIVE(remove_all_button))
	{
	  gtk_widget_set_sensitive(GTK_WIDGET(remove_all_button), TRUE);
	}
      
      order_length_column(tree_view);      
      remove_status_message();
    }
  else
    {
      //if we already have a equal splitpoint
      put_status_message((gchar *)_(" error : you already have"
				    " the splitpoint in table "));
    }
  
  g_snprintf(current_description, 255,
	     "%s", (gchar *)_("description here"));
  
  //
  update_add_button();
  refresh_drawing_area();
  check_update_down_progress_bar();
}

//adds a row to the table
void add_row (GtkWidget *button, gpointer data)
{
  Split_point my_split_point;
  
  my_split_point.mins = spin_mins;
  my_split_point.secs = spin_secs;
  my_split_point.hundr_secs = spin_hundr_secs;
  
  add_splitpoint(my_split_point,-1);  
}

//remove a row from the table
void remove_row (GtkWidget *widget, gpointer data)
{
  GtkTreeSelection *selection;
  GList *selected_list = NULL;
  GList *current_element = NULL;
  GtkTreeView *tree_view = 
    (GtkTreeView *)data;
  GtkTreeModel *model;
  //indice
  gint i;
  //the path
  GtkTreePath *path;
  
  model = gtk_tree_view_get_model(tree_view);
  
  //get the selection
  selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(tree_view));
  
  //get selected rows
  selected_list = gtk_tree_selection_get_selected_rows(selection, &model);
  
  //while the list is not empty and we have numbers in the table
  //(splitnumber >0)
  while ((g_list_length(selected_list) > 0)
	 && (splitnumber > 0))
    {
      //get the last element
      current_element = g_list_last(selected_list);
      path = current_element->data;
      i = gtk_tree_path_get_indices (path)[0];
      
      remove_splitpoint(i,TRUE);
      
      //remove the path from the selected list
      selected_list = g_list_remove(selected_list, path);
      //free memory
      gtk_tree_path_free(path);
    }
  
  //we free the selected elements
  g_list_foreach (selected_list, 
		  (GFunc)gtk_tree_path_free, NULL);
  g_list_free (selected_list);
}

//removes all rows from the table
void remove_all_rows (GtkWidget *widget, 
		      gpointer data)
{
  GtkTreeIter iter;
  GtkTreeModel *model;
  
  model = gtk_tree_view_get_model(tree_view);
  
  //for all the splitnumbers
  while (splitnumber > 0)
    {
      gtk_tree_model_get_iter_first(model, &iter);
      gtk_list_store_remove (GTK_LIST_STORE (model), &iter);
      //remove values from the splitpoint array
      g_array_remove_index (splitpoints, (splitnumber-1));
      splitnumber--;
    }
  
  //disable remove all button
  if(GTK_WIDGET_SENSITIVE(remove_all_button))
    gtk_widget_set_sensitive(GTK_WIDGET(remove_all_button), FALSE);
  
  //disable remove button
  if(GTK_WIDGET_SENSITIVE(remove_row_button))
    gtk_widget_set_sensitive(GTK_WIDGET(remove_row_button), FALSE);
  
  remove_status_message();
  cancel_quick_preview_all();
  //
  update_add_button();
  refresh_drawing_area();
  check_update_down_progress_bar();
}

//creates and and initialise a spinner
GtkWidget *create_init_spinner(GtkWidget *bottomhbox1, 
			       gint min, gint max, 
			       gchar *label_text,
			       gint type)
{
  //the spinner
  GtkWidget *spinner;
  //the adjustment
  GtkAdjustment *adj;
  //vertical box for the label
  GtkWidget *spinner_box;
  //spinner label
  GtkWidget *label;

  spinner_box = gtk_vbox_new (FALSE, 0); 
  label = gtk_label_new (label_text);
  //adds label to spinner box
  gtk_box_pack_start (GTK_BOX (spinner_box), label, TRUE, FALSE, 0);
  adj = (GtkAdjustment *) gtk_adjustment_new (0.0, min, max, 1.0,
					      10.0, 0.0);
  spinner = gtk_spin_button_new (adj, 0, 0);
  gtk_spin_button_set_wrap (GTK_SPIN_BUTTON (spinner), TRUE);

  //0 means minutes
  if (type == 0)
    {
      g_signal_connect (G_OBJECT (spinner), "value_changed",
			G_CALLBACK (update_minutes_from_spinner), NULL);
    }
  else 
    //1 means seconds
    if (type == 1)
      {
	g_signal_connect (G_OBJECT (spinner), "value_changed",
			  G_CALLBACK (update_seconds_from_spinner), NULL);
      }
    else
      {
	g_signal_connect (G_OBJECT (spinner), "value_changed",
			  G_CALLBACK (update_hundr_secs_from_spinner), NULL);
      }
  
  //adds spinner to the spinner box
  gtk_box_pack_start (GTK_BOX (spinner_box), spinner, TRUE, FALSE, 0);
  //adds spinner box to the horizontal box1
  gtk_box_pack_start (GTK_BOX (bottomhbox1), spinner_box, TRUE, FALSE, 5);
  
  return spinner;
}

//minutes ,seconds spinners ; add, delete buttons
GtkWidget *create_init_spinners_buttons(GtkTreeView *tree_view)
{
  GtkWidget *hbox;

  hbox = gtk_hbox_new (FALSE, 0);
  gtk_container_set_border_width (GTK_CONTAINER (hbox), 0);

  /* minutes and seconds spinners */
  spinner_minutes = create_init_spinner(hbox, 
					0, 2000,
					(gchar *)_("Minutes:"),
					//0 means spinner minutes
					0);
  spinner_seconds = create_init_spinner(hbox, 
					0, 59,
					(gchar *)_("Seconds:"), 
					//1 means spinner seconds
					1);
  //hundredth spinner
  spinner_hundr_secs = create_init_spinner(hbox, 
					   0, 99,
					   (gchar *)_("Hundredths:"), 
					   //2 means spinner hundredth
					   2);

  GtkTooltips *tooltip;
  tooltip = gtk_tooltips_new();
  
  /* add button */
  add_button = (GtkWidget *)create_cool_button(GTK_STOCK_ADD,
					       (gchar *)_("_Add"), FALSE);
  gtk_button_set_relief(GTK_BUTTON(add_button), GTK_RELIEF_NONE);
  gtk_widget_set_sensitive(GTK_WIDGET(add_button), TRUE);
  g_signal_connect (G_OBJECT (add_button), "clicked",
		    G_CALLBACK (add_row), tree_view);
  gtk_box_pack_start (GTK_BOX (hbox), add_button, TRUE, FALSE, 5);
  gtk_tooltips_set_tip(tooltip, add_button,(gchar *)_("add splitpoint"),"");

  /* remove row button */
  remove_row_button = (GtkWidget *)
    create_cool_button(GTK_STOCK_REMOVE, (gchar *)_("_Remove"),
		       FALSE);
  gtk_button_set_relief(GTK_BUTTON(remove_row_button), GTK_RELIEF_NONE);
  gtk_widget_set_sensitive(GTK_WIDGET(remove_row_button), FALSE);
  g_signal_connect (G_OBJECT (remove_row_button), "clicked",
		    G_CALLBACK (remove_row), tree_view);
  gtk_box_pack_start (GTK_BOX (hbox), remove_row_button, TRUE, FALSE, 5);
  gtk_tooltips_set_tip(tooltip, remove_row_button,(gchar *)_("remove rows"),"");

  /* remove all rows button */
  remove_all_button = (GtkWidget *)
    create_cool_button(GTK_STOCK_DELETE,(gchar *)_("Remove all"),
		       FALSE);
  gtk_button_set_relief(GTK_BUTTON(remove_all_button), GTK_RELIEF_NONE);
  gtk_widget_set_sensitive(GTK_WIDGET(remove_all_button), FALSE);
  g_signal_connect (G_OBJECT (remove_all_button), "clicked",
		    G_CALLBACK (remove_all_rows), tree_view);
  gtk_box_pack_start (GTK_BOX (hbox), remove_all_button, TRUE, FALSE, 5);
  gtk_tooltips_set_tip(tooltip, remove_all_button,(gchar *)_("remove all rows"),"");

  return hbox;
}

//returns the hundreths of seconds from a Split_point
gint splitpoint_to_hundreths(Split_point point)
{
  return (point.secs + point.mins*60)*1000+
    point.hundr_secs * 10;
}

//returns the name of the splitpoint
//result must be freed
gchar *get_splitpoint_name(gint index)
{
  GtkTreeModel *model;
  GtkTreeIter iter;
  GtkTreePath *path = NULL;
  model = gtk_tree_view_get_model(tree_view);
  gchar *description = NULL;
  if(gtk_tree_model_get_iter_first(model, &iter))
    {
      if (index == -1)
	index = 0;
      
      if (index >= 0)
	{
	  path = 
	    gtk_tree_path_new_from_indices (index ,-1);
	  //get the iter correspondig to the path
	  gtk_tree_model_get_iter(model, &iter, path);
	  //we get the description
	  gtk_tree_model_get(GTK_TREE_MODEL(model), &iter,
			     COL_DESCRIPTION,&description,
			     -1);
	  //free memory
	  gtk_tree_path_free(path);
	}
    }
  
  return description;
}

//returns a splitpoint from the table
gint get_splitpoint_time(gint this_splitpoint)
{
  if (this_splitpoint != -1)
    {
      return splitpoint_to_hundreths(g_array_index(splitpoints,
						   Split_point,
						   this_splitpoint));
    }
  else
    return -1;
}

//the split preview
void split_preview(gpointer *data)
{
  //lock gtk
  gdk_threads_enter();
  
  gchar *old_filename = NULL;
  
  //if it's not the last one
  if (this_row+1 != splitnumber)
    {
      gint confirmation;
      
      //we put the splitpoints in the state
      the_state->split.splitpoints[0] = 
	preview_start_position / 10;
      if (quick_preview_end_splitpoint == -1)
	{
	  the_state->split.splitpoints[1] = -1;
	}
      else
	{
	  the_state->split.splitpoints[1] =
	    get_splitpoint_time(quick_preview_end_splitpoint)
	    / 10;
	}
	
      //we save the current name of the first
      //file to split
      old_filename = 
	g_strdup(the_state->split.fn[0]);
      //we put the name of the splitted file
      g_snprintf(the_state->split.fn[0],255,
		 "preview");
      
      //put the options from the preferences
      put_options_from_preferences();
			
      //set the options
      the_state->options.option_mins_secs = FALSE;
      the_state->options.option_auto_adjust = FALSE;
      the_state->options.split_mode = MP3SPLT_OPTION_NORMAL_MODE;
      
      gchar *fname;
      gchar *fname_path;
			
      //we cut the preferences filename path
      //to find the ~/.mp3splt directory
      fname_path = get_preferences_filename();
      fname_path[strlen(fname_path)-18] = '\0';
      
      //we get the filename to split
      fname = (gchar *)
	gtk_entry_get_text(GTK_ENTRY(entry));
      
      //remove old splitted files
      remove_all_splitted_rows();  
      
      //unlock gtk
      gdk_threads_leave();
      
      //effective split, returns confirmation or error;
      //2 because we have only 2 splitpoints
      confirmation = 
	libmp3splt_split_file(the_state,fname,2,
			      fname_path);
      
      //lock gtk
      gdk_threads_enter();
      
      //we put the old filename
      g_snprintf(the_state->split.fn[0],255,"%s",
		 old_filename);
      //free memory
      g_free(old_filename);
      
      //we show infos about the splitted action
      print_status_bar_confirmation(confirmation);
      
      gchar *splitted_file; 
      //here the first one
      splitted_file = get_filename_from_splitted_files(1);
      
      if (splitted_file != NULL)
	{
	  //we play the splitted file
	  //set the entry with the current filename
	  gtk_entry_set_text(GTK_ENTRY(entry),
			     splitted_file);
	  
	  g_free(splitted_file);
      
	  //only if success
	  if (confirmation > 0)
	    {
	      //connecting to player
	      connect_button_event (NULL, NULL);
	      //starts playing, 0 means start playing
	      connect_to_player_with_song(0);
	    }
	}
      
      //only if success
      if (confirmation > 0)
	{
	  //we have finished, put 100 to the progress bar
	  //we update the progress
	  gtk_progress_bar_set_fraction(GTK_PROGRESS_BAR(percent_progress_bar),
					1.0);
	  //we write the progress on the bar
	  gtk_progress_bar_set_text(GTK_PROGRESS_BAR(percent_progress_bar),
				    _(" finished "));
	}
      
      //free memory
      g_free(fname_path);
    }
  else
    {
      put_status_message((gchar *)
			 _(" cannot split preview"
			   " last splitpoint"));
    }
  
  //unlock gtk
  gdk_threads_leave();
}

//the row clicked event, preview the song
void preview_song (GtkTreeView *tree_view,
		   GtkTreePath *path,
		   GtkTreeViewColumn *col,
		   gpointer user_data)
{
  GtkTreeModel *model;
  model = gtk_tree_view_get_model(tree_view);
  
  gint number = (gint)
    g_object_get_data (G_OBJECT(col), "col");
  //only if connected to player
  if (timer_active)
    {
      //we get the split begin position to find the 
      //end position
      this_row = gtk_tree_path_get_indices (path)[0];
      //if we click COL_PREVIEW
      if (number == COL_PREVIEW)
	{
	  player_quick_preview(this_row);
	}
      else
	{
	  //if we have the split preview
	  if (number == COL_SPLIT_PREVIEW)
	    {
	      g_thread_create((GThreadFunc)split_preview,
			      NULL, TRUE, NULL);
	    }
	}
    }
  else
    {
      put_status_message((gchar *)
			 _(" cannot preview,"
			   " not connected to player "));
    }
}

//creates columns for the tree
void create_columns (GtkTreeView *tree_view)
{
  //cells renderer
  GtkCellRendererText *renderer;
  GtkCellRendererPixbuf *renderer_pix;
  //columns
  GtkTreeViewColumn *column_number;
  GtkTreeViewColumn *column_description;
  GtkTreeViewColumn *column_hundr_secs;
  GtkTreeViewColumn *column_minutes;
  GtkTreeViewColumn *column_seconds;
  GtkTreeViewColumn *column_preview;
  GtkTreeViewColumn *column_split_preview;
  
  /* description */
  //renderer creation
  renderer = GTK_CELL_RENDERER_TEXT(gtk_cell_renderer_text_new ());
  //cell edited events
  g_signal_connect(renderer, "edited",
		   G_CALLBACK(cell_edited_event),
		   tree_view);
  //enable cell editing
  g_object_set(renderer, "editable", TRUE, NULL);
  g_object_set_data(G_OBJECT(renderer), "col", (gchar *)COL_DESCRIPTION);
  column_description = gtk_tree_view_column_new_with_attributes
    ((gchar *)_("Filename"), GTK_CELL_RENDERER(renderer),
     "text", COL_DESCRIPTION,
     NULL);
  
  /* seconds */
  //renderer creation
  renderer = GTK_CELL_RENDERER_TEXT(gtk_cell_renderer_text_new ());
  //cell edited events
  g_signal_connect(renderer, "edited",
		   G_CALLBACK(cell_edited_event),
		   tree_view);
  //enable cell editing
  g_object_set(renderer, "editable", TRUE, NULL);
  g_object_set_data(G_OBJECT(renderer), "col", (gint *)COL_SECONDS);
  column_seconds = gtk_tree_view_column_new_with_attributes
    ((gchar *)_("Secs"), GTK_CELL_RENDERER(renderer),
     "text", COL_SECONDS,
     NULL);
  
  /* minutes */
  //renderer creation
  renderer = GTK_CELL_RENDERER_TEXT(gtk_cell_renderer_text_new ());
  //cell edited events
  g_signal_connect(renderer, "edited",
		   G_CALLBACK(cell_edited_event),
		   tree_view);
  //enable cell editing
  g_object_set(renderer, "editable", TRUE, NULL);
  g_object_set_data(G_OBJECT(renderer), "col", (gint *)COL_MINUTES);
  column_minutes = gtk_tree_view_column_new_with_attributes 
    ((gchar *)_("Mins"), GTK_CELL_RENDERER(renderer),
     "text", COL_MINUTES,
     NULL);

  /* hundr secs */
  //renderer creation
  renderer = GTK_CELL_RENDERER_TEXT(gtk_cell_renderer_text_new ());
  //cell edited events
  g_signal_connect(renderer, "edited",
		   G_CALLBACK(cell_edited_event),
		   tree_view);
  //enable cell editing
  g_object_set(renderer, "editable", TRUE, NULL);
  g_object_set_data(G_OBJECT(renderer), "col", (gint *)COL_HUNDR_SECS);
  column_hundr_secs = gtk_tree_view_column_new_with_attributes 
    ((gchar *)_("Hundr"), GTK_CELL_RENDERER(renderer),
     "text", COL_HUNDR_SECS, NULL);
  
  /* Length column */
  //renderer creation
  renderer = GTK_CELL_RENDERER_TEXT(gtk_cell_renderer_text_new ());
  g_object_set_data(G_OBJECT(renderer), "col", (gint *)COL_NUMBER);
  //middle alignment
  g_object_set (G_OBJECT (renderer), "xalign", 1.0, NULL);
  column_number = gtk_tree_view_column_new_with_attributes 
    ((gchar *)_("Length"), GTK_CELL_RENDERER(renderer),
     "text", COL_NUMBER, NULL);
  
  /* column preview */
  //renderer creation
  renderer_pix = GTK_CELL_RENDERER_PIXBUF(gtk_cell_renderer_pixbuf_new ());
  //set the icon
  g_object_set(renderer_pix,"stock-id",GTK_STOCK_MEDIA_PLAY,
	       "stock-size",GTK_ICON_SIZE_MENU,NULL);
  //create the column
  column_preview = gtk_tree_view_column_new_with_attributes 
    ((gchar *)"LiveP", GTK_CELL_RENDERER(renderer_pix), 
     "pixbuf",COL_PREVIEW, NULL);
  g_object_set_data(G_OBJECT(column_preview), "col", 
		    (gint *)COL_PREVIEW);
  
  /* split preview */
  renderer_pix = GTK_CELL_RENDERER_PIXBUF(gtk_cell_renderer_pixbuf_new ());
  //set the icon
  g_object_set(renderer_pix,"stock-id",GTK_STOCK_MEDIA_PLAY,
	       "stock-size",GTK_ICON_SIZE_MENU,NULL);
  //create the column
  column_split_preview = gtk_tree_view_column_new_with_attributes 
    ((gchar *)"SplitP", GTK_CELL_RENDERER(renderer_pix), 
     "pixbuf",COL_SPLIT_PREVIEW, NULL);
  g_object_set_data(G_OBJECT(column_split_preview), "col", 
		    (gint *)COL_SPLIT_PREVIEW);
  
  //appends columns to the list of columns of tree_view
  gtk_tree_view_insert_column (GTK_TREE_VIEW (tree_view),
			       GTK_TREE_VIEW_COLUMN (column_description),COL_DESCRIPTION);
  gtk_tree_view_insert_column (GTK_TREE_VIEW (tree_view),
			       GTK_TREE_VIEW_COLUMN (column_minutes),COL_MINUTES);
  gtk_tree_view_insert_column (GTK_TREE_VIEW (tree_view),
			       GTK_TREE_VIEW_COLUMN (column_seconds),COL_SECONDS);
  gtk_tree_view_insert_column (GTK_TREE_VIEW (tree_view),
			       GTK_TREE_VIEW_COLUMN (column_hundr_secs),COL_HUNDR_SECS);
  gtk_tree_view_insert_column (GTK_TREE_VIEW (tree_view),
			       GTK_TREE_VIEW_COLUMN (column_number),COL_NUMBER);
  gtk_tree_view_insert_column (GTK_TREE_VIEW (tree_view),
			       GTK_TREE_VIEW_COLUMN (column_preview),COL_PREVIEW);
  gtk_tree_view_insert_column (GTK_TREE_VIEW (tree_view),
			       GTK_TREE_VIEW_COLUMN (column_split_preview),
			       COL_SPLIT_PREVIEW);
  //middle alignment of the column name
  gtk_tree_view_column_set_alignment(GTK_TREE_VIEW_COLUMN(column_description),
				     0.5);
  gtk_tree_view_column_set_alignment(GTK_TREE_VIEW_COLUMN(column_minutes),
				     0.5);
  gtk_tree_view_column_set_alignment(GTK_TREE_VIEW_COLUMN(column_seconds),
				     0.5);
  gtk_tree_view_column_set_alignment(GTK_TREE_VIEW_COLUMN(column_hundr_secs),
				     0.5);
  gtk_tree_view_column_set_alignment(GTK_TREE_VIEW_COLUMN(column_number),
				     0.5);
  gtk_tree_view_column_set_alignment(GTK_TREE_VIEW_COLUMN(column_preview),
				     0.5);
  gtk_tree_view_column_set_alignment(GTK_TREE_VIEW_COLUMN(column_split_preview),
				     0.5);

  //set the auto resizing for columns
  gtk_tree_view_column_set_sizing (GTK_TREE_VIEW_COLUMN(column_description),
				   GTK_TREE_VIEW_COLUMN_AUTOSIZE);
  gtk_tree_view_column_set_sizing (GTK_TREE_VIEW_COLUMN(column_minutes),
				   GTK_TREE_VIEW_COLUMN_AUTOSIZE);
  gtk_tree_view_column_set_sizing (GTK_TREE_VIEW_COLUMN(column_seconds),
				   GTK_TREE_VIEW_COLUMN_AUTOSIZE);
  gtk_tree_view_column_set_sizing (GTK_TREE_VIEW_COLUMN(column_hundr_secs),
				   GTK_TREE_VIEW_COLUMN_AUTOSIZE);
  gtk_tree_view_column_set_sizing (GTK_TREE_VIEW_COLUMN(column_number),
				   GTK_TREE_VIEW_COLUMN_AUTOSIZE);
  gtk_tree_view_column_set_sizing (GTK_TREE_VIEW_COLUMN(column_preview),
				   GTK_TREE_VIEW_COLUMN_AUTOSIZE);
  gtk_tree_view_column_set_sizing (GTK_TREE_VIEW_COLUMN(column_split_preview),
				   GTK_TREE_VIEW_COLUMN_AUTOSIZE);
  
  //sets resize 
  gtk_tree_view_column_set_resizable(column_description, TRUE);
  //set column reorderable
  gtk_tree_view_column_set_reorderable(column_description, TRUE);
  gtk_tree_view_column_set_reorderable(column_minutes, TRUE);
  gtk_tree_view_column_set_reorderable(column_seconds, TRUE);
  gtk_tree_view_column_set_reorderable(column_hundr_secs, TRUE);
  gtk_tree_view_column_set_reorderable(column_number, TRUE);
  gtk_tree_view_column_set_reorderable(column_preview, TRUE);
  gtk_tree_view_column_set_reorderable(column_split_preview, TRUE);
  //set column expand
  gtk_tree_view_column_set_expand (column_description, TRUE);
}

//when closing the new window after detaching
void close_popup_window_event( GtkWidget *window,
			       gpointer data )
{
  GtkWidget *window_child;

  window_child = gtk_bin_get_child(GTK_BIN(window));

  gtk_widget_reparent(GTK_WIDGET(window_child), GTK_WIDGET(handle_box));

  gtk_widget_destroy(window);
}

//when we detach the handle
void handle_detached_event (GtkHandleBox *handlebox,
			    GtkWidget *widget,
			    gpointer data)
{
  //new window
  GtkWidget *window;

  /* window */
  window = gtk_window_new(GTK_WINDOW_TOPLEVEL);

  gtk_widget_reparent(GTK_WIDGET(widget), GTK_WIDGET(window));

  g_signal_connect (G_OBJECT (window), "delete_event",
		    G_CALLBACK (close_popup_window_event),
		    NULL);

  gtk_widget_show(GTK_WIDGET(window));
}

//creates the choose splitpoints frame, who has the spinners, arrows,
//the tree, add, remove buttons
GtkWidget *create_choose_splitpoints_frame(GtkTreeView *tree_view)
{
  //choose splitpoints box, has tree, spinner, arrows..
  GtkWidget *choose_splitpoints_vbox;
  //scrolled window used for the tree
  GtkWidget *scrolled_window;
  //spinners + add and remove buttons box
  GtkWidget *spinners_buttons_hbox;
  //horizontal box for tree and arrows
  GtkWidget *tree_hbox;

  /* the tree */
  GtkTreeSelection *selection;

  /* choose splitpoins vbox */
  choose_splitpoints_vbox = gtk_vbox_new (FALSE, 0);
  gtk_container_set_border_width (GTK_CONTAINER (choose_splitpoints_vbox), 0);

  /* handle box for detaching */
  handle_box = gtk_handle_box_new();
  gtk_container_add(GTK_CONTAINER (handle_box), GTK_WIDGET(choose_splitpoints_vbox));
  //handle event
  g_signal_connect(handle_box, "child-detached",
		   G_CALLBACK(handle_detached_event),
		   NULL);

  /* spinner buttons hbox */
  spinners_buttons_hbox = create_init_spinners_buttons(tree_view);
  gtk_box_pack_start (GTK_BOX (choose_splitpoints_vbox), spinners_buttons_hbox, FALSE, FALSE, 7);
  
  /* horizontal box for the tree */
  tree_hbox = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (choose_splitpoints_vbox), tree_hbox, TRUE, TRUE, 0);

  /* scrolled window for the tree */
  scrolled_window = gtk_scrolled_window_new (NULL, NULL);
  gtk_widget_set_size_request(scrolled_window, 300,130);
  gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW(scrolled_window), GTK_SHADOW_NONE);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_window),
				  GTK_POLICY_AUTOMATIC,
				  GTK_POLICY_AUTOMATIC);
  gtk_box_pack_start (GTK_BOX (tree_hbox), scrolled_window, TRUE, TRUE, 0);

  //get the selection
  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (tree_view));
  gtk_tree_selection_set_mode(selection, GTK_SELECTION_MULTIPLE);
  //create columns
  create_columns (tree_view);
  //add the tree to the scrolled window
  gtk_container_add (GTK_CONTAINER (scrolled_window), GTK_WIDGET(tree_view));

  return handle_box;
}

//transform garray to array
void garray_to_array(GArray *spltpoints, 
		     gint *hundredth)
{
  gint i;
  Split_point point;
  
  for(i = 0; i < splitnumber; i++ )
    {
      point = g_array_index(splitpoints, Split_point, i);
      //convert to hundreds
      hundredth[i] = point.mins*6000 
	+ point.secs*100 + point.hundr_secs;
    }
}

//puts the splitpoints in the state
void put_splitpoints_in_the_state(splt_state *state)
{
  gint hundr[splitnumber];
  garray_to_array(splitpoints, hundr);
  gint i;
  
  //for getting the filename
  GtkTreeModel *model;
  model = gtk_tree_view_get_model(tree_view);
  GtkTreeIter iter;
  GtkTreePath *path = NULL;
  gchar *description = NULL;
  
  //we put all the splitpoints with the file names
  for (i = 0; i < splitnumber; i++)
    {
      path = 
	gtk_tree_path_new_from_indices (i ,-1);
      //get the iter correspondig to the path
      gtk_tree_model_get_iter(model, &iter, path);
      //we get the description
      gtk_tree_model_get(GTK_TREE_MODEL(model), &iter,
			 COL_DESCRIPTION,&description,
			 -1);
      
      libmp3splt_put_splitpoint(state,i,hundr[i], description);
      
      //free memory
      gtk_tree_path_free(path);
    }
}

//creates the tree view
GtkTreeView *create_tree_view()
{
  GtkTreeModel *model;
  //create the model
  model = create_model();
  //create the tree view
  tree_view = (GtkTreeView *)gtk_tree_view_new_with_model (model);
  
  //preview_song callback when clicking on the row
  g_signal_connect(tree_view,"row-activated",
		   G_CALLBACK(preview_song),tree_view);
  
  //set the selection signal for enabling/disabling buttons
  //the tree selection
  GtkTreeSelection *selection;
  selection = gtk_tree_view_get_selection(tree_view);
  g_signal_connect(selection, "changed",
		   G_CALLBACK(row_selection_event),
		   NULL);
  
  return tree_view;
}

