#!/usr/bin/ruby -w
#
# lyricstime - www.lyricstime.com implementation of MusicSite
#
# version: $Id: lyricstime.rb 331 2004-07-07 22:48:11Z kapheine $
#
# Copyright (C) 2003-2006 Zachary P. Landau <kapheine@hypa.net>
#                         Tony Cebzanov <tonyc@tonyc.org>
# All rights reserved.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#

require 'musicextras/musicsite'
require 'musicextras/album'
require 'musicextras/song'
require 'musicextras/debuggable'

module MusicExtras

  # This is an implementation of MusicSite for www.lyricstime.com
  class LyricsTime < MusicSite
    attr_accessor :song, :album, :artist # :nodoc:

    register()

    NAME = 'LyricsTime'
    URL = 'www.lyricstime.com'
    DESCRIPTION = 'Lyrics and album covers for multiple genres'

    # Image used by the site when an album cover is unavailable
    NO_COVER_URL = '/images/l_no_cover.gif'

    def initialize
      super(NAME, URL)

      Song::register_plugin(self, :lyrics, CACHE_PATH['lyrics'])
      Album::register_plugin(self, :cover, CACHE_PATH['album_cover'])
    end

    def urlify(txt)
      txt.gsub!(/[ ']/, "-") #'
      txt.gsub!(%r![^\w\s-]+!, "")
      txt.gsub!(/-+/, "-")
      txt.downcase!()
      return txt
    end

    # Fetches lyrics from the site, returning the lyrics as string
    # [+song+] a Song object
    #
    # Note: returns nil if something went wrong (including just not being able
    # to find the lyrics)
    def lyrics(new_song)
      @song = new_song

      unless @song.title and @song.artist
	debug(1, "song title or artist not specified")
	return nil
      end
  
      artist = @song.artist.name
      title = @song.title
      a = urlify(artist)
      t = urlify(title)
      song_url = "http://www.lyricstime.com/#{a}-#{t}-lyrics.html"
      debug_var { :song_url }
      return nil unless song_url
      debug_var { :song_url }
      page = fetch_page(song_url)
      l = extract_text(page, %r!<span>lyrics</span></a></b>"</p><p>(.*?)</p>!)
      return l
    end

    # Fetches album cover from the site, returning the image as binary
    # [+album+] an Album object
    #
    # Note: returns nil if something went wrong (including just not being able
    # to find the album cover)
    def cover(new_album)
      @album = new_album
      @song = Song.with_album('', @album)

      artist = @song.artist.name

      a = urlify(artist)
      artist_url = "http://www.lyricstime.com/#{a}-lyrics.html"
      debug_var { :artist_url }
      page = fetch_page(artist_url)
      page.scan(%r!<h2 class="clear">([^<]+)</h2><div class="ainfo"><img src="([^"]+)"/>!) do |alb, album_cover_url|
        next if album_cover_url == "./img/nc.gif"
        if match?(alb, new_album.title)
          cover = fetch_page(album_cover_url)
          debug_var { :album_cover_url }  
          return cover
        end
      end

      debug(1, "could not find url for #{@song.title} by #{@song.artist.name}")
      return nil
    end

    def test
      passed = true
      problems = []

      artist = Artist.new("Guns 'n Roses")
      album = Album.new("Appetite for Destruction", artist)
      song = Song.new("Sweet Child O'Mine", artist)

      if !lyrics(song)
        passed = false
        problems << "lyrics"
      end
      if !cover(album)
        passed = false
        problems << "album cover"
      end

      [passed, problems]
    end

  end
end
