/****************************************************************************
** cachethumbs.cpp
**
**   Created : December 20'st 2007
**        by : Varol Okan unsing XEmacs
** Copyright : (c) Varol Okan
**   License : GPL v 2.0
**
** This file holds the implementation of the export filter for the srt 
** subtitles file.
*****************************************************************************/

#include <qfileinfo.h>
#include <qstring.h>
#include <qimage.h>
#include <qfile.h>
#include <qdir.h>
#include <qdom.h>

#include "utils.h"
#include "messagebox.h"
#include "cachethumbs.h"
#include "qplayer/mediacreator.h"

namespace Cache
{

Thumbs::MultiThumbRequest::MultiThumbRequest ( QObject *pObj, Entry *p )
  : ExecuteJob ( pObj )
{
  pEntry          = p;
  pEntry->iScanStatus = 1; // Scanning
  iDelta          = -1;
  iMovieLength    = -1;
  iCurrentThumb   = -1;

  if ( pEntry->arrayOfThumbs )
       pEntry->freeImages  ( );
  
  pEntry->arrayOfThumbs      = new QImage *[pEntry->iNumberOfThumbs];
  for ( uint t=0; t<pEntry->iNumberOfThumbs; t++ )
    pEntry->arrayOfThumbs[t] = new QImage ( QImage ( ).fromMimeSource( "error.jpg" ) );
  
  // init base class variables
  theType    = TYPE_MULTI_THUMB;
  qsFileName = pEntry->qsFileName;
  pImage     = new QImage;
}

Thumbs::MultiThumbRequest::~MultiThumbRequest ( )
{
  if ( pImage )
    delete pImage;
  pImage = NULL;
}

bool Thumbs::MultiThumbRequest::response ( )
{
  // qsFileName == length ...
  if ( iCurrentThumb < 0 ) {
    pEntry->qsLength = qsFileName;
    pEntry->pCache->saveCacheDB ( ); // store the new found length in the xml file
    // The first time around we have to init the offset
    if ( iMSecondsOffset == -1 )
         pEntry->iScanStatus = 2;
    Utils theUtils;
    iMovieLength    = theUtils.getMsFromString ( qsFileName );
    iDelta          = (long)( (double)( iMovieLength + 500 ) / pEntry->iNumberOfThumbs );
    iCurrentThumb   = 0;
    iMSecondsOffset = 0;
    return true;
  }
  // here we store the image and increase the iMScOffset for the next thumbnail.
  if ( iCurrentThumb-1 < (int)pEntry->iNumberOfThumbs )
    *pEntry->arrayOfThumbs[iCurrentThumb] = pImage->smoothScale ( 300, 300, QImage::ScaleMin );

  iMSecondsOffset = ++iCurrentThumb * iDelta;

  if ( iMSecondsOffset >= iMovieLength ) {
    pEntry->iScanStatus = 2;
    pEntry->saveImages ( );
  }
  return true;
}

Thumbs::Entry::Entry ( Thumbs *pParent )
{
  iHashValue      = 0;
  iNumberOfThumbs = 0;
  iStarRating     = 0;
  iScanStatus     = 0;
  bSourceExists   = true;
  arrayOfThumbs   = NULL;
  pCache          = pParent;
  dateCreated     = QDateTime::currentDateTime ( );
  dateLastUsed    = QDateTime::currentDateTime ( );
}

bool Thumbs::Entry::readXml ( QDomElement *pSource )
{
  QString qsDateCreated, qsDateLastUsed, qsVirtualFolder;

  QDomAttr a = pSource->attributeNode ( CACHE_FILE_SIZE );
  iFileSize        = a.value ( ).toUInt ( );
  a = pSource->attributeNode ( CACHE_RATING );
  iStarRating      = a.value ( ).toUInt( );
  a = pSource->attributeNode ( CACHE_NR_OF_THUMBS );
  iNumberOfThumbs  = a.value ( ).toUInt( );
  a = pSource->attributeNode ( CACHE_VIRTUAL );
  qsVirtualFolder  = a.value ( );
  a = pSource->attributeNode ( CACHE_CREATED );
  qsDateCreated    = a.value ( );
  a = pSource->attributeNode ( CACHE_LENGTH );
  qsLength         = a.value ( );
  a = pSource->attributeNode ( CACHE_FILE_NAME);
  qsFileName       = a.value ( );
  a = pSource->attributeNode ( CACHE_LAST_USED );
  qsDateLastUsed   = a.value ( );

  dateCreated  = QDateTime::fromString ( qsDateCreated  );
  dateLastUsed = QDateTime::fromString ( qsDateLastUsed );
  listVirtualFolders = QStringList::split ( ";", qsVirtualFolder );

  // At this point we have all bu the thumbnails
  //iHashValue = pCache->hashFromFile ( qsFileName );
  bool bOkay = true;
  iHashValue = pSource->text ( ).toULongLong ( &bOkay, 16 );
  if ( ! bOkay )
    iHashValue = pCache->hashFromFile ( qsFileName );
  //printf ( "Thumbs::Entry::readXml <0x%016llX> <%s> <%s>\n", iHashValue, pSource->text ( ).ascii ( ), qsFileName.ascii ( ) );
  if ( iHashValue == 0LL ) {
    // File does not exist anymore. We want to keep the record around
    // in case this was on a memory stick or some other removable media
    bSourceExists   = false;
    return false;
  }
  return true;
}

bool Thumbs::Entry::writeXml ( QDomElement *pCacheElement )
{
  QString qsHash, qsVirtualFolder;
  qsHash.sprintf ( "%016llX", iHashValue );
  QDomDocument   theDoc = pCacheElement->ownerDocument ( );
  QDomElement entryNode = theDoc.createElement  ( CACHE_TAG_SOURCE );	// <vmm>
  QDomText    entryText = theDoc.createTextNode ( qsHash );

  qsVirtualFolder = listVirtualFolders.join ( ";" );

  // Here we set the attributes of the <Source> tag
  entryNode.setAttribute ( CACHE_FILE_SIZE,    QString ( "%1" ).arg ( iFileSize       ) );
  entryNode.setAttribute ( CACHE_RATING,       QString ( "%1" ).arg ( iStarRating     ) );
  entryNode.setAttribute ( CACHE_NR_OF_THUMBS, QString ( "%1" ).arg ( iNumberOfThumbs ) );
  entryNode.setAttribute ( CACHE_VIRTUAL,   qsVirtualFolder           );
  entryNode.setAttribute ( CACHE_CREATED,   dateCreated.toString  ( ) );
  entryNode.setAttribute ( CACHE_LENGTH,    qsLength                  );
  entryNode.setAttribute ( CACHE_FILE_NAME, qsFileName                );
  entryNode.setAttribute ( CACHE_LAST_USED, dateLastUsed.toString ( ) );
  entryNode.appendChild  ( entryText );

  pCacheElement->appendChild ( entryNode );
  return true;
}

void Thumbs::Entry::loadImages ( )
{
  if ( arrayOfThumbs )
    freeImages ( );

  uint t;
  QImage *pImage, theImage;
  QString qsFilePath, qsHashBaseName, qsFullName;
  QValueList<QImage *>list;
  QFileInfo fileInfo ( qsFileName );

  qsFilePath     = pCache->getCachePath ( fileInfo.baseName ( )[0] );
  qsHashBaseName = qsFilePath + qsHashBaseName.sprintf ( "%016llX", iHashValue );

  // Read in all files $HOME/.qdvdauthor/cache/a/7653ABC87683ABCD1234_xx.jpg
  for ( t=0;t<iNumberOfThumbs; t++ ) {
    qsFullName = qsFullName.sprintf ( "%s_%02d.jpg", qsHashBaseName.ascii ( ), t );
    if ( theImage.load  ( qsFullName ) ) {
      pImage = new QImage ( theImage );
      list.append ( pImage );
    }
  }
  if ( list.count ( ) > 0 ) {
    iNumberOfThumbs = list.count ( );
    arrayOfThumbs   = new QImage *[list.count ( )];
    for ( t=0; t<list.count ( ); t++ ) {
      arrayOfThumbs[t] = list[t];
    }
  }
  else {
    iNumberOfThumbs  = 1;
    arrayOfThumbs    = new QImage *[1];
    arrayOfThumbs[0] = new QImage ( QImage ( ).fromMimeSource( "error.jpg" ) );
  }
}

void Thumbs::Entry::saveImages ( )
{
  if ( qsFileName.length ( ) < 1 )
    return;

  uint    t;
  QImage *pImage;
  QString qsFilePath, qsFile;
  QFileInfo fileInfo ( qsFileName );
  qsFile = pCache->getCachePath ( fileInfo.baseName ( )[0] );

  // Next we need to ensure that the path exists ...
  QDir   theDir ( qsFile );
  if ( ! theDir.exists ( ) ) {
    Utils theUtils;
    theUtils.recMkdir ( qsFile );
  }

  qsFilePath = qsFilePath.sprintf ( "%s%016llX", qsFile.ascii ( ), iHashValue );
  for ( t=0; t<iNumberOfThumbs; t++ )  {
    qsFile.sprintf ( "%s_%02d.jpg", qsFilePath.ascii ( ), t );
    pImage = arrayOfThumbs[t];
    pImage->save ( qsFile, "JPEG", 85 );
    //printf ( "  Cache::Thumbs::saveImages <%s>\n", qsFile.ascii ( ) );
  }
}

void Thumbs::Entry::freeImages ( )
{
  if ( ! arrayOfThumbs )
    return;
  
  uint t;
  for ( t=0; t<iNumberOfThumbs; t++ )
    delete arrayOfThumbs[t];
  delete []arrayOfThumbs;
  arrayOfThumbs = NULL;
}

void Thumbs::Entry::scanImages ( QObject *pTarget )
{
  MultiThumbRequest *pRequest = new MultiThumbRequest ( pTarget, this );
  MediaCreator::registerWithMediaScanner ( pRequest );
}

Thumbs::VirtualFolder::VirtualFolder ( Thumbs *pThumbs )
{
  pCache     = pThumbs;
  iHashValue = 0;
}

Thumbs::VirtualFolder::~VirtualFolder ( )
{
  QValueList<VFile *>::iterator it;
  it = listOfFiles.begin ( );
  while ( it != listOfFiles.end ( ) )
    delete *it++;

  listOfFiles.clear ( );
}

bool Thumbs::VirtualFolder::readXml  ( QDomElement *pFolder )
{
  QString qsDateCreated;

  QDomAttr a = pFolder->attributeNode ( VIRTUAL_CREATED );
  qsDateCreated    = a.value ( );
  a = pFolder->attributeNode ( VIRTUAL_NAME );
  qsFolderName     = a.value ( );

  dateCreated = QDateTime::fromString ( qsDateCreated  );
  iHashValue  = (unsigned int)pCache->SDBMHash ( qsFolderName );

  VFile   *pFile;
  QDomNode xmlNode = pFolder->firstChild ( );
  while( ! xmlNode.isNull ( ) )  {
    QDomElement fileElem = xmlNode.toElement ( );
    if ( fileElem.tagName ( ) == VIRTUAL_TAG_ENTRY )  {
      pFile = new VFile;
      pFile->qsFileName     = fileElem.attributeNode  ( VIRTUAL_FILE_NAME ).value ( );
      pFile->iFileHashValue = fileElem.text ( ).toULongLong ( 0, 16 );
      listOfFiles.append    ( pFile );
    }
    xmlNode = xmlNode.nextSibling ( );
  };

  return true;
}

bool Thumbs::VirtualFolder::writeXml ( QDomElement *pFolderElement )
{
  QString qsHash;
  qsHash.sprintf ( "%08X", iHashValue );
  QDomDocument    theDoc = pFolderElement->ownerDocument ( );
  QDomElement  entryNode;
  QDomElement folderNode = theDoc.createElement  ( VIRTUAL_TAG_FOLDER );	// <Folder>
  QDomText     entryText;

  folderNode.setAttribute ( VIRTUAL_HASH,    qsHash );
  folderNode.setAttribute ( VIRTUAL_NAME,    qsFolderName );
  folderNode.setAttribute ( VIRTUAL_CREATED, dateCreated.toString ( ) );

  folderNode.appendChild  ( entryText );

  VFile *pFile;
  QValueList<VFile *>::iterator it;
  it = listOfFiles.begin ( );
  while ( it != listOfFiles.end ( ) ) {
    pFile = *it++;
    qsHash.sprintf ( "%016llX", pFile->iFileHashValue );
    entryText = theDoc.createTextNode ( qsHash );
    entryNode = theDoc.createElement  ( VIRTUAL_TAG_ENTRY );	// <FolderEntry>
    entryNode.setAttribute ( VIRTUAL_FILE_NAME, pFile->qsFileName );
    entryNode.appendChild  ( entryText );
    folderNode.appendChild ( entryNode );
  }

  pFolderElement->appendChild ( folderNode );
  return true;
}

void Thumbs::VirtualFolder::append ( Entry *pEntry )
{
  if ( ! pEntry )
    return;
  VFile *pFile;
  QValueList<VFile *>::iterator it;
  it = listOfFiles.begin ( );
  while ( it != listOfFiles.end ( ) ) {
    pFile = *it++;
    if ( pFile->iFileHashValue == pEntry->iHashValue )
      return;
  }
  pFile = new VFile;
  pFile->iFileHashValue = pEntry->iHashValue;
  pFile->qsFileName     = pEntry->qsFileName;
  listOfFiles.append  ( pFile );
  pCache->saveVirtualFolder ( );
}

void Thumbs::VirtualFolder::clear ( )
{
  QValueList<VFile *>::iterator it;
  it = listOfFiles.begin ( );
  while ( it != listOfFiles.end ( ) )
    delete *it++;
  listOfFiles.clear ( );
}

Thumbs::VirtualFolder *Thumbs::VirtualFolder::clone ( )
{
  VirtualFolder *pFolder = new VirtualFolder ( pCache );
  VFile         *pFile,  *pNewFile;
  QValueList<VFile *>::iterator it;
  it = listOfFiles.begin ( );
  while ( it != listOfFiles.end ( ) ) {
    pFile = *it++;
    if ( pFile ) {
      pNewFile = new VFile;
      pNewFile->qsFileName     = pFile->qsFileName;
      pNewFile->iFileHashValue = pFile->iFileHashValue;
      pFolder ->listOfFiles.append ( pNewFile );
    }
  }

  int      iCount         = 1;
  QString qsClone         = QObject::tr ( "Clone_" );
  pFolder->qsFolderName   = qsClone + qsFolderName;

  while ( pCache->findVirtualFolder ( pFolder->qsFolderName ) ) 
    pFolder->qsFolderName = qsClone + QString ( "%1_" ).arg ( iCount ++ ) + qsFolderName;
    
  pFolder->dateCreated    = QDateTime::currentDateTime ( );
  pFolder->iHashValue     = (unsigned int)pCache->SDBMHash ( pFolder->qsFolderName );

  pCache->getVirtualFolders ( ).append ( pFolder );
  pCache->saveVirtualFolder ( );
  return pFolder;
}

uint Thumbs::VirtualFolder::count ( )
{
  return listOfFiles.count ( );
}

Thumbs::Thumbs ( )
{
  // The first thing to do is to load the cache_db.xml file into memory, 
  // so we know what we have.
  loadCacheDB ( );
  loadVirtualFolder ( );
}

Thumbs::~Thumbs ( )
{
  clearCache ( );
  clearVirtualFolder ( );
}

void Thumbs::clearCache ( )
{
  QValueList<Entry *>::iterator it;
  it = m_listOfThumbs.begin ( );
  while ( it != m_listOfThumbs.end ( ) ) 
    delete *it++;
  m_listOfThumbs.clear ( );
}

void Thumbs::clearVirtualFolder ( )
{
  QValueList<VirtualFolder *>::iterator it;
  it = m_listOfVirtualFolders.begin ( );
  while ( it != m_listOfVirtualFolders.end ( ) ) 
    delete *it++;
  
  m_listOfVirtualFolders.clear ( );
}

void Thumbs::freeImages ( )
{ 
  // to free some space after closing dialog
  Entry *pEntry;
  QValueList<Entry *>::iterator it;
  it = m_listOfThumbs.begin ( );
  while ( it != m_listOfThumbs.end ( ) ) {
    pEntry = *it++;
    pEntry->freeImages ( );
  }
}

QString Thumbs::getCachePath ( QChar c )
{
  QString qsPath = QString ( "%1/.qdvdauthor/cache/%2/" ). arg ( QDir::homeDirPath ( ) ).arg ( c.lower ( ) );
  return qsPath;
}

bool Thumbs::loadCacheDB ( )
{
  // Assign the file
  QString qsPath     = QString ( "%1/.qdvdauthor/cache/" ). arg ( QDir::homeDirPath ( ) );
  QString qsFileName = qsPath + QString ( CACHE_DB_FILE );

  QDir theDir ( qsPath );
  if ( ! theDir.exists ( ) ) {
    Utils theUtils;
    theUtils.recMkdir ( qsPath );
  }

  QFile projectFile ( qsFileName );
  // Check if we have not yet created a cach_db.xml file.
  if ( ! projectFile.exists ( ) )
    return true;

  if ( ! projectFile.open ( IO_ReadOnly ) )
    return false;

  QDomDocument xmlDoc( CACHE_DOCTYPE );
  if ( ! xmlDoc.setContent ( &projectFile ) )  {
    // Error handling ...
    projectFile.close ( );
    MessageBox::warning ( NULL, QObject::tr ("Can not create cache file."),
			  QObject::tr ( "Please make sure you have write permissins set to\n%1" ).arg ( qsPath ),
			  QMessageBox::Ok, QMessageBox::NoButton );
    return false;
  }
  // And at last lets try to read the information of the file.
  QDomElement docElem = xmlDoc.documentElement ( );
  if ( docElem.tagName ( ) != CACHE )
    return false;

  Entry   *pEntry;
  QDomNode xmlNode = docElem.firstChild ( );
  while( ! xmlNode.isNull ( ) )  {
    QDomElement sourceElem = xmlNode.toElement ( );
    if ( sourceElem.tagName ( ) == CACHE_TAG_SOURCE ) {
      pEntry  = new Entry ( this );
      pEntry->readXml ( &sourceElem );
      // If there has been a problem then return false.
      //printf ( "Thumbs::loadCacheDB Entry<%s> <%016llX> \n", pEntry->qsFileName.ascii(), pEntry->iHashValue );
      m_listOfThumbs.append ( pEntry );
    }
    // Otherwise go to the next node ...
    xmlNode = xmlNode.nextSibling ( );
  }
  
  projectFile.close ( );
  return true;
}

bool Thumbs::saveCacheDB ( )
{
  // Assign the file
  QString qsPath     = QString ( "%1/.qdvdauthor/cache/" ). arg ( QDir::homeDirPath ( ) );
  QString qsFileName = qsPath  + QString ( CACHE_DB_FILE );

  QFile projectFile ( qsFileName );
  if ( ! projectFile.open ( IO_WriteOnly ) )
    return false;
  
  QDomDocument xmlDoc( CACHE_DOCTYPE );
  QDomElement  root = xmlDoc.createElement ( CACHE );	// <CacheThumbs>
  // And now proceed to writing the rest of the file.
  xmlDoc.appendChild( root );

  Entry *pEntry = NULL;
  QValueList<Entry *>::iterator it;
  it = m_listOfThumbs.begin ( );
  while ( it != m_listOfThumbs.end ( ) ) {
    pEntry = *it++;
    pEntry->writeXml ( &root );
  }

  QString xml = xmlDoc.toString ( );
  // QDomDocument converts '<' to "&lt;", So if we want comments in
  // the generated file, we should replace them here.
  xml.replace ( "&lt;", "<"); 
  //debug_out ("%s\n", (const char *)xml);
  projectFile.writeBlock ( xml, qstrlen ( xml ) );
  projectFile.close ( );

  return true;
}

void Thumbs::cleanCacheDB ( )
{
  // Weeds out entries which exceed the max age of the record, to keep the cache from growing too big.
}

bool Thumbs::loadVirtualFolder ( )
{
  QString qsPath     = QString ( "%1/.qdvdauthor/cache/" ). arg ( QDir::homeDirPath ( ) );
  QString qsFileName = qsPath + QString ( VIRTUAL_DB_FILE );

  QDir theDir ( qsPath );
  if ( ! theDir.exists ( ) ) {
    Utils theUtils;
    theUtils.recMkdir ( qsPath );
  }

  QFile projectFile ( qsFileName );
  // Check if we have not yet created a cach_db.xml file.
  if ( ! projectFile.exists ( ) )
    return true;

  if ( ! projectFile.open ( IO_ReadOnly ) )
    return false;

  QDomDocument xmlDoc( CACHE_DOCTYPE );
  if ( ! xmlDoc.setContent ( &projectFile ) )  {
    // Error handling ...
    projectFile.close ( );
    MessageBox::warning ( NULL, QObject::tr ("Can not create cache file."),
			  QObject::tr ( "Please make sure you have write permissins set to\n%1" ).arg ( qsPath ),
			  QMessageBox::Ok, QMessageBox::NoButton );
    return false;
  }
  // And at last lets try to read the information of the file.
  QDomElement docElem = xmlDoc.documentElement ( );
  if ( docElem.tagName ( ) != VIRTUAL )
    return false;

  VirtualFolder   *pFolder;
  QDomNode xmlNode = docElem.firstChild ( );
  while( ! xmlNode.isNull ( ) )  {
    QDomElement sourceElem = xmlNode.toElement ( );
    if ( sourceElem.tagName ( ) == VIRTUAL_TAG_FOLDER ) {
      pFolder  = new VirtualFolder ( this );
      pFolder->readXml ( &sourceElem );
      // If there has been a problem then return false.
      //printf ( "Thumbs::loadVirtualFolder Entry<%s> <%08X> \n", pFolder->qsFolderName.ascii(), pFolder->iHashValue );
      m_listOfVirtualFolders.append ( pFolder );
    }
    // Otherwise go to the next node ...
    xmlNode = xmlNode.nextSibling ( );
  }
  
  projectFile.close ( );
  return true;
}

bool Thumbs::saveVirtualFolder ( )
{
  // Assign the file
  QString qsPath     = QString ( "%1/.qdvdauthor/cache/" ). arg ( QDir::homeDirPath ( ) );
  QString qsFileName = qsPath  + QString ( VIRTUAL_DB_FILE );

  QFile projectFile ( qsFileName );
  if ( ! projectFile.open ( IO_WriteOnly ) )
    return false;
  
  QDomDocument xmlDoc( CACHE_DOCTYPE );
  QDomElement  root = xmlDoc.createElement ( VIRTUAL );	// <VirtualFolder>
  // And now proceed to writing the rest of the file.
  xmlDoc.appendChild( root );

  VirtualFolder *pFolder = NULL;
  QValueList<VirtualFolder *>::iterator it;
  it = m_listOfVirtualFolders.begin ( );
  while ( it != m_listOfVirtualFolders.end ( ) ) {
    pFolder = *it++;
    pFolder->writeXml ( &root );
  }

  QString xml = xmlDoc.toString ( );
  // QDomDocument converts '<' to "&lt;", So if we want comments in
  // the generated file, we should replace them here.
  xml.replace ( "&lt;", "<"); 
  //debug_out ("%s\n", (const char *)xml);
  projectFile.writeBlock ( xml, qstrlen ( xml ) );
  projectFile.close ( );

  return true;
}

Thumbs::Entry *Thumbs::find ( QString &qsFileName )
{
  unsigned long long iHash = hashFromFile ( qsFileName );
  return find ( iHash );
}

Thumbs::Entry *Thumbs::find ( unsigned long long iHash )
{
  Entry *pEntry = NULL;
  QValueList<Entry *>::iterator it;
  it = m_listOfThumbs.begin ( );
  while ( it != m_listOfThumbs.end ( ) ) {
    pEntry = *it++;
    //printf ( "  Looking at <%s> = <0x%016llX>\n", pEntry->qsFileName.ascii(), pEntry->iHashValue );
    if ( pEntry->iHashValue == iHash )
      return pEntry;
  }

  return NULL;
}

Thumbs::Entry *Thumbs::append ( QString &qsFile, QString qsLength, uint iThumbs, QImage **ppArray )
{
  // Here we create a entry in the cache for this file and copy the images over
  // plus we have to store them into the cache under $HOME/.qdvdauthor/cache/X/<HasName>_xx.png
  QFileInfo fileInfo ( qsFile );
  Entry *pEntry = new Entry ( this );

  pEntry->qsFileName      = qsFile;
  pEntry->qsLength        = qsLength;
  pEntry->iHashValue      = hashFromFile ( qsFile );
  pEntry->arrayOfThumbs   = ppArray;
  pEntry->iNumberOfThumbs = iThumbs;
  pEntry->dateCreated     = fileInfo.created ( );

  if ( pEntry->iHashValue == 0LL ) {
    delete pEntry;
    return NULL;
  }

  m_listOfThumbs.append ( pEntry );
  saveCacheDB ( );
  
  // And after we have stored the xml file we will save the images in the cache.
  if ( ppArray )
    pEntry->saveImages ( );
  else 
    pEntry->scanImages ( );

  //  printf ( "Cache::Thumbs::append <%s> = <%016llX>\n", pEntry->qsFileName.ascii(), pEntry->iHashValue );
  return pEntry; 
}

Thumbs::Entry *Thumbs::append ( QString &qsFile, QObject *pTarget, uint iThumbs )
{
  // Here we create a entry in the cache for this file and copy the images over
  // plus we have to store them into the cache under $HOME/.qdvdauthor/cache/X/<HasName>_xx.png
  QFileInfo fileInfo ( qsFile );
  Entry *pEntry = new Entry ( this );

  pEntry->qsFileName      = qsFile;
  pEntry->iHashValue      = hashFromFile ( qsFile );
  pEntry->iNumberOfThumbs = iThumbs;
  pEntry->dateCreated     = fileInfo.created ( );

  if ( pEntry->iHashValue == 0LL ) {
    delete pEntry;
    return NULL;
  }

  m_listOfThumbs.append ( pEntry );
  saveCacheDB ( );
  
  pEntry->scanImages ( pTarget );

  //  printf ( "Cache::Thumbs::append <%s> = <%016llX>\n", pEntry->qsFileName.ascii(), pEntry->iHashValue );
  return pEntry; 
}

Thumbs::VirtualFolder *Thumbs::findVirtualFolder ( unsigned int iHash )
{
  VirtualFolder *pVFolder = NULL;
  QValueList<VirtualFolder *>::iterator it;
  it = m_listOfVirtualFolders.begin ( );
  while ( it != m_listOfVirtualFolders.end ( ) ) {
    pVFolder = *it++;
    //printf ( "  Looking at <%s> = <0x%016llX>\n", pEntry->qsFileName.ascii(), pEntry->iHashValue );
    if ( pVFolder->iHashValue == iHash )
      return pVFolder;
  }
  return NULL;
}
  
Thumbs::VirtualFolder *Thumbs::findVirtualFolder ( QString &qsFolderName )
{
  VirtualFolder *pVFolder = NULL;
  QValueList<VirtualFolder *>::iterator it;
  it = m_listOfVirtualFolders.begin ( );
  while ( it != m_listOfVirtualFolders.end ( ) ) {
    pVFolder = *it++;
    //printf ( "  Looking at <%s> = <0x%016llX>\n", pEntry->qsFileName.ascii(), pEntry->iHashValue );
    if ( pVFolder->qsFolderName == qsFolderName )
      return pVFolder;
  }
  return NULL;
}
  
unsigned int Thumbs::addVirtualFolder ( QString &qsFolderName )
{
  unsigned int iHash = (unsigned int)SDBMHash ( qsFolderName );

  VirtualFolder *pVFolder = findVirtualFolder ( iHash );
  if ( pVFolder )
    return 0;  // Ooops folder exists already

  pVFolder = new VirtualFolder ( this );
  pVFolder->qsFolderName = qsFolderName;
  pVFolder->iHashValue   = iHash;
  pVFolder->dateCreated  = QDateTime::currentDateTime ( );

  m_listOfVirtualFolders.append ( pVFolder );
  saveVirtualFolder ( );
  return pVFolder->iHashValue;
}

void Thumbs::deleteVirtualFolder ( QString &qsFolderName )
{
  unsigned int iHash = (unsigned int)SDBMHash ( qsFolderName );
  if ( iHash > 0 )
    deleteVirtualFolder ( iHash );
}

void Thumbs::deleteVirtualFolder ( unsigned int iHash )
{
  VirtualFolder *pVFolder = findVirtualFolder ( iHash );
  //  printf ( "Thumbs::deleteVirtualFolder <%08X><%p>\n", iHash, pVFolder );
  if ( pVFolder ) {
    m_listOfVirtualFolders.remove ( pVFolder );
    delete pVFolder;
  }
  saveVirtualFolder ( );
}

QValueList<Thumbs::VirtualFolder *> &Thumbs::getVirtualFolders ( )
{
  return m_listOfVirtualFolders;
}

QValueList<Thumbs::Entry *> Thumbs::getScannedThumbs ( )
{
  QValueList<Entry *> list;

  Entry *pEntry;
  QValueList<Entry *>::iterator it;
  it = m_listOfThumbs.begin ( );
  while ( it != m_listOfThumbs.end ( ) ) {
    pEntry = *it++;
    if ( pEntry->iScanStatus == 2 ) // 2 == Scanning done
      list.append ( pEntry );
  }

  return list;
}

QValueList<Thumbs::Entry *> Thumbs::getThumbsByStar ( uint iStar )
{
  QValueList<Entry *> list;

  Entry *pEntry;
  QValueList<Entry *>::iterator it;
  it = m_listOfThumbs.begin ( );
  while ( it != m_listOfThumbs.end ( ) ) {
    pEntry = *it++;
    if ( pEntry->iStarRating ==  iStar )
      list.append ( pEntry );
  }

  return list;
}

unsigned long long Thumbs::hashFromFile ( QString &qsFileName )
{
  QString   qsFingerprint;
  QFileInfo fileInfo ( qsFileName );

  // Sanity check.
  if ( ! fileInfo.exists ( ) )
    return 0LL;

  uint      iSize = fileInfo.size    ( );
  QDateTime  time = fileInfo.created ( );
  qsFingerprint = qsFileName + QString ( "%1" ).arg ( iSize ) + time.toString ( "yyyyMMddhhmmss" );

  return SDBMHash ( qsFingerprint );
}

unsigned long long Thumbs::SDBMHash ( QString &str )
{
   unsigned long long hash = 0;
   for ( unsigned int i = 0; i < str.length ( ); i++ )
     hash = str[i].latin1 ( ) + (hash << 6) + (hash << 16) - hash;

   return hash;
}

}; // End of namespace Thumbs
