/*
 *
 *    soniK digital audio editor
 *    Copyright (C) 2003-2006  Robert Walker <rob@tenfoot.org.uk>
 *
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */
#include "uicontrol.h"

#include "controls/slider.h"

#include <kdebug.h>

#include <qmetaobject.h>

using Sonik::UiControl;
using Sonik::UiControlPtrList;

UiControl::UiControl()
  : mId(QString::null), mName(QString::null), mWidget(0)
{
}

UiControl::UiControl(const QString& id, const QString& name,
                     QWidget *widget, const QString& valProperty)
  : mId(id), mName(name), mWidget(widget), mValProperty(valProperty)
{
}

UiControl::UiControl(const QString& id, const QString& name, Type type,
                     float val, float min, float max,
                     int precision, bool log,
                     QWidget* parent)
  : mId(id), mName(name)
{
  // should really be subclasses
  switch (type)
  {
    case HSLIDER:
      mWidget = new Slider(name, min, max, precision, (max-min)/100,
                           log, Qt::Horizontal, parent, id);
      mValProperty = "value";
      break;

    case VSLIDER:
      mWidget = new Slider(name, min, max, precision, (max-min)/100,
                           log, Qt::Vertical, parent, id);
      mValProperty = "value";
      break;

    case KNOB:
      break;

    case SWITCH:
      break;

    default:
      break;
  }

  setValue(val);

  QVariant v = mWidget->property(mValProperty);
  kdDebug(60606) << "UiControl::UiControl: "
                 << v.toDouble(0) << ", " << value()
                 << "\n";
}

UiControl::~UiControl()
{
}

void UiControl::setWidget(QWidget* widget, const QString& valProperty)
{
  mWidget = widget;
  mValProperty = valProperty;
}

QWidget* UiControl::widget()
{
  return mWidget;
}

float UiControl::value() const
{
  if (!mWidget)
    return 0.0f;

  QVariant value = mWidget->property(mValProperty);
  if (!value.isValid())
    return 0.0f;

  return value.toDouble(0);
}

void UiControl::setValue(float v)
{
  if (mWidget)
  {
    QVariant value(v);
    mWidget->setProperty(mValProperty, value);
  }
}


QString UiControl::typeToString(Type t)
{
  QString res;

  switch (t)
  {
    case NONE:    res = "NONE";        break;
    case HSLIDER: res = "HSLIDER";     break;
    case VSLIDER: res = "VSLIDER";     break;
    case KNOB:    res = "KNOB";        break;
    case SWITCH:  res = "SWITCH";      break;
    case CUSTOM:  res = "CUSTOM";      break;
    default:      res = QString::null; break;
  }

  return res;
}


UiControl::Type UiControl::stringToType(const QString& s)
{
  Type res = NONE;
  QString n = s.upper();

  if (n == "HSLIDER")
    res = HSLIDER;
  else if (n == "VSLIDER")
    res = VSLIDER;
  else if (n == "KNOB")
    res = KNOB;
  else if (n == "SWITCH")
    res = SWITCH;
  else if (n == "CUSTOM")
    res = CUSTOM;

  return res;
}


UiControl* UiControlPtrList::find(const QString& id) const
{
  for (UiControlPtrList::const_iterator it = begin();
       it != end();
       ++it)
  {
    if ((*it)->id() == id)
      return (*it);
  }

  return 0;
}

void UiControlPtrList::alignAllHorizontalSliders()
{
  SliderPtrVector sliders;

  for (UiControlPtrList::iterator it = begin();
       it != end();
       ++it)
  {
    Slider* slider = dynamic_cast<Slider *>((*it)->widget());
    if (slider)
      sliders.push_back(slider);
  }

  Sonik::alignHorizontalSliders(sliders);
}
