#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "libpcm.h"
#include "libaiff.h"
#include "libwave.h"
#include "../libtoolame/toolame.h"

#define MP2BUFSIZE 16384
#define AUDIOBUFSIZE 8192

#define MAX_NAME_SIZE 256

typedef struct frontend_options_struct {
  int singleFrameMode;
  int downmix;
  int byteswap;
  int channelswap;
  int sampleFreq;
  int numChannels;
} frontend_options;

frontend_options *frontend_init(void) {
  frontend_options *frontOptions;
  
  frontOptions = (frontend_options *)calloc(1, sizeof(frontend_options));
  frontOptions->singleFrameMode = FALSE;
  frontOptions->downmix = FALSE;
  frontOptions->byteswap = FALSE;
  frontOptions->channelswap = FALSE;
  frontOptions->sampleFreq = 44100;
  frontOptions->numChannels = 2;
  return(frontOptions);
}

/************************************************************************
*
* usage
*
* PURPOSE:  Writes command line syntax to the file specified by #stderr#
*
************************************************************************/

void usage (void)
{				/* print syntax & exit */
  /* FIXME: maybe have an option to display better definitions of help codes, and
     long equivalents of the flags */
  fprintf (stdout, "\ntooLAME version %s (http://toolame.sourceforge.net)\n", "0.2m");
  fprintf (stdout, "MPEG Audio Layer II encoder\n\n");
  fprintf (stdout, "usage: \n");
  fprintf (stdout, "\ttooLAME [options] <input> <output>\n\n");

  fprintf (stdout, "Options:\n");
  fprintf (stdout, "Input\n");
  fprintf (stdout, "\t-a       downmix from stereo to mono\n");
  fprintf (stdout, "\t-x       force byte-swapping of input\n");
  fprintf (stdout, "\t-g       swap channels of input file\n");
  fprintf (stdout, "\t-N nch   Number of channels for raw PCM input (def: 2)\n");
  fprintf (stdout, "\t-s sfrq  input smpl rate in Hz for raw PCM (def: 44100)\n");
  fprintf (stdout, "Output\n");
  fprintf (stdout, "\t-m mode  channel mode : s/d/j/m   (dflt s)\n");
  fprintf (stdout, "\t-p psy   psychoacoustic model 0/1/2/3 (dflt 3)\n");
  fprintf (stdout, "\t-b br    total bitrate in kbps    (dflt 192)\n");
  fprintf (stdout, "\t-v lev   vbr mode\n");
  fprintf (stdout, "\t-l lev   ATH level (dflt 0)\n");
  fprintf (stdout, "Operation\n");
  fprintf (stdout,
	   "\t-q num   quick mode. only calculate psy model every num frames\n");
  fprintf (stdout, "Misc\n");
  fprintf (stdout, "\t-d emp   de-emphasis n/5/c        (dflt: (n)one)\n");
  fprintf (stdout, "\t-c       mark as copyright\n");
  fprintf (stdout, "\t-o       mark as original\n");
  fprintf (stdout, "\t-e       add error protection\n");
  fprintf (stdout, "\t-r       force padding bit/frame on\n");
  fprintf (stdout, "\t-D len   add DAB extensions of length [len]\n");
  fprintf (stdout, "\t-t       talkativity 0=no messages (dflt 2)\n");
  fprintf (stdout, "\t-u ind   Set the upper bitrate when in VBR mode\n");
  fprintf (stdout, "\t-R num   Set the number of reserved bits at the end of frame\n");
  fprintf (stdout, "\t-E       turn on energy level extensions\n");
  fprintf (stdout, "Files\n");
  fprintf (stdout,
	   "\tinput    input sound file. (WAV,AIFF,PCM or use '/dev/stdin')\n");
  fprintf (stdout, "\toutput   output bit stream of encoded audio\n");
  fprintf (stdout,
	   "\n\tAllowable bitrates for 16, 22.05 and 24kHz sample input\n");
  fprintf (stdout,
	   "\t   8,  16,  24,  32,  40,  48,  56,  64,  80,  96, 112, 128, 144, 160\n");
  fprintf (stdout,
	   "\n\tAllowable bitrates for 32, 44.1 and 48kHz sample input\n");
  fprintf (stdout,
	   "\t  32,  48,  56,  64,  80,  96, 112, 128, 160, 192, 224, 256, 320, 384\n");
  fprintf (stdout,
	   "brate indx 1    2    3    4    5    6    7    8    9   10   11   12   13   14\n");

  fprintf(stdout,"\n");
  exit (1);
}

/************************************************************************
*
* parse_args
*
* PURPOSE:  Sets encoding parameters to the specifications of the
* command line.  Default settings are used for parameters
* not specified in the command line.
*
* SEMANTICS:  The command line is parsed according to the following
* syntax:
*
* -m  is followed by the mode
* -p  is followed by the psychoacoustic model number
* -s  is followed by the sampling rate
* -b  is followed by the total bitrate, irrespective of the mode
* -d  is followed by the emphasis flag
* -c  is followed by the copyright/no_copyright flag
* -o  is followed by the original/not_original flag
* -e  is followed by the error_protection on/off flag
* -f  turns off psy model (fast mode)
* -q <i>  only calculate psy model every ith frame
* -a  downmix from stereo to mono 
* -r  turn off padding bits in frames.
* -x  force byte swapping of input
* -g  swap the channels on an input file
* -t  talkativity. how verbose should the program be. 0 = no messages. 
* -u  is followed by the upper bitrate for VBR mode
* -R  is followed by an int argument for the number of reserved bits at the end of the frame
*
* If the input file is in AIFF format, the sampling frequency is read
* from the AIFF header.
*
* The input and output filenames are read into #inpath# and #outpath#.
*
************************************************************************/
void parse_args (int argc, char **argv, toolame_options *glopts, frontend_options *frontOptions, 
		 char inPath[MAX_NAME_SIZE], char outPath[MAX_NAME_SIZE])
{
  int brate;
  int err = 0, i = 0;

  brate = toolame_getBitrate(glopts);

  /* preset defaults */
  inPath[0] = '\0';
  outPath[0] = '\0';

  /* process args */
  while (++i < argc && err == 0) {
    char c, *token, *arg, *nextArg;
    int argUsed;

    token = argv[i];
    if (*token++ == '-') {
      if (i + 1 < argc)
	nextArg = argv[i + 1];
      else
	nextArg = "";
      argUsed = 0;
      if (!*token) {
	/* The user wants to use stdin and/or stdout. */
	if (inPath[0] == '\0')
	  strncpy (inPath, argv[i], MAX_NAME_SIZE);
	else if (outPath[0] == '\0')
	  strncpy (outPath, argv[i], MAX_NAME_SIZE);
      }
      while ((c = *token++)) {
	if (*token /* NumericQ(token) */ )
	  arg = token;
	else
	  arg = nextArg;
	switch (c) {
	case 'm':
	  argUsed = 1;
	  if (*arg == 's') {
	    toolame_setMode(glopts, MPG_MD_STEREO);
	  } else if (*arg == 'd') {
	    toolame_setMode(glopts, MPG_MD_DUAL_CHANNEL);
	  } else if (*arg == 'j') {
	    toolame_setMode(glopts, MPG_MD_JOINT_STEREO);
	  } else if (*arg == 'm') {
	    toolame_setMode(glopts, MPG_MD_JOINT_STEREO);
	  } else {
	    fprintf (stderr, "-m mode must be s/d/j/m not %s\n", arg);
	    err = 1;
	  }
	  break;
	case 'p':
	  toolame_setPsymodel(glopts, atoi (arg));
	  argUsed = 1;
	  break;

	case 's':
	  argUsed = 1;
	  toolame_setSampleFreq(glopts, atoi (arg));	    
	  frontOptions->sampleFreq = atoi(arg);
	  break;
	case 'N':
	  argUsed = 1;
	  frontOptions->numChannels = atoi(arg);
	  break;
	case 'b':
	  argUsed = 1;
	  toolame_setBitrate(glopts, atoi (arg));
	  break;
	case 'd':
	  argUsed = 1;
	  if (*arg == 'n')
	    toolame_setEmphasis(glopts, 0);
	  else if (*arg == '5')
	    toolame_setEmphasis(glopts, 1);
	  else if (*arg == 'c')
	    toolame_setEmphasis(glopts, 3);
	  else {
	    fprintf (stderr, "-d emp must be n/5/c not %s\n", arg);
	    err = 1;
	  }
	  break;
	case 'D':
	  argUsed = 1;
	  toolame_setErrorProtection(glopts, TRUE);
	  toolame_setDAB(glopts, TRUE);
	  toolame_setDABCRCLength(glopts, 2);
	  toolame_setDABXPADLength(glopts, atoi(arg));
	  // 2 bytes for default DAB + 2 bytes for CRC + Xpad bytes
	  toolame_setNumAncillaryBits(glopts, (2 + 2 + atoi(arg) ) * 8);

	  frontOptions->singleFrameMode = TRUE;
	  break;
	case 'c':
	  toolame_setCopyright(glopts, TRUE);
	  break;
	case 'o':
	  toolame_setOriginal(glopts, TRUE);
	  break;
	case 'e':
	  toolame_setErrorProtection(glopts, TRUE);
	  break;
	case 'r':
	  toolame_setPadding(glopts, TRUE);
	  break;
	case 'q':
	  argUsed = 1;
  	  toolame_setQuickMode(glopts, TRUE);
	  toolame_setQuickCount(glopts, atoi(arg));
	  break;
	case 'a':
	  frontOptions->downmix = TRUE;
	  toolame_setMode(glopts, MPG_MD_MONO);
	  break;
	case 'x':
	  frontOptions->byteswap = TRUE;
	  break;
	case 'v':
	  argUsed = 1;
	  toolame_setVBR(glopts, TRUE);
	  toolame_setVBRLevel(glopts, atof (arg));
	  toolame_setPadding(glopts, FALSE);	/* don't use padding for VBR */
	  /* MFC Feb 2003: in VBR mode, joint stereo doesn't make
	     any sense at the moment, as there are no noisy subbands 
	     according to bits_for_nonoise in vbr mode */
	  toolame_setMode(glopts, MPG_MD_STEREO); /* force stereo mode */
	  break;
	case 'l':
	  argUsed = 1;
	  toolame_setATHLevel(glopts,atof(arg));
	  break;
	case 'h':
	  usage ();
	  break;
	case 'g':
	  frontOptions->channelswap = TRUE;
	  break;
	case 't':
	  argUsed = 1;
	  toolame_setVerbosity(glopts, atoi(arg));
	  break;
	case 'u':
	  argUsed = 1;
	  toolame_setVBRUpperBitrate(glopts, atoi(arg));
	  break;
	case 'R':
	  argUsed = 1;
	  toolame_setNumAncillaryBits(glopts, atoi(arg));
	  // MFC FIX: Need to cross validate this option
	  // with the energylevel (-E) setting.
	  break;
	case 'E':
	  toolame_setEnergyLevels(glopts, TRUE);
	  toolame_setNumAncillaryBits(glopts, 40); // 5 bytes for the stereo energy info
	  // NOTE: Assume it's stereo, and then change to 2 bytes
	  // if it turns out to be mono
	  // MFC FIX: This option must be mutually exclusive with the 
	  // reservebits (-R) option *UNLESS* the number
	  // of explicitly reserved bits > 5 bytes.
	  break;
	default:
	  fprintf (stderr, "Unrecognised option %c\n", c);
	  err = 1;
	  break;
	}
	if (argUsed) {
	  if (arg == token)
	    token = "";		/* no more from token */
	  else
	    ++i;		/* skip arg we used */
	  arg = "";
	  argUsed = 0;
	}
      }
    } else {
      if (inPath[0] == '\0')
	strcpy (inPath, argv[i]);
      else if (outPath[0] == '\0')
	strcpy (outPath, argv[i]);
      else {
	fprintf (stderr, "excess arg %s\n", argv[i]);
	err = 1;
      }
    }
  }

  if (toolame_getDAB(glopts)) {
    /* in 48 kHz */
    /* if the bit rate per channel is less then 56 kbit/s, we have 2 scf-crc */
    /* else we have 4 scf-crc */
    /* in 24 kHz, we have 4 scf-crc, see main loop */
    if (brate / (toolame_getMode(glopts) == MPG_MD_MONO ? 1 : 2) >= 56)
      toolame_setDABCRCLength(glopts, 4);
  }


  if (err || inPath[0] == '\0')
    usage ();			/* If no infile defined, or err has occured, then call usage() */

  if (outPath[0] == '\0') {
    /* replace old extension with new one, 1992-08-19, 1995-06-12 shn */
    toolame_newExtension (inPath, ".mp2", outPath);
  }

}

/************************************************************************
*
* print_config
*
* PURPOSE:  Prints the encoding parameters used
*
************************************************************************/

void print_config (toolame_options *glopts, frontend_options *frontOptions, char *inPath,  char *outPath)
{

  if (!toolame_getVerbosity(glopts))
    return;

  fprintf (stderr, "--------------------------------------------\n");
  fprintf (stderr, "Input File : '%s'   %.1f kHz\n",
	   (strcmp (inPath, "-") ? inPath : "stdin"), toolame_getSampleFreq(glopts)/1000.0);
  fprintf (stderr, "Output File: '%s'\n",
	   (strcmp (outPath, "-") ? outPath : "stdout"));
  fprintf (stderr, "%d kbps ", toolame_getBitrate(glopts) );
  fprintf (stderr, "%s ", toolame_getVersionName(toolame_getVersion(glopts)));
  if (toolame_getMode(glopts) != MPG_MD_JOINT_STEREO)
    fprintf (stderr, "Layer II %s Psycho model=%d \n",
	     toolame_getModeName(toolame_getMode(glopts)), toolame_getPsymodel(glopts));
  else
    fprintf (stderr, "Layer II %s Psy model %d \n", toolame_getModeName(toolame_getMode(glopts)),
	     toolame_getPsymodel(glopts));

  fprintf (stderr, "[De-emph:%s\tCopyright:%s\tOriginal:%s\tCRC:%s]\n",
	   ((toolame_getEmphasis(glopts)) ? "On" : "Off"),
	   ((toolame_getCopyright(glopts)) ? "Yes" : "No"),
	   ((toolame_getOriginal(glopts)) ? "Yes" : "No"),
	   ((toolame_getErrorProtection(glopts)) ? "On" : "Off"));

  fprintf (stderr, "[Padding:%s\tByte-swap:%s\tChanswap:%s\tDAB:%s]\n",
	   ((toolame_getPadding(glopts)) ? "Normal" : "Off"),
	   ((frontOptions->byteswap) ? "On" : "Off"),
	   ((frontOptions->channelswap) ? "On" : "Off"),
	   ((toolame_getDAB(glopts)) ? "On" : "Off")); 

  if (toolame_getVBR(glopts))
    fprintf (stderr, "VBR Enabled. Using MNR boost of %f\n", toolame_getVBRLevel(glopts));
  fprintf(stderr,"ATH adjustment %f\n",toolame_getATHLevel(glopts));
  fprintf(stderr,"Reserving %i Ancillary bits\n", toolame_getNumAncillaryBits(glopts));

  fprintf (stderr, "--------------------------------------------\n");
}



/*********************************************
 * void short_usage(void)
 ********************************************/
void short_usage (void)
{
  /* print a bit of info about the program */
  fprintf (stderr, "tooLAME (http://toolame.sourceforge.net)\n");
  fprintf (stderr, "MPEG Audio Layer II encoder\n\n");
  fprintf (stderr, "USAGE: tooLAME [options] <infile> [outfile]\n\n");
  fprintf (stderr, "Try \"tooLAME -h\" for more information.\n");
  exit (0);
}


int main (int argc, char **argv) {
  toolame_options *encodeOptions;
  frontend_options *frontOptions;
  char inputfilename[256], outputfilename[256];
  int i;
  FILE *outfile;
  short int *pcmaudio[2];
  int num_samples;
  int frames=0;
  unsigned char mp2buffer[MP2BUFSIZE]; //mp2buffer will never be bigger than PCM input
  int mp2buffer_size = MP2BUFSIZE;
  int mp2fill_size;
  sound_info_t *sound_info = NULL;
  int audioReadSize = AUDIOBUFSIZE; // Default to reading AUDIOBUFSIZE bytes of audio data *per channel*

  if (argc == 1) 
    short_usage();

  /* Allocate some space for the PCM audio data */
  for (i=0;i<2;i++)
    if ( (pcmaudio[i]=(short int *)calloc(AUDIOBUFSIZE, sizeof(short int)))==NULL) {
      fprintf(stdout,"pcmaudio %i alloc failed\n", i);
      exit(99);
    }

  /* grab a set of default options */
  encodeOptions = toolame_init();
  frontOptions = frontend_init();

  /* tweak them or get them from the command line */
  parse_args(argc, argv, encodeOptions, frontOptions, inputfilename, outputfilename);

  /* Frontend is responsible for opening sound file */
  if ( (sound_info = aiff_init(inputfilename)) != NULL ) {
    // This is an aiff input file
  } else {
    if ( (sound_info = wave_init(inputfilename)) != NULL) {
      // This is a wave input file 
    } else {
      // Force toolame to interpret the file as raw PCM
      if ( (sound_info = pcm_init(inputfilename, frontOptions->numChannels, frontOptions->sampleFreq)) != NULL) {
	// This is a raw pcm file
      }
    }
  }    
  
  if (sound_info == NULL) {
    fprintf(stdout,"No input file opened\n");
    exit(99);
  } else {
    // Use sound file to over-ride preferences for
    // mono/stereo and sampling-frequency
    if (sound_info->numChannels == 1) 
      toolame_setMode(encodeOptions, MPG_MD_MONO);
    toolame_setSampleFreq(encodeOptions, sound_info->sampleRate);
    sound_info->byteswap = frontOptions->byteswap;
  }
  
  /* If energy information is required, see if we're in MONO mode
     in which case, we only need 16 bits of ancillary data */
  if ( toolame_getEnergyLevels(encodeOptions) )
    if ( toolame_getMode(encodeOptions)==MPG_MD_MONO )
      toolame_setNumAncillaryBits(encodeOptions, 16); // only 2 bytes needed for energy level for mono channel

  /* initialise toolame with this set of options */
  toolame_init_params(encodeOptions);
  
  /* dump the config for a look at what you're about to do */
  print_config(encodeOptions, frontOptions, inputfilename, outputfilename);
  
  /* Open the output file for the encoded MP2 data */
  if ((outfile = fopen(outputfilename, "w"))==0) {
    fprintf(stdout,"output file error opening %s\n",outputfilename);
    exit(99);
  }

  if (frontOptions->singleFrameMode)
    audioReadSize = 1152;
  
  /* Now do the buffering/encoding/writing */
  while ( (num_samples = pcm_get_audio(sound_info, pcmaudio, audioReadSize)) != 0 ) { 
    // Read num_samples of  audio data *per channel* from the input file
    frames += toolame_encode_buffer(encodeOptions, pcmaudio[0], pcmaudio[1], num_samples, mp2buffer, mp2buffer_size, &mp2fill_size);
    fwrite(mp2buffer, sizeof(unsigned char), mp2fill_size, outfile);
    fprintf(stdout,"[%04i]\r", frames);
    fflush(stdout);
  }
  
  /* flush any remaining audio. (don't send any new audio data)
     There should only ever be a max of 1 frame on a flush.
     There may be zero frames if the audio data was an exact
     multiple of 1152 */
  toolame_encode_flush(encodeOptions, mp2buffer, mp2buffer_size, &mp2fill_size);
  fwrite(mp2buffer, sizeof(unsigned char), mp2fill_size, outfile);

  toolame_deinit(encodeOptions);
  pcm_deinit();
  fprintf(stdout,"Finished nicely\n");
  return(0);
}
