#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include "libpcm.h"
#include "libaiff.h"

typedef float Single;

#define	kFloatLength	4
#define	kDoubleLength	8
#define	kExtendedLength	10

# define FloatToUnsigned(f)	((unsigned long)(((long)((f) - 2147483648.0)) + 2147483647L + 1))
# define UnsignedToFloat(u)	(((double)((long)((u) - 2147483647L - 1))) + 2147483648.0)

double ConvertFromIeeeExtended (char *bytes)
{
  double f;
  long expon;
  unsigned long hiMant, loMant;

  expon = ((bytes[0] & 0x7F) << 8) | (bytes[1] & 0xFF);
  hiMant = ((unsigned long) (bytes[2] & 0xFF) << 24)
    | ((unsigned long) (bytes[3] & 0xFF) << 16) |
    ((unsigned long) (bytes[4] & 0xFF) << 8) |
    ((unsigned long) (bytes[5] & 0xFF));
  loMant = ((unsigned long) (bytes[6] & 0xFF) << 24)
    | ((unsigned long) (bytes[7] & 0xFF) << 16) |
    ((unsigned long) (bytes[8] & 0xFF) << 8) |
    ((unsigned long) (bytes[9] & 0xFF));

  if (expon == 0 && hiMant == 0 && loMant == 0) {
    f = 0;
  } else {
    if (expon == 0x7FFF) {	/* Infinity or NaN */
      f = HUGE_VAL;
    } else {
      expon -= 16383;
      f = ldexp (UnsignedToFloat (hiMant), expon -= 31);
      f += ldexp (UnsignedToFloat (loMant), expon -= 32);
    }
  }

  if (bytes[0] & 0x80)
    return -f;
  else
    return f;
}

void ReadBytes (FILE *fp, char *p, int n)
{
  while (!feof (fp) & (n-- > 0))
    *p++ = getc (fp);
}

int Read16BitsHighLow (FILE *fp)
{
  int first, second, result;

  first = 0xff & getc (fp);
  second = 0xff & getc (fp);

  result = (first << 8) + second;
  return (result);
}

double ReadIeeeExtendedHighLow (FILE *fp)
{
  char bits[kExtendedLength];

  ReadBytes (fp, bits, kExtendedLength);
  return ConvertFromIeeeExtended (bits);
}

int Read32BitsHighLow (FILE *fp)
{
  int first, second, result;

  first = 0xffff & Read16BitsHighLow (fp);
  second = 0xffff & Read16BitsHighLow (fp);

  result = (first << 16) + second;
  return (result);
}




/*****************************************************************************
 *
 *  Read Audio Interchange File Format (AIFF) headers.
 *
 *****************************************************************************/

int aiff_read_headers (FILE * file_ptr, sound_info_t * aiff_ptr)
{
  int chunkSize, subSize, sound_position;
  unsigned long offset;

  if (fseek (file_ptr, 0, SEEK_SET) != 0)
    return -1;

  if (Read32BitsHighLow (file_ptr) != IFF_ID_FORM)
    return -1;

  chunkSize = Read32BitsHighLow (file_ptr);

  if (Read32BitsHighLow (file_ptr) != IFF_ID_AIFF)
    return -1;

  sound_position = 0;
  while (chunkSize > 0) {
    chunkSize -= 4;
    switch (Read32BitsHighLow (file_ptr)) {

    case IFF_ID_COMM:
      chunkSize -= subSize = Read32BitsHighLow (file_ptr);
      aiff_ptr->numChannels = Read16BitsHighLow (file_ptr);
      subSize -= 2;
      aiff_ptr->numSamples = Read32BitsHighLow (file_ptr);
      subSize -= 4;
      aiff_ptr->sampleSize = Read16BitsHighLow (file_ptr);
      subSize -= 2;
      aiff_ptr->sampleRate = (int)ReadIeeeExtendedHighLow (file_ptr);
      subSize -= 10;
      while (subSize > 0) {
	getc (file_ptr);
	subSize -= 1;
      }
      break;

    case IFF_ID_SSND:      
      chunkSize -= subSize = Read32BitsHighLow (file_ptr);
      //aiff_ptr->blkAlgn.offset = Read32BitsHighLow (file_ptr);
      offset = Read32BitsHighLow (file_ptr);
      subSize -= 4;
      //aiff_ptr->blkAlgn.blockSize = Read32BitsHighLow (file_ptr);
      Read32BitsHighLow (file_ptr);
      subSize -= 4;
      sound_position = ftell (file_ptr) + offset; //aiff_ptr->blkAlgn.offset;
      if (fseek (file_ptr, (long) subSize, SEEK_CUR) != 0)
	return -1;
      aiff_ptr->sampleType = IFF_ID_SSND;
      break;
      
    default:
      chunkSize -= subSize = Read32BitsHighLow (file_ptr);
      while (subSize > 0) {
	getc (file_ptr);
	subSize -= 1;
      }
      break;
    }
  }


  return sound_position;
}

/*****************************************************************************
 *
 *  Seek past some Audio Interchange File Format (AIFF) headers to sound data.
 *
 *****************************************************************************/

int aiff_seek_to_sound_data (FILE * file_ptr)
{
  if (fseek
      (file_ptr, AIFF_FORM_HEADER_SIZE + AIFF_SSND_HEADER_SIZE,
       SEEK_SET) != 0)
    return (-1);
  return (0);
}

sound_info_t *aiff_init(char *inPath)
{
  sound_info_t *file_info;
  long soundPosition;
  FILE *infile;

  if ( (infile=fopen(inPath, "r")) == NULL) {
    fprintf(stdout,"Error init'ing aiff file\n");
    return(NULL);
  }

  file_info = (sound_info_t *)calloc(1, sizeof(sound_info_t));
  file_info->soundfile = infile;

  if ((soundPosition = aiff_read_headers (infile, file_info)) != -1) {
    fprintf (stderr, ">>> Using Audio IFF sound file headers\n");
    if (fseek (infile, soundPosition, SEEK_SET) != 0) {
      fprintf (stderr, "Could not seek to PCM sound data in \"%s\".\n",
	       inPath);
      fclose(file_info->soundfile);
      free(file_info);
      return(NULL);
    }
    fprintf (stderr, "Parsing AIFF audio file \n");
    fprintf (stderr, ">>> %i Hz sampling frequency selected\n", file_info->sampleRate);

    /* Determine number of samples in sound file */
    //    glopts->num_samples = file_info.numChannels * file_info.numSampleFrames;


    return(file_info);
  }

  fclose(file_info->soundfile);
  free(file_info);
  return(NULL);

}

