/***************************************************************************
 *   Copyright (C) 2005 by Sergio Pistone                                  *
 *   sergio_pistone@yahoo.com.ar                                           *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "application.h"
#include <pluginsmanager.h>

#include <signal.h>

#include <kstandarddirs.h>
#include <klocale.h>
#include <kurl.h>
#include <dcopref.h>
#include <kmessagebox.h>
#include <kpassivepopup.h>

using namespace TransKode;

Application::Application():
	KApplication(),
	m_listener(),
	m_logger( Logger::instance() ),
	m_config( ::locate( "data", "amarok/scripts/transkode/defaultsrc" ) ),
	m_configDialog( ::locate( "data", "amarok/scripts/transkode/defaultsrc" ), true ),
	m_commonQueue(),
	m_amarokQueue()
{
	signal( SIGTERM, cleanUp );

	m_config.load( ::locate( "data", "amarok/scripts-data/transkoderc" ) );
	m_configDialog.setConfig( m_config );

	m_logger.setFile( m_config.generalOption( GeneralOpt::log ), true );
	m_logger.setMsgPrefix( "[script] " );

	// If there are no user defined settings show config dialog
	if ( ! System::isReadable( ::locate( "data", "amarok/scripts-data/transkoderc" ) ) )
		slotPreferences();

	connect( &m_configDialog, SIGNAL( configSaved() ), this, SLOT( slotConfig() ) ) ;
	slotConfig();

	m_logger.log( "starting main worker thread" );
	new Worker( m_commonQueue );

	m_logger.log( "starting Amarok's TTMD dedicated worker thread" );
	new Worker( m_amarokQueue );

	m_jobs.setAutoDelete( true ); // only a precaution

	connect( &m_listener, SIGNAL( communicationClosed() ), this, SLOT( quit() ) );
	connect( &m_listener, SIGNAL( configureRequested() ), this, SLOT( slotPreferences() ) );
	connect( &m_listener, SIGNAL( transferToMediaDeviceRequested(const QString&, const QString&) ),
			 this, SLOT( addTransferToMediaDeviceJob(const QString&, const QString&) ) );
	connect( &m_listener, SIGNAL( transcodeRequested(const QStringList&, const QString&) ),
			 this, SLOT( addTranscodeJobs(const QStringList&, const QString&) ) );

	connect( &m_commonQueue, SIGNAL( mtWorkersIdle(long,int,int,int) ), this, SLOT( displayRoundStats(long,int,int,int) ) );
	connect( &m_amarokQueue, SIGNAL( mtWorkersIdle(long) ), this, SLOT( amarokDone() ) );
}

void Application::cleanUp( int ) { kapp->quit(); }
Application::~Application()
{
	setupMenuEntries( remove ); // Remove menu entries
}

void Application::slotPreferences()
{
// 	QSize size = m_configDialog.size();
// 	m_configDialog.setInitialSize( size );
	m_configDialog.show();
}

void Application::slotConfig()
{
	setupMenuEntries( remove ); // remove old menu entries
	QString oldTargetFormats = m_config.supportedEncodingExtensions(); // save previously supported formats
	QString oldLogFile = m_logger.file(); // save previous log file

	m_config = m_configDialog.getConfig();
	m_config.save( ::locateLocal( "data", "amarok/scripts-data/transkoderc" ) );

	setupMenuEntries( add ); // insert new menu entries

	if ( m_config.generalOption( GeneralOpt::log ) != oldLogFile )
	{
		m_logger.setFile( m_config.generalOption( GeneralOpt::log ) );
		m_logger.truncate();
	}

	// NOTE: if the script was installed globally, only root will be able to update this file!
// 	QString newTargetFormats = m_config.supportedEncodingExtensions();
// 	if ( newTargetFormats != oldTargetFormats )
// 	{
// 		KConfig specsFile( ::locate( "data", "amarok/scripts/transkode/transkode.spec" ) );
// 		specsFile.setGroup( "Transcode" );
// 		specsFile.writeEntry( "target_formats", newTargetFormats );
// 	}
}

void Application::setupMenuEntries( TAction action )
{
	QCString function = action == add ? "addCustomMenuItem(QString,QString)" : "removeCustomMenuItem(QString,QString)";
	DCOPRef amarokscript( "amarok", "script" );
	QStringList profiles = m_config.validProfiles();
	for ( QStringList::Iterator it = profiles.begin(); it != profiles.end(); ++it )
	{
		EncodingProfile* options = m_config.profileOptions( *it );
		if ( options == 0 )
			continue;
		if ( options->show() )
			amarokscript.send( function, QString( "transKode" ), QString( "[ " + *it + " ]" ) );
		delete options;
	}
}

void Application::addTransferToMediaDeviceJob( const QString& url, const QString& extension )
{
	m_logger.log( "TTMD request received from Amarok" );

	KURL srcURL( url );
	QString profile = "Amarok TTMD - " + extension.lower();

	if ( ! m_config.profileIsValid( profile ) )
	{
		amarokError( 0, "invalid profile requested", srcURL.url() );
		return;
	}

	if ( ! m_config.profileIsValid( profile ) || srcURL.protocol() != "file" ) // can't handle anything else than local files ATM
	{
		amarokError( 0, "invalid input protocol", srcURL.url() );
		return;
	}

	m_logger.log( "profile: " + profile );
	m_logger.log( "file: " + srcURL.path() );

	FileJob* job = new FileJob( m_config, srcURL.path() );
	job->setProfile( profile );
	m_jobs.append( job );

	connect( job, SIGNAL( mtFinishedOK(long,const QString&,const QString&) ),
		this, SLOT( amarokSuccess(long,const QString&,const QString&) ) );

	connect( job, SIGNAL( mtFinishedWithError(long,const QString&,const QString&) ),
		this, SLOT( amarokError(long,const QString&,const QString&) ) );

	m_amarokQueue.enqueue( job );
	m_logger.log( "job enqueued in Amarok's TTMD worker thread" );
}

void Application::addTranscodeJobs( const QStringList& urls, const QString& profile )
{
	m_logger.log( "transcode request received from Amarok" );

	EncodingProfile* options = m_config.profileOptions( profile );

	if ( ! m_config.profileOptionsAreValid( options ) )
	{
		m_logger.log( "invalid profile requested, message discarted" );
		if ( options != 0 )
			delete options;
		return;
	}

	if ( m_config.generalOption( GeneralOpt::confirm ) == "true" )
		if ( KMessageBox::No == KMessageBox::questionYesNo(
			0,
			i18n(	"<qt>Process selection with chosen profile?<br/><br/>"
					"<b>Profile</b>: %1<br/>"
					"<b>Encode Type</b>: %2<br/>"
					"<b>Switches</b>: %3</qt>" )
						.arg( profile )
						.arg( options->encodeType() )
						.arg( options->switches() ),
			"transKode" ) )
		{
			m_logger.log( "operation cancelled by user" );
			delete options;
			return;
		}

	delete options;

	m_logger.log( "profile: " + profile );

	for ( QStringList::ConstIterator it = urls.begin(); it != urls.end(); ++it )
	{
		KURL srcURL( *it );
		if ( srcURL.protocol() == "file" )
		{
			m_logger.log( "file: " + srcURL.path() );

			FileJob* job = new FileJob( m_config, srcURL.path() );
			job->setProfile( profile );
			m_jobs.append( job );

			connect( job, SIGNAL( mtStarted(long,const QString&,const QString&) ),
				this, SLOT( displayJobStarted(long,const QString&,const QString&) ) ) ;

			// NOTE: don't connect to deleteJob directly if something else needs
			// to get done because we can't tell wich signal will be called first
			connect( job, SIGNAL( mtFinishedOK(long,const QString&) ),
				this, SLOT( deleteJob(long) ) );
			connect( job, SIGNAL( mtFinishedWithError(long,const QString&,const QString&) ),
				this, SLOT( displayJobError(long,const QString&,const QString&) ) );

			m_commonQueue.enqueue( job );
			m_logger.log( "job enqueued in main worker thread" );
		}
		else
		{
			//TODO: add job to stream worker (not implemented yet)
		}
	}
}

/*void Application::displayStreamRipStarted( int, int, const QString& URL )
{
	if ( m_config.generalOption( GeneralOpt::notify ) != "true" )
		return;

	QString showMsg = i18n( "<qt><font size=\"+1\"><b>transKode: ripping started</b></font><br/><br/>"
							"<b>URL</b>: %1<br/>"
							"<b>Duration</b>: %2 seconds</qt>" )
						.arg( URL )
						.arg( m_config.generalOption( GeneralOpt::rip_duration ) );

	displayPopup( showMsg );
}*/

/*void Application::displayStreamRipFinished( int, int, const QString& URL, const QStringList& rippedFiles )
{
	if ( m_config.generalOption( GeneralOpt::notify ) != "true" )
		return;

	QString showMsg = i18n(	"<font size=\"+1\"><b>transKode: done ripping %1</b></font><br/><br/>"
							"<b>Ripped files:</b>" )
						.arg( URL );

	for ( QStringList::ConstIterator it = rippedFiles.begin() ; it != rippedFiles.end(); ++it )
		showMsg += QString( "<br/>" ) + *it;

	displayPopup( "<qt>" + showMsg + "</qt>");
}*/

void Application::displayJobStarted( long, const QString& dstPath, const QString& srcPath )
{
	if ( m_config.generalOption( GeneralOpt::notify ) != "true" )
		return;

	QString showMsg = i18n( "<qt><font size=\"+1\"><b>transKode: job started</b></font><br/><br/>"
							"<b>Source</b>: %1<br/>"
							"<b>Target</b>: %2</qt>" )
						.arg( srcPath )
						.arg( dstPath );

	if ( ! showMsg.isEmpty() )
		displayPopup( showMsg );
}

void Application::displayJobError( long jobID, const QString& errorMsg, const QString& srcPath )
{
	if ( m_config.generalOption( GeneralOpt::notify ) != "true" )
		return;

	displayPopup( i18n( "<qt><b><font size=\"+1\" color=\"#ff0000\">transKode: %1</font><br/><br/>"
						"Source</b>: %2</qt>" )
					.arg( errorMsg )
					.arg( srcPath )
	);

	deleteJob( jobID );
}

void Application::displayRoundStats( long, int okCount, int errorCount, int cancelledCount )
{
	if ( m_config.generalOption( GeneralOpt::notify ) != "true" )
		return;

	QString showMsg( "" );

	if ( okCount + errorCount )
		showMsg += "<br/>" + i18n( "%1/%2 files succesfully encoded" )
			.arg( okCount )
			.arg( okCount + errorCount );

	if ( cancelledCount ) // NOTE: is not really possible to cancel jobs from Amarok ATM
		showMsg += "<br/>" + i18n( "1 job cancelled by user", "%n jobs cancelled by user", cancelledCount );

	if ( ! showMsg.isEmpty() )
		displayPopup( i18n( "<qt><font size=\"+1\"><b>transKode: all done</b></font><br/>%1</qt>" ).arg( showMsg ) );
}

void Application::displayPopup( const QString& text )
{
	DCOPRef amarokmedia( "amarok", "playlist" );
	amarokmedia.send( "popupMessage", text );

// 	KPassivePopup::message( "", text, QPixmap(), (QWidget*)0, 0, msec );
}

void Application::amarokSuccess( long jobID, const QString& dstPath, const QString& srcPath )
{
	m_logger.log( "Amarok's TTMD job completed succesfully" );

	m_logger.log( "sending success notification to Amarok's TTMD" );
	//m_logger.log( "transcodingFinished '" + KURL::fromPathOrURL( src ).url() + "' '" + KURL::fromPathOrURL( dst ).url() + "'");
	DCOPRef amarokmedia( "amarok", "mediabrowser" );
	amarokmedia.send( "transcodingFinished", KURL::fromPathOrURL( srcPath ).url(), KURL::fromPathOrURL( dstPath ).url() );

	deleteJob( jobID );
}

void Application::amarokError( long jobID, const QString&, const QString& srcPath )
{
	m_logger.log( "Amarok's TTMD job completed with error" );

	m_logger.log( "sending error notification to Amarok's TTMD" );
	//m_logger.log( "transcodingFinished '" + KURL::fromPathOrURL( src ).url() + "' ''");
	DCOPRef amarokmedia( "amarok", "mediabrowser" );
	amarokmedia.send( "transcodingFinished", KURL::fromPathOrURL( srcPath ).url(), QString() );

	deleteJob( jobID );
}

void Application::amarokDone()
{
	m_logger.log( "Amarok's TTMD worker thread done processing" );
}

void Application::deleteJob( long jobID )
{
	for ( FileJob* job = m_jobs.first(); job != 0; job = m_jobs.next() )
	{
		if ( job->id() == jobID )
		{
			m_jobs.remove();
			break;
		}
	}
}

