/***************************************************************************
 *   Copyright (C) 2005 by Sergio Pistone                                  *
 *   sergio_pistone@yahoo.com.ar                                           *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "commondefs.h"

#include <qthread.h>
#include <qmutex.h>
#include <qstring.h>
#include <qstringlist.h>
#include <qdir.h>
#include <qfileinfo.h>
#include <qdeepcopy.h>
#include <qdatastream.h>
#include <qtextcodec.h>
#include <qregexp.h>

#include <kapplication.h>
#include <kstandarddirs.h>

#ifndef Q_WS_WIN
	#include <unistd.h>
#endif

/**
 * GeneralOpt
*/

QString GeneralOpt::toQString( GeneralOpt::TGeneralOpt option )
{
	switch ( option )
	{
		case GeneralOpt::naming_scheme:		return "naming_scheme";
		case GeneralOpt::notify:			return "notify";
		case GeneralOpt::confirm:			return "confirm";
		case GeneralOpt::overwrite:			return "overwrite";
		case GeneralOpt::case_insensitive:	return "case_insensitive";
		case GeneralOpt::replace_chars:		return "replace_chars";
		case GeneralOpt::proc_priority:		return "proc_priority";
		case GeneralOpt::threads:			return "threads";
		case GeneralOpt::rt_progress:		return "rt_progress";
		case GeneralOpt::show_tray:			return "show_tray";
		case GeneralOpt::log:				return "log";
		case GeneralOpt::temp_dir:			return "temp_dir";
		default:							return "Unknown";
	}
}

GeneralOpt::TGeneralOpt GeneralOpt::fromQString( const QString& option )
{
	if		( option == "naming_scheme" )		return GeneralOpt::naming_scheme;
	else if	( option == "notify" )				return GeneralOpt::notify;
	else if	( option == "confirm" )				return GeneralOpt::confirm;
	else if	( option == "overwrite" )			return GeneralOpt::overwrite;
	else if	( option == "case_insensitive" ) 	return GeneralOpt::case_insensitive;
	else if	( option == "replace_chars" )		return GeneralOpt::replace_chars;
	else if	( option == "proc_priority" )		return GeneralOpt::proc_priority;
	else if	( option == "threads" )				return GeneralOpt::threads;
	else if ( option == "rt_progress" )			return GeneralOpt::rt_progress;
	else if	( option == "show_tray" )			return GeneralOpt::show_tray;
	else if	( option == "log" )					return GeneralOpt::log;
	else if	( option == "temp_dir" )			return GeneralOpt::temp_dir;
	else										return GeneralOpt::UNKNOWN;
}


/**
 * ProfileOpt
*/

QString ProfileOpt::toQString( ProfileOpt::TProfileOpt option )
{
	switch ( option )
	{
		case ProfileOpt::naming_scheme:	return "naming_scheme";
		case ProfileOpt::encode_type:	return "encode_type";
		case ProfileOpt::extension:		return "extension";
		case ProfileOpt::switches:		return "switches";
		case ProfileOpt::show:			return "show";
		case ProfileOpt::bypass:		return "bypass";
		default:						return "Unknown";
	}
}

ProfileOpt::TProfileOpt ProfileOpt::fromQString( const QString& option )
{
	if 		( option == "naming_scheme" )	return ProfileOpt::naming_scheme;
	else if	( option == "encode_type" )		return ProfileOpt::encode_type;
	else if	( option == "extension" )		return ProfileOpt::extension;
	else if	( option == "switches" )		return ProfileOpt::switches;
	else if	( option == "show" )			return ProfileOpt::show;
	else if ( option == "bypass" )			return ProfileOpt::bypass;
	else									return ProfileOpt::UNKNOWN;
}


/**
 * TextTrans
*/

QString TextTrans::toQString( TextTrans::TTextTrans trans )
{
	switch ( trans )
	{
		case TextTrans::leaveasis:		return "";
		case TextTrans::titlecase:		return "titlecase";
		case TextTrans::capitalize:		return "capitalize";
		case TextTrans::lowercase:		return "lowercase";
		case TextTrans::uppercase:		return "uppercase";
		case TextTrans::ascii:			return "ascii";
		default:						return "Unknown";
	}
}

TextTrans::TTextTrans TextTrans::fromQString( const QString& trans )
{
	if		( trans == "" )				return TextTrans::leaveasis;
	else if ( trans == "titlecase" )	return TextTrans::titlecase;
	else if	( trans == "capitalize" )	return TextTrans::capitalize;
	else if	( trans == "lowercase" )	return TextTrans::lowercase;
	else if	( trans == "uppercase" )	return TextTrans::uppercase;
	else if	( trans == "ascii" )		return TextTrans::ascii;
	else								return TextTrans::UNKNOWN;
}


/**
 * String
 */

QString String::title( const QString& text )
{
	QString auxText = text.lower();
	const QString separators = " -_([:,;./\\\t\n\"";
	int len = strlen( separators ), sepIdx = 0;
	while ( sepIdx < len )
	{
		QStringList auxWordsList;
		QStringList wordsList = QStringList::split( separators[sepIdx], auxText, true );
		for ( QStringList::Iterator it = wordsList.begin(); it != wordsList.end(); ++it )
			if ( ! (*it).isEmpty() )
				auxWordsList.append( (*it)[0].upper()+(*it).mid(1) );
			else
				auxWordsList.append( "" );
		auxText = auxWordsList.join( separators[sepIdx] );
		sepIdx++;
	}
	return auxText;
}

QString String::capitalize( const QString& text )
{
	return text.isEmpty() ? "" : text[0].upper() + text.mid(1).lower();
}

QString String::ascii( const QString& text )
{
	static const unsigned SIZE = 48;

	static const QString charsToSearch[SIZE] = {
		QString::fromUtf8( "āăǻаàáâãäåą" ), QString::fromUtf8( "ΆÀÁÂÃÄÅĀĂĄǺΑА" ),
		QString::fromUtf8( "ЪЬъь" ), QString::fromUtf8( "ΒВв" ),
		QString::fromUtf8( "сçćĉċč" ), QString::fromUtf8( "ÇĆĈĊČС" ),
		QString::fromUtf8( "đď" ), QString::fromUtf8( "ÐĎĐ" ),
		QString::fromUtf8( "ȩеѐёєèéêëēĕėęě" ), QString::fromUtf8( "ÈÉÊËĒĔĖĘĚΈȨΕЀЁЕ" ),
		QString::fromUtf8( "ģĝğġ" ), QString::fromUtf8( "ĜĞĠĢ" ),
		QString::fromUtf8( "нңҺĥħһћђ" ), QString::fromUtf8( "ĤΉΗНҢ" ),
		QString::fromUtf8( "ĩīĭįıϊίіїΐìíîï" ), QString::fromUtf8( "ÌÍÎÏĨĪĬĮİΊІЇΙΪ" ),
		QString::fromUtf8( "ĵј" ), QString::fromUtf8( "ĴЈ" ),
		QString::fromUtf8( "ķĸқκкќ" ), QString::fromUtf8( "ĶΚҚЌК" ),
		QString::fromUtf8( "łĺļľŀ" ), QString::fromUtf8( "ĹĻĽĿŁ" ),
		QString::fromUtf8( "м" ), QString::fromUtf8( "ΜМ" ),
		QString::fromUtf8( "ñńņňŉ" ), QString::fromUtf8( "ŃŅŇÑΝ" ),
		QString::fromUtf8( "ōŏőơόоòóôõö" ), QString::fromUtf8( "ÒÓÔÕÖŌŎŐƠΌОΟ" ),
		QString::fromUtf8( "р" ), QString::fromUtf8( "ΡР" ),
		QString::fromUtf8( "яŕŗřѓ" ), QString::fromUtf8( "ŔŖŘЯ" ),
		QString::fromUtf8( "śŝşšѕș" ), QString::fromUtf8( "ŚŜŞŠȘЅ" ),
		QString::fromUtf8( "тţťț" ), QString::fromUtf8( "ŢŤȚΤТ" ),
		QString::fromUtf8( "ũūŭůűùúûüư" ), QString::fromUtf8( "ŨŪŬŮŰŲÙÚÛÜƯ" ),
		QString::fromUtf8( "ΰυϋύ" ),
		QString::fromUtf8( "ŵωώ" ), QString::fromUtf8( "Ŵ" ),
		QString::fromUtf8( "хҳχ" ), QString::fromUtf8( "ҲΧХ" ),
	 	QString::fromUtf8( "ŷýÿүγуў" ), QString::fromUtf8( "ÝŶŸΎҮΥΫЎУ" ),
		QString::fromUtf8( "źżž" ), QString::fromUtf8( "ŹŻŽΖ" ),
		QString::fromUtf8( " ¡¢£¤¥¦§¨©ª«¬­®¯°±²³´µ¶·¸¹º»¼½¾¿Æ×ØÞßæð÷øþ" ) // latin1, non ASCII
	};

	static const QString charsToReplace[SIZE] = {
		QString( "a" ), QString( "A" ),
		QString( "b" ), QString( "B" ),
		QString( "c" ), QString( "C" ),
		QString( "d" ), QString( "D" ),
		QString( "e" ), QString( "E" ),
		QString( "g" ), QString( "G" ),
		QString( "h" ), QString( "H" ),
		QString( "i" ), QString( "I" ),
		QString( "j" ), QString( "J" ),
		QString( "k" ), QString( "K" ),
		QString( "l" ), QString( "L" ),
		QString( "m" ), QString( "M" ),
		QString( "n" ), QString( "N" ),
		QString( "o" ), QString( "O" ),
		QString( "p" ), QString( "P" ),
		QString( "r" ), QString( "R" ),
		QString( "s" ), QString( "S" ),
		QString( "t" ), QString( "T" ),
		QString( "u" ), QString( "U" ),
		QString( "v" ),
		QString( "w" ), QString( "W" ),
		QString( "x" ), QString( "X" ),
	 	QString( "y" ), QString( "T" ),
		QString( "z" ), QString( "Z" ),
		QString( "_" )
	};

	static const QString filler( "_" );

	QString ret( text );

	for ( unsigned idx = 0; idx < SIZE; ++idx )
	{
		const QString& toSearch = charsToSearch[idx];
		const QString& toReplace = charsToReplace[idx];

		for ( unsigned idx2 = 0; idx2 < text.length(); ++idx2 )
			if ( toSearch.contains( ret.at( idx2 ) ) )
				ret.replace( idx2, 1, toReplace );
	}

	QTextCodec* codec = QTextCodec::codecForName( "ISO-8859-1" );
	if ( codec == 0 )
		kdDebug() << "no text codec found for ISO-8859-1" << endl;
	else
		for ( unsigned idx = 0; idx < text.length(); ++idx )
		{
			if ( ! codec->canEncode( ret.at( idx ) ) )
				ret.replace( idx, 1, filler );
		}

	return ret;
}

int String::rfindFunctionStart( const QString& text )
{
	int pos[5] = {
		text.findRev( "capitalize{" ),
		text.findRev( "titlecase{" ),
		text.findRev( "uppercase{" ),
		text.findRev( "lowercase{" ),
		text.findRev( "ascii{" ),
	};

	return QMAX( QMAX( QMAX( QMAX( pos[0], pos[1] ), pos[2] ), pos[3] ), pos[4] );
}

int String::rfindFunctionEnd( const QString& text, int startPos )
{
	int pos = startPos, len = text.length();
	while ( pos < len )
	{
		// next is safe because idx will always be greater than 0
		if ( text[pos] == '}' && text[pos-1] != '\\' )
			return pos;
		pos++;
	}
	return -1;
}

bool String::isValidTextTransformationExpression( const QString& expression )
{
	if ( expression.isEmpty() )
		return false;

    QString expr( expression );

	expr.replace( "\\{", "" );
	expr.replace( "\\}", "" );

	int startPos, endPos;
	while ( (startPos = String::rfindFunctionStart( expr )) != -1 )
	{
		endPos = String::rfindFunctionEnd( expr, startPos );
		if ( endPos == -1 )
			return false;
		QString funcParam = expr.mid( startPos, endPos - startPos );
		if ( funcParam.contains( '{' ) != 1 || funcParam.contains( '}' ) )
			return false;
		expr = expr.left( startPos ) + expr.mid( endPos + 1 );
	}

	return ( expr.contains( '{' ) || expr.contains( '}' ) ) ? false : true;
}

QString String::evalTextTransformationExpression( const QString& expression )
{
	int startPos, endPos;
	QString expr = expression;

	while ( (startPos = rfindFunctionStart( expr )) != -1 )
	{
		endPos = rfindFunctionEnd( expr, startPos );
		QString transformed = "";
		if ( expr[startPos] == 't' )
			transformed = title( expr.mid( startPos + strlen( "titlecase{" ), endPos - startPos - strlen( "titlecase{" ) ) );
		else if ( expr[startPos] == 'c' )
			transformed = capitalize( expr.mid( startPos + strlen( "capitalize{" ), endPos - startPos - strlen( "capitalize{" ) ) );
		else if ( expr[startPos] == 'l' )
			transformed = expr.mid( startPos + strlen( "lowercase{" ), endPos - startPos - strlen( "lowercase{" ) ).lower();
		else if ( expr[startPos] == 'u' )
			transformed = expr.mid( startPos + strlen( "uppercase{" ), endPos - startPos - strlen( "uppercase{" ) ).upper();
		else if ( expr[startPos] == 'a' )
			transformed = ascii( expr.mid( startPos + strlen( "ascii{" ), endPos - startPos - strlen( "ascii{" ) ) );

		expr = expr.left( startPos ) + transformed + expr.mid( endPos + 1 );
	}

	return expr;
}

QString String::evalTextTransformation( const QString& text, const QString& trans )
{
	return trans.isEmpty() ? text : evalTextTransformationExpression( trans + "{" + text + "}" );
}

QString String::evalTextTransformation( const QString& text, TextTrans::TTextTrans trans )
{
	return trans == TextTrans::leaveasis ?
		text : evalTextTransformationExpression( TextTrans::toQString( trans ) + "{" + text + "}" );
}

QString String::quoteForShell( const QString& text )
{
	return "\"" + QString( text ).replace( "\"","\\\"" ).replace( "`","\\`" ) + "\"";
}

QString String::padNumber( const QString& string, unsigned width )
{
	if ( string.length() >= width )
		return string;

	QString paddedNumber;
	bool ok;
	int number = string.toUInt( &ok, 10 );
	if ( ok )
		return paddedNumber.sprintf( "%0" + QString::number( width ) + "d", number );
	else
		return string;
}

QString String::deepCopy( const QString& string )
{
	return QDeepCopy<QString>( string );
}

QStringList String::deepCopy( const QStringList& stringList )
{
	return QDeepCopy<QStringList>( stringList );
}


/**
 * System:
 */

QStringList System::s_potentialFiles;
QMutex System::s_potentialFilesMutex;

void System::addPotentialFile( const QString& path )
{
	if ( ! path.isEmpty() )
	{
		s_potentialFilesMutex.lock();

		s_potentialFiles.append( path );

		s_potentialFilesMutex.unlock();
	}
}

void System::delPotentialFile( const QString& path )
{
	if ( ! path.isEmpty() )
	{
		s_potentialFilesMutex.lock();

		s_potentialFiles.remove( path );

		s_potentialFilesMutex.unlock();
	}
}

void System::clearPotentialFiles()
{
	s_potentialFilesMutex.lock();

	s_potentialFiles.clear();

	s_potentialFilesMutex.unlock();
}

QString System::isPotentialFile( const QString& path, bool vfat )
{
	s_potentialFilesMutex.lock();

	QString ret;
	if ( vfat )
	{
		ret = "";
		for ( unsigned idx = 0; idx < s_potentialFiles.size(); ++idx )
			if ( path.lower() == s_potentialFiles[idx].lower() )
			{
				ret = s_potentialFiles[idx];
				break;
			}
	}
	else
		ret = s_potentialFiles.contains( path ) ? path : "";

	s_potentialFilesMutex.unlock();

	return ret;
}

// If necesary, changes the part of a given path to match the case of the existing one
QString System::correctVFATPath( const QString& path )
{
	bool tokenFound = true;
	QDir parcialPath( "/" );
	QStringList pathTokens = QStringList::split( "/", path );
	for ( QStringList::Iterator tokenIt = pathTokens.begin(); tokenIt != pathTokens.end(); ++tokenIt )
	{
		if ( tokenFound )
		{
			if ( ! QFileInfo( parcialPath.path() + "/" + *tokenIt ).exists() )
			{
// 				kdDebug() << "correctVFATPath: '" << *tokenIt << "' not found in '" << parcialPath.path() + "'" << endl;
// 				kdDebug() << "correctVFATPath: performing case insensitive search" << endl;

				QString lowerCaseToken = (*tokenIt).lower();
				QStringList dirContents = parcialPath.entryList( "*", QDir::All|QDir::System );
				tokenFound = false;
				for ( QStringList::Iterator dirContentsIt = dirContents.begin(); dirContentsIt != dirContents.end(); ++dirContentsIt )
				{
					if ( (*dirContentsIt).lower() == lowerCaseToken )
					{
// 						kdDebug() << "correctVFATPath: found '" << *dirContentsIt << "'" << endl;
						*tokenIt = *dirContentsIt;
						tokenFound = true;
						break;
					}
				}
			}
		}
		parcialPath.setPath( parcialPath.path() + "/" + *tokenIt );
	}

	return parcialPath.path();
}

// Checks if the given path exists; returns the existing path if it does or else returns "".
// If vfat is true, path will be searched in a case insensitive way but the returned value
// will have the actual case of the existing path.
QString System::conflictivePath( const QString& path, bool vfat, bool potentialFiles, const QString& exception )
{
	QString collidingPath = "";

	if ( vfat ) // case insensitive test
	{
		// check for collitions with potential files:
		if ( potentialFiles )
		{
			collidingPath = isPotentialFile( path, true );
			if ( ! collidingPath.isEmpty() ) // there was a potential file with 'path'
				return collidingPath.lower() == exception.lower() ? "" /*exception*/ : collidingPath /*no exception*/;
		}

		// check for collitions with real files:
		collidingPath = QFileInfo( collidingPath = correctVFATPath( path ) ).exists() ? collidingPath : "";
		if ( ! collidingPath.isEmpty() ) // there was a real file with 'path'
			return collidingPath;
	}
	else // case sensitive test
	{
		// check for collitions with potential files:
		if ( potentialFiles )
		{
			collidingPath = isPotentialFile( path, false );
			if ( ! collidingPath.isEmpty() ) // there was a potential file with 'path'
				return collidingPath == exception ? "" /*exception*/ : collidingPath /*no exception*/;
		}

		// check for collitions with real files:
		collidingPath = QFileInfo( path ).exists() ? path : "";
		if ( ! collidingPath.isEmpty() ) // there was a real file with 'path'
			return collidingPath;
	}

	return "";
}

// If the path exists, appends a suffix before the extension wich is incremented
// until the resulting path doesn't exists. If onlyDir is true the resulting
// path will not conflict with a dir but may conflict with a file.
QString System::nonConflictivePath( const QString& path, bool onlyDir, bool vfat, bool potentialFiles, const QString& exception )
{
	if ( onlyDir ) // waste of time
		potentialFiles = false;

	QFileInfo fileInfo( vfat ? correctVFATPath( path ) : path );
	QString extension = fileInfo.extension( false );
	if ( ! extension.isEmpty() )
		extension = "." + extension;
	QString dirAndName = fileInfo.filePath();
	dirAndName.truncate( dirAndName.length() - extension.length() );
	if ( dirAndName.right( 1 ) == "." )
		dirAndName.truncate( dirAndName.length() - 1 );

	unsigned suffix = 2;

	while ( ! conflictivePath( fileInfo.filePath(), vfat, potentialFiles, exception ).isEmpty() &&
			( ! onlyDir || fileInfo.isDir() || fileInfo.isSymLink() ) )
		fileInfo.setFile( dirAndName + "_" + QString::number( suffix++ ) + extension );

	return fileInfo.filePath();
}

bool System::recursiveMakeDir( const QString& path, QStringList* createdDirsList, bool vfat )
{
	if ( createdDirsList )
		createdDirsList->clear();

	QDir parcialPath( "/" );
	QStringList tokens = QStringList::split( "/", vfat ? correctVFATPath( path ) : path );
	for ( QStringList::Iterator it = tokens.begin(); it != tokens.end(); ++it )
	{
		parcialPath.setPath( parcialPath.path() + "/" + *it );
		if ( ! QFileInfo( parcialPath.path() ).exists() )
		{
			if ( ! QDir().mkdir( parcialPath.path() ) )
				return false;
			if ( createdDirsList )
				createdDirsList->prepend( parcialPath.path() );
		}
	}

	return true;
}

bool System::copy( const QString& srcPath, const QString& dstPath, bool srcVFAT, bool dstVFAT )
{
	QFile srcFile( srcVFAT ? correctVFATPath( srcPath ) : srcPath );
	if ( ! srcFile.open( IO_ReadOnly ) )
	{
		kdDebug() << "copy: couldn't open input file " << srcFile.name() << endl;
		return false;
	}

	QFile dstFile( dstVFAT ? correctVFATPath( dstPath ) : dstPath );
	if ( ! dstFile.open( IO_WriteOnly ) )
	{
		kdDebug() << "copy: couldn't open output file " << dstFile.name() << endl;
		return false;
	}

	char* data = new char[1000000];
	int read = srcFile.readBlock( data, 1000000 );
	int write = 0;
	while ( read != -1 && read != 0 && write != -1 )
	{
		write = dstFile.writeBlock( data, read );
		read = srcFile.readBlock( data, 1000000 );
	}

	delete [] data;

	srcFile.close();
	dstFile.close();

	if ( read == -1 )
	{
		kdDebug() << "copy: error reading data from input file " << srcFile.name() << endl;
		return false;
	}
	else if ( write == -1 )
	{
		kdDebug() << "copy: error writing data to output file " << dstFile.name() << endl;
		return false;
	}
	else
		return true;
}

bool System::move( const QString& srcPath, const QString& dstPath, bool srcVFAT, bool dstVFAT )
{
	QString src = srcVFAT ? correctVFATPath( srcPath ) : srcPath;
	QString dst = dstVFAT ? correctVFATPath( dstPath ) : dstPath;

	if ( ! System::copy( src, dst, false, false ) )
	{
		kdDebug() << "move: error copying " << src << " to " << dst << endl;

		return false;
	}

	if ( ! QFile::remove( src ) )
	{
		kdDebug() << "move: error removing " << src << endl;

		return false;
	}

	return true;
}

bool System::remove( const QString& path, bool vfat )
{
	return QFile::remove( vfat ? correctVFATPath( path ) : path );
}

bool System::isReadable( const QString& path )
{
	QDir absPath( path );
	absPath.convertToAbs();
	QFileInfo fileInfo( absPath.path() );
	return fileInfo.isFile() && fileInfo.isReadable();
}

bool System::isWritable( const QString& path )
{
	QDir absPath( path );
	absPath.convertToAbs();
	QFileInfo fileInfo( absPath.path() );
	return fileInfo.isFile() && fileInfo.isWritable();
}

QString System::user()
{
#ifdef Q_WS_WIN
	// TODO Windows code not tested!
	// Taken from http://lists.trolltech.com/qt-interest/2002-12/thread00130-0.html
	#ifdef UNICODE
	if ( qWinVersion() & Qt::WV_NT_based )
	{
		TCHAR winUserName[UNLEN + 1]; // UNLEN is defined in LMCONS.H
		DWORD winUserNameSize = sizeof( winUserName );
		GetUserName( winUserName, &winUserNameSize );
		return qt_winQString( winUserName );
	}
	else
	#endif
	{
		char winUserName[UNLEN + 1]; // UNLEN is defined in LMCONS.H
		DWORD winUserNameSize = sizeof( winUserName );
		GetUserNameA( winUserName, &winUserNameSize );
		return QString::fromLocal8Bit( winUserName );
	}
#else
	char* userName = getlogin();
	if ( userName )
		return QString( userName );
	else
		if ( (userName = getenv( "USER" )) )
			return QString( userName );
		else
			return QString::null;
#endif
}

QString System::homeDir()
{
	return QDir::homeDirPath();
}

QString System::tempDir()
{
	QString tmpDir = kapp->dirs()->saveLocation( "tmp" );
	tmpDir.replace( QRegExp( "/$" ), "" );
	return tmpDir;
}
