/***************************************************************************
 *   Copyright (C) 2005 by Sergio Pistone                                  *
 *   sergio_pistone@yahoo.com.ar                                           *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "appconfig.h"
#include <pluginsmanager.h>

#include <qfileinfo.h>
#include <qdir.h>
#include <qstring.h>
#include <qstringlist.h>
#include <qregexp.h>

#include <ksimpleconfig.h>

using namespace TransKode;

AppConfig::AppConfig( const QString& defaultFile ):
	QObject(),
	m_configMutex( true ), // recursive mutex
	m_inited( false )
{
	m_defaultFile = String::deepCopy( defaultFile );

	reloadEncodeTypes();

	loadDefaults();

	m_inited = true;
}

AppConfig::AppConfig( const AppConfig& cfg ):
	QObject(),
	m_configMutex( true ), // recursive mutex
	m_inited( true )
{
	cfg.m_configMutex.lock();

	m_defaultFile = String::deepCopy( cfg.m_defaultFile );

	for ( int idx = 0; idx < GeneralOpt::SIZE; ++idx )
		m_generalOptions[idx] = String::deepCopy( cfg.m_generalOptions[idx] );

	for ( int idx = 0; idx < Metadata::SIZE; ++idx )
		m_tagDefaults[idx] = String::deepCopy( cfg.m_tagDefaults[idx] );

	for ( int idx = 0; idx < Metadata::SIZE; ++idx )
		m_tagTrans[idx] = cfg.m_tagTrans[idx];

	for ( QMap<QString,bool>::ConstIterator it = cfg.m_suppEncodeTypes.begin(); it != cfg.m_suppEncodeTypes.end(); ++it )
		m_suppEncodeTypes[String::deepCopy(it.key())] = it.data();

	for ( QMap<QString,EncodingProfile>::ConstIterator it = cfg.m_profiles.begin(); it != cfg.m_profiles.end(); ++it )
	{
		m_profiles[String::deepCopy( it.key() )] = it.data();
		m_profiles[it.key()].setDefaultNamingScheme( cfg.m_generalOptions[GeneralOpt::naming_scheme] );
	}

	cfg.m_configMutex.unlock();
}

AppConfig& AppConfig::operator=( const AppConfig& c )
{
	if ( this == &c )
		return *this;

	// This seems inefficient, but is the best way (I could think of) not to risk deadlock
	AppConfig cfg( c );

	m_configMutex.lock();

	bool emitChanged = false;
	bool emitChangedDefault = false;
	QString prevDefaultFile;
	QValueList<int> changedOptions;
	QValueList<int> changedDefaults;
	QValueList<int> changedTagTrans;
	QStringList removedSuppEncodeType;
	QStringList addedSuppEncodeType;
	QStringList removedProf;
	QStringList addedProf;
	QStringList changedProf;

	setDefaultFile( cfg.m_defaultFile, emitChanged, prevDefaultFile );

	for ( QMap<QString,EncodingProfile>::ConstIterator it = cfg.m_profiles.begin(); it != cfg.m_profiles.end(); ++it )
	{
		if ( ! m_profiles.contains( it.key() ) ) // profile was added
		{
			m_profiles[it.key()] = it.data();
			addedProf.append( it.key() );
			emitChanged = true;
		}
		else if ( m_profiles[it.key()] != it.data() ) // profile was changed
		{
			m_profiles[it.key()] = it.data();
			changedProf.append( it.key() );
			emitChanged = true;
		}
		m_profiles[it.key()].setDefaultNamingScheme( m_generalOptions[GeneralOpt::naming_scheme] );
		m_profiles[it.key()].setNamingScheme( it.data().displayNamingScheme() );
	}

	for ( QMap<QString,EncodingProfile>::Iterator it = m_profiles.begin(); it != m_profiles.end(); ++it )
		if ( ! cfg.m_profiles.contains( it.key() ) ) // profile was removed
			removedProf.append( it.key() );
	for ( QStringList::Iterator it = removedProf.begin(); it != removedProf.end(); ++it )
		m_profiles.remove( *it );
	if ( removedProf.size() )
		emitChanged = true;


	for ( int idx = 0; idx < GeneralOpt::SIZE; ++idx )
		setGeneralOption( (GeneralOpt::TGeneralOpt)idx, cfg.m_generalOptions[idx], emitChanged, changedOptions, changedProf );

	for ( QMap<QString,bool>::ConstIterator it = cfg.m_suppEncodeTypes.begin(); it != cfg.m_suppEncodeTypes.end(); ++it )
	{
		if ( m_suppEncodeTypes[it.key()] != it.data() )
		{
			if ( it.data() )
				addedSuppEncodeType.append( it.key() );
			else
				removedSuppEncodeType.append( it.key() );
			m_suppEncodeTypes[it.key()] = it.data();
		}
	}


	for ( int idx = 0; idx < Metadata::SIZE; ++idx )
		setTagDefault( (Metadata::Tag)idx, cfg.m_tagDefaults[idx], emitChanged, changedDefaults );

	for ( int idx = 0; idx < Metadata::SIZE; ++idx )
		setTagTransformation( (Metadata::Tag)idx, cfg.m_tagTrans[idx], emitChanged, changedTagTrans );

	m_configMutex.unlock();

	if ( emitChanged )						emit changed();
	if ( emitChangedDefault )				emit changedDefaultFile( prevDefaultFile );
	if ( changedOptions.size() )			emit changedGeneralOpts( changedOptions );
	if ( changedDefaults.size() )			emit changedTagDefaults( changedDefaults );
	if ( changedTagTrans.size() )			emit changedTagTransformations( changedTagTrans );
	if ( addedSuppEncodeType.size() )		emit addedSuppEncodeTypes( addedSuppEncodeType );
	if ( removedSuppEncodeType.size() )		emit removedSuppEncodeTypes( removedSuppEncodeType );
	if ( addedProf.size() )					emit addedProfiles( addedProf );
	if ( removedProf.size() )				emit removedProfiles( removedProf );
	if ( changedProf.size() )				emit changedProfiles( changedProf );

	return *this;
}


QString AppConfig::generalOption( GeneralOpt::TGeneralOpt option ) const
{
	if ( option >= GeneralOpt::SIZE )
		return QString( "" );

	m_configMutex.lock();

	QString ret = String::deepCopy( m_generalOptions[option] );

	m_configMutex.unlock();

	return ret;
}

QString AppConfig::programPath( const QString& program, bool quoted ) const
{
	QString path = PluginsManager::programPath( program );

	return quoted ? String::quoteForShell( path ) : path;
}

TextTrans::TTextTrans AppConfig::tagTransformation( Metadata::Tag tag ) const
{
	if ( tag >= Metadata::SIZE )
		return TextTrans::leaveasis;

	m_configMutex.lock();

	TextTrans::TTextTrans ret = m_tagTrans[tag];

	m_configMutex.unlock();

	return ret;
}

QString AppConfig::tagDefault( Metadata::Tag tag ) const
{
	if ( tag >= Metadata::SIZE )
		return QString();

	m_configMutex.lock();

	QString ret = String::deepCopy( m_tagDefaults[tag] );

	m_configMutex.unlock();

	return ret;
}

QStringList AppConfig::profiles() const
{
	m_configMutex.lock();

	QStringList ret;
	for ( QMap<QString,EncodingProfile>::ConstIterator it = m_profiles.begin(); it != m_profiles.end(); ++it )
		ret.append( String::deepCopy( it.key() ) );

	m_configMutex.unlock();

	return ret;
}

bool AppConfig::encodeTypeIsSupported( const QString& encodeType ) const
{
	if ( encodeType == "Bypass" )
		return true;

	m_configMutex.lock();

	bool ret = m_suppEncodeTypes.contains( encodeType ) ? m_suppEncodeTypes[encodeType] : false;

	m_configMutex.unlock();

	return ret;
}

QString AppConfig::supportedEncodingExtensions() const
{
	m_configMutex.lock();

	QStringList extensions;

	for ( QMap<QString,bool>::ConstIterator it = m_suppEncodeTypes.begin(); it != m_suppEncodeTypes.end(); ++it )
	{
		if ( ! it.data() )
			continue;

		const EncoderPlugin* plugin = PluginsManager::encoderPlugin( it.key() );
		if ( plugin == 0 )
			continue;

		QStringList pluginExtensions = plugin->extensions();
		for ( QStringList::Iterator it = pluginExtensions.begin(); it != pluginExtensions.end(); ++it )
			if ( ! extensions.contains( *it) )
				extensions.append( *it );
	}

	m_configMutex.unlock();

	return extensions.join( " " );
}

/*bool AppConfig::programIsSupported( Program::TProgram program ) const
{
	m_configMutex.lock();

	bool ret = KStandardDirs::findExe( m_programPaths[program] ) != QString::null;

	m_configMutex.unlock();

	return ret;
}*/

/*bool AppConfig::baseDirIsValid() const
{
	m_configMutex.lock();

	bool ret;

	QDir absBaseDir( m_generalOptions[GeneralOpt::base_dir] );
	absBaseDir.convertToAbs();
	QFileInfo dirInfo( absBaseDir.path() );
	ret = dirInfo.isDir() && dirInfo.isReadable() && dirInfo.isWritable() && dirInfo.isExecutable();

	m_configMutex.unlock();

	return ret;
}*/

/*bool AppConfig::profileHasValidBaseDir( const QString& profile ) const
{
	m_configMutex.lock();

	bool ret;

	if ( m_profiles.contains( profile ) )
		ret = m_profiles[profile].haveValidBaseDir();
	else
		ret = false;

	m_configMutex.unlock();

	return ret;
}

bool AppConfig::profileHasValidNamingScheme( const QString& profile ) const
{
	m_configMutex.lock();

	bool ret;

	if ( m_profiles.contains( profile ) )
		ret = m_profiles[profile].haveValidNamingScheme();
	else
		ret = false;

	m_configMutex.unlock();

	return ret;
}*/

bool AppConfig::profileIsValid( const QString& profile ) const
{
	EncodingProfile* options = profileOptions( profile );
	if ( options == 0 )
		return false;
	else
	{
		bool valid = profileOptionsAreValid( options );
		delete options;
		return valid;
	}
}

bool AppConfig::profileOptionsAreValid( const EncodingProfile* options ) const
{
	if ( options == 0 )
		return false;

	return options->haveValidNamingScheme() && encodeTypeIsSupported( options->encodeType() );
}

QStringList AppConfig::validProfiles() const
{
	m_configMutex.lock();

	QStringList ret;
	for ( QMap<QString,EncodingProfile>::ConstIterator it = m_profiles.begin(); it != m_profiles.end(); ++it )
		if ( profileOptionsAreValid( &it.data() ) )
			ret.append( String::deepCopy( it.key() ) );

	m_configMutex.unlock();

	return ret;
}

QStringList AppConfig::invalidProfiles() const
{
	m_configMutex.lock();

	QStringList ret;
	for ( QMap<QString,EncodingProfile>::ConstIterator it = m_profiles.begin(); it != m_profiles.end(); ++it )
		if ( ! profileOptionsAreValid( &it.data() ) )
			ret.append( String::deepCopy( it.key() ) );

	m_configMutex.unlock();

	return ret;
}


EncodingProfile* AppConfig::profileOptions( const QString& profile ) const
{
	m_configMutex.lock();

	EncodingProfile* ret;
	if ( m_profiles.contains( profile ) )
		ret = new EncodingProfile( m_profiles[profile] );
	else
		ret = 0;

	m_configMutex.unlock();

	return ret;
}

QString AppConfig::defaultFile() const
{
	m_configMutex.lock();

	QString ret = String::deepCopy( m_defaultFile );

	m_configMutex.unlock();

	return ret;
}


void AppConfig::setGeneralOption(
	GeneralOpt::TGeneralOpt option,
	const QString& value,
	bool& emitChanged,
	QValueList<int>& changedOptions,
	QStringList& changedProfiles )
{
	if ( option >= GeneralOpt::SIZE )
		return;

	QString aux = value;

	if ( option == GeneralOpt::proc_priority )
	{
		int priority = aux.toInt();
		if ( priority < 0 )
			aux = "0";
		else if ( priority > 19 )
			aux = "19";
	}
	else if ( option == GeneralOpt::threads )
	{
		int threads = aux.toInt();
		if ( threads < 1 )
			aux = "1";
		else if ( threads > 8 )
			aux = "4";
	}
	else if ( option == GeneralOpt::naming_scheme || option == GeneralOpt::log )
	{
		aux.replace( QRegExp( "^(~|\\$\\{HOME\\})" ), System::homeDir() );
		aux.replace( QRegExp( "^\\$\\{TMPDIR\\}" ), System::tempDir() );
	}
	else if ( option == GeneralOpt::temp_dir && value.isEmpty() )
		aux = System::tempDir();

	if ( m_generalOptions[option] != aux )
	{
		m_generalOptions[option] = String::deepCopy( aux );
		changedOptions.append( option );
		emitChanged = true;

		if ( option == GeneralOpt::naming_scheme )
		{
			for ( QMap<QString,EncodingProfile>::Iterator it = m_profiles.begin(); it != m_profiles.end(); ++it )
			{
				if ( it.data().usesDefaultNamingScheme() )
					changedProfiles.append( it.key() );
				it.data().setDefaultNamingScheme( m_generalOptions[GeneralOpt::naming_scheme] );
			}
		}
	}
}

void AppConfig::setGeneralOption( GeneralOpt::TGeneralOpt option, const QString& value )
{
	m_configMutex.lock();

	bool emitChanged = false;
	QValueList<int> changedOptions;
	QStringList changedProf;
	setGeneralOption( option, value, emitChanged, changedOptions, changedProf );

	m_configMutex.unlock();

	if ( emitChanged )				emit changed();
	if ( changedProf.size() )		emit changedProfiles( changedProf );
	if ( changedOptions.size() )	emit changedGeneralOpts( changedOptions );
}

void AppConfig::reloadEncodeTypes()
{
	// TODO need to improve this to allow for dynamic loading of plugins

	QStringList pluginNames = PluginsManager::encoderPluginNames();
	m_suppEncodeTypes.clear();
	for ( QStringList::ConstIterator it = pluginNames.begin(); it != pluginNames.end(); ++it )
	{
		const EncoderPlugin* plugin = PluginsManager::encoderPlugin( *it );
		if ( plugin != 0 )
			m_suppEncodeTypes[*it] = plugin->isLocallySupported();
	}
}

void AppConfig::setProgramPath(
	const QString& program,
	const QString& path,
	bool& emitChanged,
	QStringList& addedSuppEncTypes,
	QStringList& removedSuppEncTypes )
{
	PluginsManager::setProgramPath( program, path );

	for ( QMap<QString,bool>::Iterator it = m_suppEncodeTypes.begin(); it != m_suppEncodeTypes.end(); ++it )
	{
		const EncoderPlugin* plugin = PluginsManager::encoderPlugin( it.key() );
		if ( plugin == 0 )
		{
			if ( it.data() )
			{
				emitChanged = true;
				m_suppEncodeTypes[it.key()] = false;
				removedSuppEncTypes.append( it.key() );
			}
		}
		else if ( plugin->isLocallySupported() != it.data() )
		{
			emitChanged = true;
			if ( it.data() )
			{
				m_suppEncodeTypes[it.key()] = false;
				removedSuppEncTypes.append( it.key() );
			}
			else
			{
				m_suppEncodeTypes[it.key()] = true;
				addedSuppEncTypes.append( it.key() );
			}
		}
	}
}


void AppConfig::setProgramPath( const QString& program, const QString& value )
{
	m_configMutex.lock();

	bool emitChanged = false;
	QStringList addedSuppEncTypes;
	QStringList removedSuppEncTypes;
	setProgramPath( program, value, emitChanged, addedSuppEncTypes, removedSuppEncTypes );

	m_configMutex.unlock();

	if ( emitChanged )					emit changed();
	if ( removedSuppEncTypes.size() )	emit removedSuppEncodeTypes( removedSuppEncTypes );
	if ( addedSuppEncTypes.size() )		emit addedSuppEncodeTypes( addedSuppEncTypes );
}

void AppConfig::setTagTransformation( Metadata::Tag tag, TextTrans::TTextTrans value, bool& emitChanged, QValueList<int>& changedTagTrans )
{
	if ( tag >= Metadata::SIZE )
		return;

	if ( m_tagTrans[tag] != value )
	{
		m_tagTrans[tag] = value;
		changedTagTrans.append( tag );
		emitChanged = true;
	}
}

void AppConfig::setTagTransformation( Metadata::Tag tag, TextTrans::TTextTrans value )
{
	m_configMutex.lock();

	bool emitChanged = false;
	QValueList<int> changedTagTrans;
	setTagTransformation( tag, value, emitChanged, changedTagTrans );

	m_configMutex.unlock();

	if ( emitChanged )				emit changed();
	if ( changedTagTrans.size() )	emit changedTagTransformations( changedTagTrans );
}

void AppConfig::setTagDefault( Metadata::Tag tag, const QString& value, bool& emitChanged, QValueList<int>& changedDefaults )
{
	if ( tag >= Metadata::SIZE )
		return;

	if ( m_tagDefaults[tag] != value )
	{
		m_tagDefaults[tag] = String::deepCopy( value );
		changedDefaults.append( tag );
		emitChanged = true;
	}
}

void AppConfig::setTagDefault( Metadata::Tag tag, const QString& value )
{
	m_configMutex.lock();

	bool emitChanged = false;
	QValueList<int> changedDefaults;
	setTagDefault( tag, value, emitChanged, changedDefaults );

	m_configMutex.unlock();

	if ( emitChanged )				emit changed();
	if ( changedDefaults.size() )	emit changedTagDefaults( changedDefaults );
}

void AppConfig::setEncodingProfile(
	const QString& profile,
	const EncodingProfile& opts,
	bool& emitChanged,
	QStringList& addedProfiles,
	QStringList& changedProfiles )
{
	if ( ! m_profiles.contains( profile ) )
	{
		m_profiles[String::deepCopy( profile )] = opts;
		addedProfiles.append( String::deepCopy( profile ) );
		emitChanged = true;
	}
	else if ( m_profiles[ profile ] != opts )
	{
		m_profiles[profile] = opts;
		changedProfiles.append( String::deepCopy( profile ) );
		emitChanged = true;
	}

	m_profiles[profile].setDefaultNamingScheme( m_generalOptions[GeneralOpt::naming_scheme] );
	m_profiles[profile].setNamingScheme( opts.displayNamingScheme() );
}


void AppConfig::setEncodingProfile( const QString& profile, const EncodingProfile& opts )
{
	m_configMutex.lock();

	bool emitChanged = false;
	QStringList addedProf;
	QStringList changedProf;
	setEncodingProfile( profile, opts, emitChanged, addedProf, changedProf );

	m_configMutex.unlock();

	if ( emitChanged )			emit changed();
	if ( addedProf.size() )		emit addedProfiles( addedProf );
	if ( changedProf.size() )	emit changedProfiles( changedProf );
}

void AppConfig::removeProfile( const QString& profile, bool& emitChanged, QStringList& removedProfiles )
{
	if ( m_profiles.contains( profile ) )
	{
		m_profiles.remove( profile );
		removedProfiles.append( String::deepCopy( profile ) );
		emitChanged = true;
	}
}

void AppConfig::removeProfile( const QString& profile )
{
	m_configMutex.lock();

	bool emitChanged = false;
	QStringList removedProf;
	removeProfile( profile, emitChanged, removedProf );

	m_configMutex.unlock();

	if ( emitChanged )			emit changed();
	if ( removedProf.size() )	emit removedProfiles( String::deepCopy( removedProf ) );
}

void AppConfig::clearProfiles()
{
	m_configMutex.lock();

	QStringList removedProf;
	for ( QMap<QString,EncodingProfile>::Iterator it = m_profiles.begin(); it != m_profiles.end(); ++it )
		removedProf.append( it.key() );
	m_profiles.clear();

	m_configMutex.unlock();

	if ( removedProf.size() )
	{
		emit changed();
		emit removedProfiles( String::deepCopy( removedProf ) );
	}
}

void AppConfig::setDefaultFile( const QString& defaultFile, bool& emitChangedDefault, QString& prevDefaultFile )
{
	if ( m_defaultFile != defaultFile )
	{
		emitChangedDefault = true;
		prevDefaultFile = m_defaultFile;
		m_defaultFile = String::deepCopy( defaultFile );
	}
}

void AppConfig::setDefaultFile( const QString& defaultFile )
{
	m_configMutex.lock();

	bool emitChangedDefault = false;
	QString prevDefaultFile;
	setDefaultFile( defaultFile, emitChangedDefault, prevDefaultFile );

	m_configMutex.unlock();

	if ( emitChangedDefault )	emit changedDefaultFile( prevDefaultFile );
}

void AppConfig::loadDefaults()
{
	if ( load( defaultFile() ) )
		return;

	m_configMutex.lock();

	QStringList removedProf, changedProf;
	for ( QMap<QString,EncodingProfile>::Iterator it = m_profiles.begin(); it != m_profiles.end(); ++it )
		removedProf.append( String::deepCopy( it.key() ) );
	m_profiles.clear();

	bool emitChanged = false;
	QValueList<int> changedOptions;
	QValueList<int> changedDefaults;
	QValueList<int> changedTagTrans;
	QStringList addedEncTypes;
	QStringList removedEncTypes;

	setGeneralOption(
		GeneralOpt::naming_scheme,
		System::homeDir() + "/" + TextTrans::toQString( TextTrans::titlecase ) +
			"{%{artist}/%{year} - %{album}/%{track##} - %{title}}." +
			TextTrans::toQString( TextTrans::lowercase ) + "{%{dst_ext}}",
		emitChanged,
		changedOptions,
		changedProf );
	setGeneralOption( GeneralOpt::notify, "true", emitChanged, changedOptions, changedProf );
	setGeneralOption( GeneralOpt::confirm, "true", emitChanged, changedOptions, changedProf );
	setGeneralOption( GeneralOpt::overwrite, "false", emitChanged, changedOptions, changedProf );
	setGeneralOption( GeneralOpt::case_insensitive, "true", emitChanged, changedOptions, changedProf );
	setGeneralOption( GeneralOpt::replace_chars, "*?\\:|<>", emitChanged, changedOptions, changedProf );
	setGeneralOption( GeneralOpt::proc_priority, "0", emitChanged, changedOptions, changedProf );
	setGeneralOption( GeneralOpt::threads, "2", emitChanged, changedOptions, changedProf );
	setGeneralOption( GeneralOpt::rt_progress, "true", emitChanged, changedOptions, changedProf );
	setGeneralOption( GeneralOpt::show_tray, "true", emitChanged, changedOptions, changedProf );
	setGeneralOption( GeneralOpt::log, "", emitChanged, changedOptions, changedProf );
	setGeneralOption( GeneralOpt::temp_dir, System::tempDir(), emitChanged, changedOptions, changedProf );

	setTagDefault( Metadata::artist, "unknown_artist", emitChanged, changedDefaults );
	setTagDefault( Metadata::album, "unknown_album", emitChanged, changedDefaults );
	setTagDefault( Metadata::year, "XXXX", emitChanged, changedDefaults );
	setTagDefault( Metadata::discnum, "XX", emitChanged, changedDefaults );
	setTagDefault( Metadata::track, "XX", emitChanged, changedDefaults );
	setTagDefault( Metadata::title, "unknown_title", emitChanged, changedDefaults );
	setTagDefault( Metadata::genre, "unknown_genre", emitChanged, changedDefaults );
	setTagDefault( Metadata::composer, "unknown_composer", emitChanged, changedDefaults );
	setTagDefault( Metadata::comment, "no_comment", emitChanged, changedDefaults );

	setTagTransformation( Metadata::artist, TextTrans::titlecase, emitChanged, changedDefaults );
	setTagTransformation( Metadata::album, TextTrans::titlecase, emitChanged, changedDefaults );
	setTagTransformation( Metadata::year, TextTrans::leaveasis, emitChanged, changedDefaults );
	setTagTransformation( Metadata::discnum, TextTrans::leaveasis, emitChanged, changedDefaults );
	setTagTransformation( Metadata::track, TextTrans::leaveasis, emitChanged, changedDefaults );
	setTagTransformation( Metadata::title, TextTrans::titlecase, emitChanged, changedDefaults );
	setTagTransformation( Metadata::genre, TextTrans::titlecase, emitChanged, changedDefaults );
	setTagTransformation( Metadata::composer, TextTrans::titlecase, emitChanged, changedDefaults );
	setTagTransformation( Metadata::comment, TextTrans::leaveasis, emitChanged, changedDefaults );

	QStringList programs = PluginsManager::programs();
	for ( QStringList::ConstIterator it = programs.begin(); it != programs.end(); ++it )
		setProgramPath( *it, PluginsManager::programPath( *it ), emitChanged, addedEncTypes, removedEncTypes );

	m_configMutex.unlock();

	if ( m_inited )
	{
		if ( emitChanged )					emit changed();
		if ( changedOptions.size() )		emit changedGeneralOpts( changedOptions );
		if ( changedDefaults.size() )		emit changedTagDefaults( changedDefaults );
		if ( changedTagTrans.size() )		emit changedTagTransformations( changedTagTrans );
		if ( addedEncTypes.size() )			emit addedSuppEncodeTypes( addedEncTypes );
		if ( removedEncTypes.size() )		emit removedSuppEncodeTypes( removedEncTypes );
		if ( removedProf.size() )			emit removedProfiles( removedProf );
	}
}

bool AppConfig::load( const QString& path )
{
	if ( ! System::isReadable( path ) )
		return false;

	m_configMutex.lock();

	QDir absPath( path );
	absPath.convertToAbs();
	KSimpleConfig* config = new KSimpleConfig( absPath.path(), true );

	bool emitChanged = false;
	QValueList<int> changedOptions;
	QValueList<int> changedDefaults;
	QValueList<int> changedTagTrans;
	QStringList addedEncTypes;
	QStringList removedEncTypes;
	QStringList removedProf;
	QStringList addedProf;
	QStringList changedProf;

	QStringList profilesGroups = config->groupList();
	QStringList settingsGroups;
	if ( profilesGroups.contains( "General" ) )
		{ profilesGroups.remove( "General" ); settingsGroups.append( "General" ); }
	if ( profilesGroups.contains( "Program Locations" ) )
		{ profilesGroups.remove( "Program Locations" ); settingsGroups.append( "Program Locations" ); }
	if ( profilesGroups.contains( "Tag Defaults" ) )
		{ profilesGroups.remove( "Tag Defaults" ); settingsGroups.append( "Tag Defaults" ); }
	if ( profilesGroups.contains( "Tag Cases" ) )
		{ profilesGroups.remove( "Tag Cases" ); settingsGroups.append( "Tag Cases" ); }

	profilesGroups.sort();
	for ( QStringList::Iterator it = profilesGroups.begin(); it != profilesGroups.end(); ++it )
	{
		config->setGroup( *it );

		EncodingProfile profileOpts;
		profileOpts.readFromConfig( config );

		if ( m_profiles.contains( *it ) )
		{
			if ( m_profiles[*it] != profileOpts ) // profile was changed
			{
				m_profiles[*it] = profileOpts;
				changedProf.append( *it );
				emitChanged = true;
			}
		}
		else // profile is added
		{
			m_profiles[*it] = profileOpts;
			addedProf.append( *it );
			emitChanged = true;
		}
	}

	for ( QMap<QString,EncodingProfile>::Iterator it = m_profiles.begin(); it != m_profiles.end(); ++it )
		if ( ! profilesGroups.contains( it.key() ) ) // profile was removed
			removedProf.append( it.key() );
	for ( QStringList::Iterator it = removedProf.begin(); it != removedProf.end(); ++it )
		m_profiles.remove( *it );
	if ( removedProf.size() )
		emitChanged = true;

	if ( settingsGroups.contains( "General" ) )
	{
		config->setGroup( "General" );

		for ( int idx = 0; idx < GeneralOpt::SIZE; ++idx )
			setGeneralOption(
				(GeneralOpt::TGeneralOpt)idx,
				config->readEntry( GeneralOpt::toQString( (GeneralOpt::TGeneralOpt)idx ), m_generalOptions[idx] ),
				emitChanged,
				changedOptions,
				changedProf );
	}

	if ( settingsGroups.contains( "Program Locations" ) )
	{
		config->setGroup( "Program Locations" );

		QStringList programs = PluginsManager::programs();
		for ( QStringList::ConstIterator it = programs.begin(); it != programs.end(); ++it )
			setProgramPath( *it, config->readEntry( *it, PluginsManager::programPath( *it ) ), emitChanged, addedEncTypes, removedEncTypes );
	}

	if ( settingsGroups.contains( "Tag Defaults" ) )
	{
		config->setGroup( "Tag Defaults" );

		for ( int idx = 0; idx < Metadata::SIZE; ++idx )
			setTagDefault( (Metadata::Tag)idx,
			config->readEntry( Metadata::toQString( (Metadata::Tag)idx ), m_tagDefaults[idx] ),
			emitChanged,
			changedDefaults );
	}

	if ( settingsGroups.contains( "Tag Cases" ) )
	{
		config->setGroup( "Tag Cases" );

		for ( int idx = 0; idx < Metadata::SIZE; ++idx )
			setTagTransformation( (Metadata::Tag)idx,
			TextTrans::fromQString( config->readEntry( Metadata::toQString( (Metadata::Tag)idx ), TextTrans::toQString( m_tagTrans[idx] ) ) ),
			emitChanged,
			changedTagTrans );
	}

	m_configMutex.unlock();

	delete config;


	if ( m_inited )
	{
		if ( emitChanged )					emit changed();
		if ( changedOptions.size() )		emit changedGeneralOpts( changedOptions );
		if ( changedDefaults.size() )		emit changedTagDefaults( changedDefaults );
		if ( changedTagTrans.size() )		emit changedTagTransformations( changedTagTrans );
		if ( addedEncTypes.size() )			emit addedSuppEncodeTypes( addedEncTypes );
		if ( removedEncTypes.size() )		emit removedSuppEncodeTypes( removedEncTypes );
		if ( addedProf.size() )				emit addedProfiles( addedProf );
		if ( removedProf.size() )			emit removedProfiles( removedProf );
		if ( changedProf.size() )			emit changedProfiles( changedProf );
	}

	return true;
}

// Saves configuration to file
bool AppConfig::save( const QString& path ) const
{
	QDir absPath( path );
	absPath.convertToAbs();
	System::remove( absPath.path() );
	KSimpleConfig* config = new KSimpleConfig( absPath.path() );

	m_configMutex.lock();

	config->setGroup( "General" );
	for ( int idx = 0; idx < GeneralOpt::SIZE; ++idx )
		config->writeEntry( GeneralOpt::toQString( (GeneralOpt::TGeneralOpt)idx ), m_generalOptions[idx] );

	config->setGroup( "Program Locations" );
	QStringList programs = PluginsManager::programs();
	for ( QStringList::ConstIterator it = programs.begin(); it != programs.end(); ++it )
		config->writeEntry( *it, PluginsManager::programPath( *it ) );

	config->setGroup( "Tag Defaults" );
	for ( int idx = 0; idx < Metadata::SIZE; ++idx )
		config->writeEntry( Metadata::toQString( (Metadata::Tag)idx ), m_tagDefaults[idx] );

	config->setGroup( "Tag Cases" );
	for ( int idx = 0; idx < Metadata::SIZE; ++idx )
		config->writeEntry( Metadata::toQString( (Metadata::Tag)idx ), TextTrans::toQString( (TextTrans::TTextTrans)m_tagTrans[idx]) );

	for ( QMap<QString,EncodingProfile>::ConstIterator it = m_profiles.begin(); it != m_profiles.end(); ++it )
	{
		config->setGroup( it.key() );

		it.data().writeToConfig( config );
	}

	m_configMutex.unlock();

	delete config;

	return System::isWritable( path );
}
