#ifndef TRANSKODE_APPLICATIONCONFIG_H
#define TRANSKODE_APPLICATIONCONFIG_H

/***************************************************************************
 *   Copyright (C) 2005 by Sergio Pistone                                  *
 *   sergio_pistone@yahoo.com.ar                                           *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
	#include <config.h>
#endif

#include "encodingprofile.h"
#include <commondefs.h>
#include <metadata.h>

#include <qobject.h>
#include <qmutex.h>
#include <qstring.h>
#include <qstringlist.h>
#include <qvaluelist.h>
#include <qmap.h>

namespace TransKode
{
	class AppConfig : public QObject
	{
		Q_OBJECT

		public:
			AppConfig( const QString& defaultFile="" );
			AppConfig( const AppConfig& config );
			AppConfig& operator=( const AppConfig& config );

			QString defaultFile() const;
			void setDefaultFile( const QString& defaultFile );

			void loadDefaults();
			bool load( const QString& path );
			bool save( const QString& path ) const;

			bool encodeTypeIsSupported( const QString& encodeType ) const;
			QString supportedEncodingExtensions() const;

			QString programPath( const QString& program, bool quoted=false ) const;
			QString generalOption( GeneralOpt::TGeneralOpt option ) const;
			TextTrans::TTextTrans tagTransformation( Metadata::Tag tag ) const;
			QString tagDefault( Metadata::Tag tag ) const;

			void setProgramPath( const QString& program, const QString& value );
			void setGeneralOption( GeneralOpt::TGeneralOpt option, const QString& value );
			void setTagTransformation( Metadata::Tag tag, TextTrans::TTextTrans value );
			void setTagDefault( Metadata::Tag tag, const QString& value );

			QStringList profiles() const;

			bool profileOptionsAreValid( const EncodingProfile* options ) const;
			bool profileIsValid( const QString& profile ) const;
			QStringList validProfiles() const;
			QStringList invalidProfiles() const;

			EncodingProfile* profileOptions( const QString& profile ) const;
			void setEncodingProfile( const QString& profile, const EncodingProfile& opts );
			void removeProfile( const QString& profile );
			void clearProfiles();

		signals:
			void changed();

			void addedProfiles( const QStringList& profiles );
			void removedProfiles( const QStringList& profiles );
			void changedProfiles( const QStringList& profiles );

			void addedSuppEncodeTypes( const QStringList& encodeTypes );
			void removedSuppEncodeTypes( const QStringList& encodeTypes );

			void changedGeneralOpts( const QValueList<int>& options );
			void changedTagTransformations( const QValueList<int>& tags );
			void changedTagDefaults( const QValueList<int>& tags );

			void changedDefaultFile( const QString& prevDefaultFile );

		protected:

			void reloadEncodeTypes();

			void setGeneralOption(
				GeneralOpt::TGeneralOpt option,
				const QString& value,
				bool& emitChanged,
				QValueList<int>& changedOptions,
				QStringList& changedProfiles );

			void setProgramPath(
				const QString& program,
				const QString& value,
				bool& emitChanged,
				QStringList& addedSuppEncTypes,
				QStringList& removedSuppEncTypes );

			void setTagTransformation( Metadata::Tag tag, TextTrans::TTextTrans value, bool& emitChanged, QValueList<int>& changedTagTrans );

			void setTagDefault( Metadata::Tag tag, const QString& value, bool& emitChanged, QValueList<int>& changedDefaults );

			void setEncodingProfile(
				const QString& profile,
				const EncodingProfile& opts,
				bool& emitChanged,
				QStringList& addedProfiles,
				QStringList& changedProfiles );

			void removeProfile( const QString& profile, bool& emitChanged, QStringList& removedProfiles );

			void setDefaultFile( const QString& defFile, bool& emitChangedDefault, QString& prevDefaultFile );


			mutable QMutex m_configMutex;

			bool m_inited; // flag to indicate if signals should be emited on modification
			QString m_defaultFile; // file with default settings & profiles

			QString m_generalOptions[GeneralOpt::SIZE];
			TextTrans::TTextTrans m_tagTrans[Metadata::SIZE];
			QString m_tagDefaults[Metadata::SIZE];
			QMap<QString,bool> m_suppEncodeTypes;
			QMap<QString,EncodingProfile> m_profiles;
	};
}

#endif
