#ifndef TRANSKODE_ENCODINGPROFILE_H
#define TRANSKODE_ENCODINGPROFILE_H

/***************************************************************************
 *   Copyright (C) 2005 by Sergio Pistone                                  *
 *   sergio_pistone@yahoo.com.ar                                           *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
	#include <config.h>
#endif

#include <commondefs.h>

#include <qmap.h>
#include <qstring.h>
#include <qregexp.h>
#include <qdir.h>
#include <qfileinfo.h>

#include <kconfig.h>
#include <kstandarddirs.h>
#include <kapplication.h>

namespace TransKode
{
	class EncodingProfile
	{
		friend class AppConfig;

		public:
			EncodingProfile()
			{
			};

			EncodingProfile( const EncodingProfile& prof )
			{
				this->operator=( prof );
			};

			EncodingProfile( KConfig* config )
			{
				readFromConfig( config, true );
			};

			EncodingProfile& operator=( const EncodingProfile& prof )
			{
				m_defaultNamingScheme = String::deepCopy( prof.m_defaultNamingScheme );
				m_namingScheme = String::deepCopy( prof.m_namingScheme );
				m_switches = String::deepCopy( prof.m_switches );
				m_extension = String::deepCopy( prof.m_extension );
				m_encodeType = String::deepCopy( prof.m_encodeType );
				m_show = prof.m_show;

				return *this;
			};

			bool operator==( const EncodingProfile& prof )
			{
				if ( namingScheme() != prof.namingScheme() )	return false;
				if ( show() != prof.show() )					return false;

				if ( bypass() && prof.bypass() )
					return true;
				else
				{
					if ( encodeType() != prof.encodeType() )		return false;
					if ( switches() != prof.switches() )			return false;
					if ( extension() != prof.extension() )			return false;
					return true;
				}
			};

			bool operator!=( const EncodingProfile& prof ) { return ! operator==( prof ); };

			void setValue( ProfileOpt::TProfileOpt option, QString value, bool deepCopy=false )
			{
				switch ( option )
				{
					case ProfileOpt::naming_scheme:		setNamingScheme( value, deepCopy );
														return;
					case ProfileOpt::encode_type:		setEncodeType( value, deepCopy );
														return;
					case ProfileOpt::extension:			setExtension( value, deepCopy );
														return;
					case ProfileOpt::switches:			setSwitches( value, deepCopy );
														return;
					case ProfileOpt::show:				setShow( value == "true" );
														return;
					case ProfileOpt::bypass:			setBypass();
														return;
					default:							return;
				}
			};

			QString getValue( ProfileOpt::TProfileOpt option, bool deepCopy=false ) const
			{
				switch ( option )
				{
					case ProfileOpt::naming_scheme:		return namingScheme( deepCopy );
					case ProfileOpt::encode_type:		return encodeType( deepCopy );
					case ProfileOpt::extension:			return extension( deepCopy );
					case ProfileOpt::switches:			return switches( deepCopy );
					case ProfileOpt::show:				return show() ? "true" : "false";
					case ProfileOpt::bypass:			return bypass() ? "true" : "false";
					default:							return QString::null;
				}
			};

			void readFromConfig( KConfig* config, bool deepCopy=false )
			{
				if ( ! config )
					return;

				m_encodeType = config->readEntry( "encode_type", "Bypass" );
				m_show = config->readEntry( "show", "false" ) == "true" ? true : false;
				m_switches = bypass() ? "" : config->readEntry( "switches", "" ).stripWhiteSpace();
				m_extension = bypass() ? "" : config->readEntry( "extension", "" );
				m_namingScheme = config->readEntry( "naming_scheme", "" ).replace( "../", "" );

				if ( config->hasGroup( "General" ) )
				{
					QString profileGroup = config->group();

					config->setGroup( "General");
					m_defaultNamingScheme = config->readEntry( "naming_scheme", "" ).replace( "../", "" );

					config->setGroup( profileGroup );
				}

				if ( deepCopy )
				{
					m_encodeType = String::deepCopy( m_encodeType );
					m_switches = String::deepCopy( m_switches );
					m_extension = String::deepCopy( m_extension );
					m_namingScheme = String::deepCopy( m_namingScheme );
					m_defaultNamingScheme = String::deepCopy( m_defaultNamingScheme );
				}
			};

			void writeToConfig( KConfig* config, bool deepCopy=false ) const
			{
				if ( ! config )
					return;

				config->writeEntry( "encode_type", deepCopy ? String::deepCopy( m_encodeType ) : m_encodeType );
				config->writeEntry( "show", m_show ? "true" : "false" );
				config->writeEntry( "switches", deepCopy ? String::deepCopy( m_switches ) : m_switches );
				config->writeEntry( "extension", deepCopy ? String::deepCopy( m_extension ) : m_extension );
				config->writeEntry( "naming_scheme", deepCopy ? String::deepCopy( m_namingScheme ) : m_namingScheme );
			};

			bool usesDefaultNamingScheme() const
			{
				return m_namingScheme.isEmpty();
			};

			bool haveValidNamingScheme() const
			{
				if ( namingScheme().isEmpty() )
					return false;

				QString expr( namingScheme() );
				expr.replace( "%{artist}", "" );
				expr.replace( "%{album}", "" );
				expr.replace( "%{year}", "" );
				expr.replace( "%{track#}", "" );
				expr.replace( "%{track##}", "" );
				expr.replace( "%{title}", "" );
				expr.replace( "%{genre}", "" );
				expr.replace( "%{comment}", "" );
				expr.replace( "%{dst_ext}", "" );
				expr.replace( "%{src_type}", "" );
				expr.replace( "%{src_ext}", "" );
				expr.replace( "%{src_name}", "" );
				expr.replace( "%{src_dir}", "" );

				return String::isValidTextTransformationExpression( expr );
			};

			QString displayNamingScheme( bool deepCopy=false ) const
			{
				return deepCopy ? String::deepCopy( m_namingScheme ) : m_namingScheme;
			};

			QString namingScheme( bool deepCopy=false ) const
			{
				QString ret = usesDefaultNamingScheme() ? m_defaultNamingScheme : m_namingScheme;
				return deepCopy ? String::deepCopy( ret ) : ret;
			};

			void setNamingScheme( const QString& namingScheme, bool deepCopy=false )
			{
				m_namingScheme = deepCopy ? String::deepCopy( namingScheme ) : namingScheme;
				m_namingScheme.replace( QRegExp( "^(~|\\$\\{HOME\\})" ), System::homeDir() );
				m_namingScheme.replace( QRegExp( "^\\$\\{TMPDIR\\}" ), System::tempDir() );
			};

			QString encodeType( bool deepCopy=false ) const
			{
				return deepCopy ? String::deepCopy( m_encodeType ) : m_encodeType;
			};

			void setEncodeType( const QString& encodeType, bool deepCopy=false )
			{
				m_encodeType = deepCopy ? String::deepCopy( encodeType ) : encodeType;
			};

			QString extension( bool deepCopy=false ) const
			{
				return deepCopy ? String::deepCopy( m_extension ) : m_extension;
			};

			void setExtension( const QString& extension,  bool deepCopy=false )
			{
				m_extension = deepCopy ? String::deepCopy( extension ) : extension;
			};

			QString switches( bool deepCopy=false ) const
			{
				return deepCopy ? String::deepCopy( m_switches ) : m_switches;
			};

			void setSwitches( const QString& switches, bool deepCopy=false )
			{
				m_switches = deepCopy ? String::deepCopy( switches ) : switches;
			};

			bool show() const
			{
				return m_show;
			};

			void setShow( bool value )
			{
				m_show = value;
			};

			bool bypass() const
			{
				return m_encodeType == "Bypass";
			};

			void setBypass()
			{
				setEncodeType( "Bypass" );
				setSwitches( "" );
				setExtension( "" );
			};


		protected:

			// This two MUST be setted by the owner config as they are added to it
			void setDefaultNamingScheme( const QString& defaultNamingScheme )
			{
				m_defaultNamingScheme = String::deepCopy( defaultNamingScheme );
			};

			QString m_defaultNamingScheme;
			QString m_namingScheme;
			QString m_extension;
			QString m_switches;
			QString m_encodeType;
			bool m_show;
	};
}

#endif
