/***************************************************************************
 *   Copyright (C) 2005 by Sergio Pistone                                  *
 *   sergio_pistone@yahoo.com.ar                                           *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "configdialog.h"
#include <pluginsmanager.h>

#include <qcheckbox.h>
#include <qpushbutton.h>
#include <qslider.h>
#include <qlabel.h>
#include <qgroupbox.h>
#include <qtooltip.h>
#include <qvbox.h>
#include <qlayout.h>
#include <qscrollview.h>
#include <kcombobox.h>
#include <klineedit.h>

#include <kmessagebox.h>
#include <kurlrequester.h>
#include <klocale.h>

using namespace TransKode;

ConfigDialog::ConfigDialog( const QString& defaultsPath, bool script, QWidget* parent, const char* name ):
	KDialogBase::KDialogBase(	KDialogBase::IconList,	// dialogFace
								0,						// flags
								parent,					// parent
								name,					// name
								false,					// modal
								"Configure",			// caption
								Ok|Apply|Cancel|Default,// buttonMask
								Ok, 					// defaultButton
								true ),					// separator
	m_iconLoader(),
	amarokScript( script ),
	acceptedConfig(),
	dirtyConfig(),
	defaultConfig( defaultsPath ),
	selectedProfile( "" )
{
	showButton( Default, true );

	addPage( generalSettings = new GeneralSettings(), "General", m_iconLoader.loadIcon( "misc", KIcon::Desktop, KIcon::SizeMedium ), "General Settings" );

	generalSettings->tempDirURLRequester->setCaption( i18n( "Select the directory for temporary files" ) );
	generalSettings->tempDirURLRequester->setMode( KFile::Directory|KFile::LocalOnly );

	generalSettings->logEventsURLRequester->setCaption( i18n( "Select the path for the log file" ) );
	generalSettings->logEventsURLRequester->setMode( KFile::File|KFile::LocalOnly );
	generalSettings->logEventsURLRequester->setFilter( "*.log" );

	addPage( profileSettings = new ProfileSettings(), "Profiles", m_iconLoader.loadIcon( "favorites", KIcon::Desktop, KIcon::SizeMedium ), "Profiles Settings" );

	QScrollView* programSettingsSV = new QScrollView( 0, "ProgramSettingsScrollView" );
	programSettingsSV->setResizePolicy( QScrollView::AutoOneFit );
// 	programSettingsSV->setFrameShape( QFrame::NoFrame );
// 	programSettingsSV->setFrameShadow( QFrame::Plain );
	QBoxLayout* layout = new QBoxLayout( QBoxLayout::TopToBottom );
	layout->addWidget( programSettingsSV );
	programSettings = new ProgramSettings( programSettingsSV->viewport() );
	programSettingsSV->addChild( programSettings );

	addPage( programSettingsSV, "Programs", m_iconLoader.loadIcon( "package_system", KIcon::Desktop, KIcon::SizeMedium ), "Program Settings" );

	addPage( tagSettings = new TagSettings(), "Metadata", m_iconLoader.loadIcon( "rss_tag", KIcon::Desktop, KIcon::SizeMedium ), "Tag Settings" );

	connect( this, SIGNAL( applyClicked(void) ), this, SLOT( acceptConfig(void) ) );
	connect( this, SIGNAL( okClicked(void) ), this, SLOT( acceptConfigAndClose(void) ) );
	connect( this, SIGNAL( cancelClicked(void) ), this, SLOT( rejectConfigAndClose(void) ) );
	connect( this, SIGNAL( defaultClicked(void) ), this, SLOT( setControlsFromDefaults(void) ) );
	connect( this, SIGNAL( defaultClicked(void) ), this, SLOT( enableApply(void) ) );

	setControlsFromConfig();

	// Apply button connections/setup
	connect( this, SIGNAL( defaultClicked(void) ), this, SLOT( enableApply(void) ) );
	connect( generalSettings, SIGNAL( settingsChanged() ), this, SLOT( enableApply(void) ) );
	connect( programSettings, SIGNAL( settingsChanged() ), this, SLOT( enableApply(void) ) );
	connect( tagSettings, SIGNAL( settingsChanged() ), this, SLOT( enableApply(void) ) );
	connect( profileSettings, SIGNAL( settingsChanged() ), this, SLOT( enableApply(void) ) );

	// Profiles setttings connections/setup
	connect( profileSettings->defaultNamingSchemeLineEdit, SIGNAL( textChanged(const QString&) ), this, SLOT( setProfileStatus(void) ) );
	connect( profileSettings->saveProfileButton, SIGNAL( clicked(void) ), this, SLOT( saveSelectedProfile(void) ) );
	connect( profileSettings->removeProfileButton, SIGNAL( clicked(void) ), this, SLOT( removeSelectedProfile(void) ) );
	connect( profileSettings->selectedProfileComboBox, SIGNAL( activated(int) ), this, SLOT( changeSelectedProfile(void) ) );

	changeSelectedProfile();
	enableButtonApply( false );

	if ( ! amarokScript ) // this widgets only make sense for amarokScript
	{
		QToolTip::add(
			profileSettings->placeInMenuCheckBox, i18n(
			"<qt>Add this profile to transKode's jobs list context menu."
			"<br/>Note: invalid profiles are <i>never</i> shown in the menu.</qt>" ) );

		generalSettings->askConfirmationCheckBox->hide();
		generalSettings->notifyProgressCheckBox->hide();
	}
	else
	{
		QToolTip::add(
			profileSettings->placeInMenuCheckBox, i18n(
			"<qt>Add this profile to 'transKode' entry in Amarok's playlist context menu."
			"<br/>Note: invalid profiles are <i>never</i> shown in the menu.</qt>" ) );

		generalSettings->showTrayCheckBox->hide();
		generalSettings->rtProgressCheckBox->hide();
		generalSettings->rtProgressCheckBox->setChecked( false );

		static const QStringList amarokTTMDExtensions = QStringList::split( " ", "aac flac m4a mp3 ogg wav wma" );
		for ( QStringList::ConstIterator it = amarokTTMDExtensions.begin(); it != amarokTTMDExtensions.end(); ++it )
			fixedProfiles.append( QString( "Amarok TTMD - " ) + *it );
	}

	profileSettings->encodingTypeComboBox->insertItem( "Bypass" );
	QStringList pluginNames = PluginsManager::encoderPluginNames();
	for ( QStringList::Iterator it = pluginNames.begin(); it != pluginNames.end(); ++it )
		profileSettings->encodingTypeComboBox->insertItem( *it );

	resize( QSize( 530, 300 ).expandedTo( minimumSizeHint() ) );
}

ConfigDialog::~ConfigDialog()
{
}

const AppConfig& ConfigDialog::getConfig() const
{
	return acceptedConfig;
}

void ConfigDialog::setConfig( const AppConfig& cfg )
{
	bool visible = isVisible();

	if ( visible )
		hide();

	acceptedConfig = dirtyConfig = cfg;
	setControlsFromConfig();

	enableButtonApply( false );

	if ( visible )
		show();
}

void ConfigDialog::saveSelectedProfile()
{
	QString oldProfile = "";
	QString newProfile = profileSettings->selectedProfileComboBox->lineEdit()->text();
	if ( newProfile == "" )
		return;

	// Alert if newProfile colides with an existing one
	int idx = -1;
	if ( profileSettings->selectedProfileComboBox->count() )
		for ( idx = 0; idx < profileSettings->selectedProfileComboBox->count(); ++idx )
		{
			oldProfile = profileSettings->selectedProfileComboBox->text( idx );
			if ( oldProfile == newProfile )
			{
				if ( KMessageBox::warningContinueCancel(
						this,
						i18n( "<qt><p align=\"center\">Do you want to overwrite it?</p></qt>" ),
						i18n( "Profile Already Exists" ) ) == KMessageBox::Cancel )
					return;
				break;
			}
		}

	EncodingProfile profileOptions;

	profileOptions.setNamingScheme( profileSettings->namingSchemeLineEdit->text() );
	profileOptions.setSwitches( profileSettings->encodingSwitchesLineEdit->text() );
	profileOptions.setShow( profileSettings->placeInMenuCheckBox->isChecked() );
	if ( ! fixedProfiles.contains( newProfile ) )
	{
		QLineEdit* lineEdit = profileSettings->outputExtensionComboBox->lineEdit();
		profileOptions.setExtension( lineEdit ? lineEdit->text() : profileSettings->outputExtensionComboBox->currentText() );
		profileOptions.setEncodeType( profileSettings->encodingTypeComboBox->currentText() );
	}
	else
	{
		EncodingProfile* fixedOptions = dirtyConfig.profileOptions( newProfile );
		if ( fixedOptions != 0 )
		{
			profileOptions.setExtension( fixedOptions->extension() );
			profileOptions.setEncodeType( fixedOptions->encodeType() );
			delete fixedOptions;
		}
		else // this should never happen!!!
			return;

		QString selectedItemProfile = profileSettings->selectedProfileComboBox->text(
			profileSettings->selectedProfileComboBox->currentItem() );
		if ( selectedItemProfile != newProfile )
		{
			KMessageBox::information(
				this,
				i18n( "<qt>Encode type and extension can't be changed for this profile</qt>" ),
				i18n( "Special Profile" ),
				"fixed profiles" );
			// if we get here, idx contains the index of the profile we are overwriting
			profileSettings->selectedProfileComboBox->setCurrentItem( idx );
		}
	}

	dirtyConfig.setEncodingProfile( newProfile, profileOptions );

	if ( oldProfile != newProfile ) // profile didn't exist before
	{
		profileSettings->selectedProfileComboBox->insertItem( newProfile );
		profileSettings->selectedProfileComboBox->setCurrentItem( profileSettings->selectedProfileComboBox->count()-1 );
	}

	changeSelectedProfile();
}

void ConfigDialog::removeSelectedProfile()
{
	if ( profileSettings->selectedProfileComboBox->count() )
	{
		int currentIdx = profileSettings->selectedProfileComboBox->currentItem();
		QString oldProfile = profileSettings->selectedProfileComboBox->text( currentIdx );
		QString newProfile = profileSettings->selectedProfileComboBox->lineEdit()->text();
		if ( oldProfile != newProfile )
			return;

		profileSettings->selectedProfileComboBox->removeItem( currentIdx );
		dirtyConfig.removeProfile( oldProfile );

		if ( profileSettings->selectedProfileComboBox->count() )
		{
			profileSettings->selectedProfileComboBox->setCurrentItem( currentIdx != 0 ? currentIdx - 1 : 0 );
			changeSelectedProfile(); // It's not automatically called
		}
		else
		{
			profileSettings->profileStatusMsgLabel->setText( "" );
			profileSettings->profileStatusImgLabel->setPixmap( QPixmap() );
		}
	}
}

void ConfigDialog::changeSelectedProfile()
{
	selectedProfile = profileSettings->selectedProfileComboBox->currentText();
	if ( selectedProfile.isEmpty() )
	{
		QStringList pluginNames = PluginsManager::encoderPluginNames();
		if ( pluginNames.size() > 0 )
		{
			profileSettings->changeEncodingType( pluginNames.first() );
			profileSettings->profileStatusMsgLabel->setText( "" );
			profileSettings->profileStatusImgLabel->setPixmap( QPixmap() );
		}
		return;
	}

	EncodingProfile* profileOptions = dirtyConfig.profileOptions( selectedProfile );
	if ( ! profileOptions )
		return;
	profileSettings->changeEncodingType( profileOptions->encodeType() );
	profileSettings->namingSchemeLineEdit->setText( profileOptions->displayNamingScheme() );
	profileSettings->encodingSwitchesLineEdit->setText( profileOptions->switches() );
	QLineEdit* lineEdit = profileSettings->outputExtensionComboBox->lineEdit();
	if ( lineEdit != 0 )
		lineEdit->setText( profileOptions->extension() );
	profileSettings->placeInMenuCheckBox->setChecked( profileOptions->show() );
	delete profileOptions;

	bool fixed = fixedProfiles.contains( selectedProfile );
	profileSettings->encodingTypeComboBox->setDisabled( fixed );
	profileSettings->outputExtensionComboBox->setDisabled( fixed );
	profileSettings->removeProfileButton->setDisabled( fixed );

	setProfileStatus();
}

void ConfigDialog::setProfileStatus()
{
	if ( selectedProfile.isEmpty() )
	{
		profileSettings->profileStatusMsgLabel->setText( "" );
		profileSettings->profileStatusImgLabel->setPixmap( QPixmap() );
		return;
	}

	QString status;
	QPixmap image;

	EncodingProfile* profileOpts = dirtyConfig.profileOptions( selectedProfile );
	if ( ! profileOpts )
		return;

	if ( ( dirtyConfig.generalOption( GeneralOpt::naming_scheme ) != profileSettings->defaultNamingSchemeLineEdit->text()
		   && profileOpts->usesDefaultNamingScheme() ) )
	{
		status = i18n(	"<qt>Modified defaults prevent validation of selected profile<br/>"
						"(press apply or undo changes to remove this warning)</qt>" );
		image = m_iconLoader.loadIcon( "messagebox_warning", KIcon::Desktop, KIcon::SizeMedium );
	}
	else
	{
		if ( ! dirtyConfig.encodeTypeIsSupported( profileOpts->encodeType() ) )
			status = i18n( " - Encoding type is not supported" );

		if ( ! profileOpts->haveValidNamingScheme() )
			status += (status.isEmpty() ? "" : "<br/>") + i18n( " - Naming scheme is invalid" );

		if ( ! status.isEmpty() )
		{
			status = "<qt>" + i18n( "Problems found with selected profile:" ) + "<br/>" + status + "</qt>";
			image = m_iconLoader.loadIcon( "messagebox_critical", KIcon::Desktop, KIcon::SizeMedium );
		}
		else
		{
			status = i18n( "<qt>No problems found with selected profile</qt>" );
			image = m_iconLoader.loadIcon( "messagebox_info", KIcon::Desktop, KIcon::SizeMedium );
		}
	}

	delete profileOpts;

	profileSettings->profileStatusMsgLabel->setText( status );
	profileSettings->profileStatusImgLabel->setPixmap( image );
}

void ConfigDialog::setControlsFromConfig()
{
	generalSettings->overwriteCheckBox->setChecked( dirtyConfig.generalOption( GeneralOpt::overwrite ) == "true" ? true:false );
	generalSettings->caseInsensitiveCheckBox->setChecked( dirtyConfig.generalOption( GeneralOpt::case_insensitive ) == "true" ? true:false );
	generalSettings->replaceCharsCheckBox->setChecked( dirtyConfig.generalOption( GeneralOpt::replace_chars ).isEmpty() ? false:true );

	generalSettings->notifyProgressCheckBox->setChecked( dirtyConfig.generalOption( GeneralOpt::notify ) == "true" ? true:false );
	generalSettings->askConfirmationCheckBox->setChecked( dirtyConfig.generalOption( GeneralOpt::confirm ) == "true" ? true:false );

	generalSettings->showTrayCheckBox->setChecked( dirtyConfig.generalOption( GeneralOpt::show_tray ) == "true" );

	generalSettings->logEventsCheckBox->setChecked( ! dirtyConfig.generalOption( GeneralOpt::log ).isEmpty() );
	if ( generalSettings->logEventsCheckBox->isChecked() )
		generalSettings->logEventsURLRequester->setURL( dirtyConfig.generalOption( GeneralOpt::log ) );

	generalSettings->tempDirURLRequester->setURL( dirtyConfig.generalOption( GeneralOpt::temp_dir ) );

	generalSettings->threadsSlider->setValue( dirtyConfig.generalOption( GeneralOpt::threads ).toInt() );

	generalSettings->prioritySlider->setValue( dirtyConfig.generalOption( GeneralOpt::proc_priority ).toInt() );

	generalSettings->rtProgressCheckBox->setChecked( dirtyConfig.generalOption( GeneralOpt::rt_progress ) == "true" );


	tagSettings->artistComboBox->setCurrentItem( dirtyConfig.tagTransformation( Metadata::artist ) );
	tagSettings->albumComboBox->setCurrentItem( dirtyConfig.tagTransformation( Metadata::album ) );
	tagSettings->titleComboBox->setCurrentItem( dirtyConfig.tagTransformation( Metadata::title ) );
	tagSettings->genreComboBox->setCurrentItem( dirtyConfig.tagTransformation( Metadata::genre ) );
	tagSettings->composerComboBox->setCurrentItem( dirtyConfig.tagTransformation( Metadata::composer ) );
	tagSettings->commentComboBox->setCurrentItem( dirtyConfig.tagTransformation( Metadata::comment ) );

	tagSettings->artistLineEdit->setText( dirtyConfig.tagDefault( Metadata::artist ) );
	tagSettings->yearLineEdit->setText( dirtyConfig.tagDefault( Metadata::year ) );
	tagSettings->albumLineEdit->setText( dirtyConfig.tagDefault( Metadata::album ) );
	tagSettings->discNumberLineEdit->setText( dirtyConfig.tagDefault( Metadata::discnum ) );
	tagSettings->trackNumberLineEdit->setText( dirtyConfig.tagDefault( Metadata::track ) );
	tagSettings->titleLineEdit->setText( dirtyConfig.tagDefault( Metadata::title ) );
	tagSettings->genreLineEdit->setText( dirtyConfig.tagDefault( Metadata::genre ) );
	tagSettings->composerLineEdit->setText( dirtyConfig.tagDefault( Metadata::composer ) );
	tagSettings->commentLineEdit->setText( dirtyConfig.tagDefault( Metadata::comment ) );

	QStringList programs = PluginsManager::programs();
	for ( QStringList::Iterator it = programs.begin(); it != programs.end(); ++it )
	{
		KURLRequester* urlRequester = programSettings->programPathURLRequester( *it );
		if ( urlRequester != 0 )
			urlRequester->setURL( PluginsManager::programPath( *it ) );
	}

	profileSettings->defaultNamingSchemeLineEdit->setText( dirtyConfig.generalOption( GeneralOpt::naming_scheme ) );

	// Add profiles to selectedProfileComboBox
	profileSettings->selectedProfileComboBox->clear();
	QStringList profilesList = dirtyConfig.profiles();
	for ( QStringList::Iterator it = profilesList.begin(); it != profilesList.end(); ++it )
		profileSettings->selectedProfileComboBox->insertItem( *it );

	if ( profileSettings->selectedProfileComboBox->count() )
	{
		profileSettings->selectedProfileComboBox->setCurrentItem( 0 );
		changeSelectedProfile();
	}
}

void ConfigDialog::setControlsFromDefaults()
{
	int activeIdx = activePageIndex();
	if ( activeIdx == 0 )
	{
		setConfigFromControls();

		for ( int idx = 0; idx < GeneralOpt::SIZE; ++idx )
		{
			GeneralOpt::TGeneralOpt option = (GeneralOpt::TGeneralOpt)idx;
			if ( option == GeneralOpt::naming_scheme )
				continue;

			dirtyConfig.setGeneralOption( option, defaultConfig.generalOption( option ) );
		}
	}
	else if ( activeIdx == 1 )
	{
		setConfigFromControls();

		dirtyConfig.setGeneralOption( GeneralOpt::naming_scheme, defaultConfig.generalOption( GeneralOpt::naming_scheme ) );

		dirtyConfig.clearProfiles();

		QStringList profiles = defaultConfig.profiles();
		for ( QStringList::Iterator it = profiles.begin(); it != profiles.end(); ++it )
		{
			EncodingProfile* profileOpts = defaultConfig.profileOptions( *it );
			if ( ! profileOpts )
				continue;
			dirtyConfig.setEncodingProfile( *it, *profileOpts );
			delete profileOpts;
		}

		setProfileStatus();
	}
	else if ( activeIdx == 2 )
	{
		setConfigFromControls();

		QStringList programs = PluginsManager::programs();
		for ( QStringList::Iterator it = programs.begin(); it != programs.end(); ++it )
		{
			KURLRequester* urlRequester = programSettings->programPathURLRequester( *it );
			if ( urlRequester != 0 )
				urlRequester->setURL( PluginsManager::programPath( *it ) );
		}

		return;
	}
	else if ( activeIdx == 3 )
	{
		setConfigFromControls();

		for ( int idx = 0; idx < Metadata::SIZE; ++idx )
			dirtyConfig.setTagTransformation( (Metadata::Tag)idx, defaultConfig.tagTransformation( (Metadata::Tag)idx ) );
		for ( int idx = 0; idx < Metadata::SIZE; ++idx )
			dirtyConfig.setTagDefault( (Metadata::Tag)idx, defaultConfig.tagDefault( (Metadata::Tag)idx ) );
	}
	else
		return;

	setControlsFromConfig();
}

void ConfigDialog::setConfigFromControls()
{
	dirtyConfig.setGeneralOption( GeneralOpt::overwrite, generalSettings->overwriteCheckBox->isChecked() ? "true" : "false" );
	dirtyConfig.setGeneralOption(
		GeneralOpt::case_insensitive,
		generalSettings->caseInsensitiveCheckBox->isChecked() ?
			"true" :
			"false"
	);
	dirtyConfig.setGeneralOption(
		GeneralOpt::replace_chars,
		generalSettings->replaceCharsCheckBox->isChecked() ?
			dirtyConfig.generalOption( GeneralOpt::replace_chars ) :
			""
	);
	dirtyConfig.setGeneralOption( GeneralOpt::notify, generalSettings->notifyProgressCheckBox->isChecked() ? "true" : "false" );
	dirtyConfig.setGeneralOption( GeneralOpt::confirm, generalSettings->askConfirmationCheckBox->isChecked() ? "true":"false" );
	dirtyConfig.setGeneralOption( GeneralOpt::show_tray, generalSettings->showTrayCheckBox->isChecked() ? "true" : "false" );
	dirtyConfig.setGeneralOption( GeneralOpt::threads, QString::number( generalSettings->threadsSlider->value() ) );
	dirtyConfig.setGeneralOption( GeneralOpt::proc_priority, QString::number( generalSettings->prioritySlider->value() ) );
	dirtyConfig.setGeneralOption( GeneralOpt::rt_progress, generalSettings->rtProgressCheckBox->isChecked() ? "true":"false" );
	dirtyConfig.setGeneralOption(
		GeneralOpt::log,
		generalSettings->logEventsCheckBox->isChecked() ?
			generalSettings->logEventsURLRequester->url() :
			""
	);
	dirtyConfig.setGeneralOption( GeneralOpt::temp_dir, generalSettings->tempDirURLRequester->url() );


	dirtyConfig.setGeneralOption( GeneralOpt::naming_scheme, profileSettings->defaultNamingSchemeLineEdit->text() );

	dirtyConfig.setTagTransformation( Metadata::artist, (TextTrans::TTextTrans)tagSettings->artistComboBox->currentItem() );
	dirtyConfig.setTagTransformation( Metadata::album, (TextTrans::TTextTrans)tagSettings->albumComboBox->currentItem() );
	dirtyConfig.setTagTransformation( Metadata::title, (TextTrans::TTextTrans)tagSettings->titleComboBox->currentItem() );
	dirtyConfig.setTagTransformation( Metadata::genre, (TextTrans::TTextTrans)tagSettings->genreComboBox->currentItem() );
	dirtyConfig.setTagTransformation( Metadata::composer, (TextTrans::TTextTrans)tagSettings->composerComboBox->currentItem() );
	dirtyConfig.setTagTransformation( Metadata::comment, (TextTrans::TTextTrans)tagSettings->commentComboBox->currentItem() );

	dirtyConfig.setTagDefault( Metadata::artist, tagSettings->artistLineEdit->text() );
	dirtyConfig.setTagDefault( Metadata::year, tagSettings->yearLineEdit->text() );
	dirtyConfig.setTagDefault( Metadata::album, tagSettings->albumLineEdit->text() );
	dirtyConfig.setTagDefault( Metadata::discnum, tagSettings->discNumberLineEdit->text() );
	dirtyConfig.setTagDefault( Metadata::track, tagSettings->trackNumberLineEdit->text() );
	dirtyConfig.setTagDefault( Metadata::title, tagSettings->titleLineEdit->text() );
	dirtyConfig.setTagDefault( Metadata::genre, tagSettings->genreLineEdit->text() );
	dirtyConfig.setTagDefault( Metadata::composer, tagSettings->composerLineEdit->text() );
	dirtyConfig.setTagDefault( Metadata::comment, tagSettings->commentLineEdit->text() );

	QStringList programs = PluginsManager::programs();
	for ( QStringList::Iterator it = programs.begin(); it != programs.end(); ++it )
	{
		KURLRequester* urlRequester = programSettings->programPathURLRequester( *it );
		if ( urlRequester != 0 )
			dirtyConfig.setProgramPath( *it, urlRequester->url() );
	}
}


void ConfigDialog::acceptConfig()
{
	setConfigFromControls();
	acceptedConfig = dirtyConfig;

	changeSelectedProfile();
	enableButtonApply( false );

	emit configSaved();
}

void ConfigDialog::acceptConfigAndClose()
{
	acceptConfig();
	hide();
}

void ConfigDialog::rejectConfig()
{
	dirtyConfig = acceptedConfig;
	setControlsFromConfig();
	enableButtonApply( false );
}

void ConfigDialog::rejectConfigAndClose()
{
	hide();
	rejectConfig();
}

void ConfigDialog::addPage( QWidget* page, const QString& itemName, const QPixmap& pixmap, const QString& header )
{
	QVBox *frame = addVBoxPage( itemName, header, pixmap );
	frame->setSpacing( 0 );
	frame->setMargin( 0 );
	page->reparent( (QWidget*)frame, 0, QPoint() );
}

void ConfigDialog::addPage( QWidget* page, const QString& itemName, const QString& pixmapName, const QString& header )
{
	QVBox *frame = addVBoxPage( itemName, header, pixmapName );
	frame->setSpacing( 0 );
	frame->setMargin( 0 );
	page->reparent( (QWidget*)frame, 0, QPoint() );
}

void ConfigDialog::enableApply()
{
	enableButtonApply( true );
}
