/***************************************************************************
*   Copyright (C) 2005 by Sergio Pistone                                  *
*   sergio_pistone@yahoo.com.ar                                           *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
*                                                                         *
*   This program is distributed in the hope that it will be useful,       *
*   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
*   GNU General Public License for more details.                          *
*                                                                         *
*   You should have received a copy of the GNU General Public License     *
*   along with this program; if not, write to the                         *
*   Free Software Foundation, Inc.,                                       *
*   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
***************************************************************************/

#include "programsettings.h"
#include <pluginsmanager.h>

#include <qvariant.h>
#include <qtooltip.h>
#include <qwhatsthis.h>

#include <kdialog.h>
#include <klocale.h>
#include <kfiledialog.h>
#include <kiconloader.h>

ProgramSettings::ProgramSettings( QWidget* parent,  const char* name ):
	SettingsWidget( parent, name )
{
	setCaption( i18n( "Program Settings" ) );

	QStringList programs = TransKode::PluginsManager::programs();
	programs.sort();

	KIconLoader iconLoader;
	QPixmap foundImage( iconLoader.loadIcon( "ok", KIcon::Desktop, 16 ) );
	QPixmap missingImage( iconLoader.loadIcon( "error", KIcon::Desktop, 16 ) );

	QGridLayout* mainLayout = new QGridLayout( this, 1, 1 );
	mainLayout->setSpacing( 6 );
	mainLayout->setMargin( 11 );

	QGroupBox* groupBox = new QGroupBox( this, "groupBox1" );
	groupBox->setTitle( i18n( "Programs' Paths" ) );

	QGridLayout* groupBoxLayout = new QGridLayout( groupBox );
	groupBoxLayout->setSpacing( 6 );
	groupBoxLayout->setMargin( 11 );
 	groupBoxLayout->setAlignment( Qt::AlignBottom );

	groupBoxLayout->addItem( new QSpacerItem( 6, 6, QSizePolicy::Fixed, QSizePolicy::Fixed ), 0, 2 );

	QWidget* last = 0;
	int row = 1;
	for ( QStringList::ConstIterator it = programs.begin(); it != programs.end(); ++it, ++row )
	{
		QLabel* programLabel = new QLabel( groupBox );
		programLabel->setText( "<b>" + *it + "</b>" );
		groupBoxLayout->addWidget( programLabel, row, 0 );
		m_programLabels[*it] = programLabel;

		StatusLabel* statusLabel = new StatusLabel( groupBox, foundImage, missingImage );
		groupBoxLayout->addWidget( statusLabel, row, 1 );

		KURLRequester* programPathURLRequester = new KURLRequester( groupBox );
		programPathURLRequester->setCaption( i18n( "Find the path to %1" ).arg( *it ) );
		programPathURLRequester->setMode( KFile::File|KFile::LocalOnly );

		groupBoxLayout->addWidget( programPathURLRequester, row, 2 );
		connect( programPathURLRequester, SIGNAL( textChanged(const QString&) ), this, SLOT( emitSettingsChanged() ) );
		connect( programPathURLRequester, SIGNAL( textChanged(const QString&) ), statusLabel, SLOT(updateStatus(const QString&)) );
		m_programPathURLRequesters[*it] = programPathURLRequester;

		if ( last != 0 ) setTabOrder( last, programPathURLRequester );

		QPushButton* guessProgramPathButton = new QPushButton( groupBox );
		guessProgramPathButton->setSizePolicy( QSizePolicy( QSizePolicy::Fixed, QSizePolicy::Fixed, 0, 0, false ) );
		guessProgramPathButton->setText( i18n( "Find" ) );
		groupBoxLayout->addWidget( guessProgramPathButton, row, 3 );
		guessProgramPathButton->installEventFilter( this );
		m_guessProgramPathButtons[*it] = guessProgramPathButton;

		setTabOrder( programPathURLRequester, guessProgramPathButton );
		last = guessProgramPathButton;
	}

	mainLayout->addWidget( groupBox, 0, 0 );

	resize( QSize( 450, 590 ).expandedTo( minimumSizeHint() ) );
}

ProgramSettings::~ProgramSettings() {}

bool ProgramSettings::eventFilter( QObject* watched, QEvent* e )
{
	if ( e->type() == QEvent::MouseButtonRelease )
	{
		for ( QMap<QString,QPushButton*>::Iterator it = m_guessProgramPathButtons.begin(); it != m_guessProgramPathButtons.end(); ++it )
			if ( watched == it.data() )
				guessProgramPath( it.key() );
	}

	return SettingsWidget::eventFilter( watched, e );
}


void ProgramSettings::guessProgramPath( const QString& program )
{
	KURLRequester* urlRequester = programPathURLRequester( program );
	if ( urlRequester == 0 )
		return;

	QString programPath = KStandardDirs::findExe( program );
	if ( programPath != QString::null )
		urlRequester->setURL( programPath );
}

KURLRequester* ProgramSettings::programPathURLRequester( const QString& program )
{
	return m_programPathURLRequesters.contains( program ) ? m_programPathURLRequesters[program] : 0;
}

QPushButton* ProgramSettings::guessProgramPathButton( const QString& program )
{
	return m_guessProgramPathButtons.contains( program ) ? m_guessProgramPathButtons[program] : 0;
}

