/***************************************************************************
 *   Copyright (C) 2005 by Sergio Pistone                                  *
 *   sergio_pistone@yahoo.com.ar                                           *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "jobevents.h"
#include "filejob.h"
#include "worker.h"

#include <pluginsmanager.h>

#include <extratags.h>
#include <taglibplugins.h>
#include <taglib_encodingtypefiletyperesolver.h>

#include <logger.h>
#include <processhandler.h>
#include <commondefs.h>

#include <kapplication.h>
#include <kurl.h>
#include <klocale.h>

#include <qfileinfo.h>
#include <qdir.h>
#include <qregexp.h>

using namespace TransKode;

FileJob::FileJob( const AppConfig& cfg, const QString& srcPath, const QString& profile ):
	Job( cfg, profile ),
	m_srcPath( QDir::cleanDirPath( srcPath ) )
{
	readMetadata();

	setDstPath( generateDstPath() );
}

FileJob::~FileJob()
{
}

QString FileJob::srcPath() const
{
	QMutexLocker locker( &m_mutex );

	return String::deepCopy( m_srcPath );
}

// call this function with the object locked
void FileJob::evaluateExpressions( QString& dstPath )
{
	Job::evaluateExpressions( dstPath );

	QFileInfo srcInfo( m_srcPath );
	QString srcDir( escapeValue( srcInfo.dirPath(), false ) );
	QString srcName( escapeValue( srcInfo.baseName( true ) ) );
	QString srcExt( escapeValue( srcInfo.extension( false ) ) );

	dstPath.replace( "%{src_dir}",	srcDir );
	dstPath.replace( "%{src_name}",	srcName );
	dstPath.replace( "%{src_ext}",	srcExt );
	if ( m_options->bypass() )
		dstPath.replace( "%{dst_ext}", srcExt );
}

QString FileJob::fetchSourceFile()
{
	return srcPath();
}

void FileJob::removeFetchedSourceFile( const QString& )
{
}

void FileJob::readMetadata()
{
	EncodingTypeFileTypeResolver* fileTypeResolver = TagLibPlugins::encodingTypeFileTypeResolver();

	if ( fileTypeResolver != 0 )
	{
		fileTypeResolver->lock();
		fileTypeResolver->setEncodeType( QString::null );
	}

	TagLib::FileRef fileRef( m_srcPath.utf8() );

	if ( fileTypeResolver != 0 )
		fileTypeResolver->unlock();

	if ( ! fileRef.isNull() )
	{
		QMutexLocker locker( &m_mutex );

		m_srcMetadata[Metadata::artist]		= TStringToQString( fileRef.tag()->artist() );
		m_srcMetadata[Metadata::album]		= TStringToQString( fileRef.tag()->album() );
		m_srcMetadata[Metadata::title]		= TStringToQString( fileRef.tag()->title() );
		m_srcMetadata[Metadata::genre]		= TStringToQString( fileRef.tag()->genre() );
		m_srcMetadata[Metadata::comment]	= TStringToQString( fileRef.tag()->comment() );
		m_srcMetadata[Metadata::year]		= QString::number( fileRef.tag()->year() );
		m_srcMetadata[Metadata::track]		= QString::number( fileRef.tag()->track() );
		m_srcMetadata[Metadata::composer]	= TStringToQString( ExtraTags::getComposer( fileRef.file() ) );
		m_srcMetadata[Metadata::discnum]	= QString::number( ExtraTags::getDiscNumber( fileRef.file() ) );
	}
}

void FileJob::normalExit( const QString& dstPath )
{
	QString signalSrcPath = srcPath();

	Job::normalExit( dstPath );

	emit finishedOK( m_id, dstPath, signalSrcPath );
}

void FileJob::abnormalExit( const QString& errorMsg )
{
	QString signalSrcPath = srcPath();

	Job::abnormalExit( errorMsg );

	emit finishedWithError( m_id, errorMsg, signalSrcPath );
}

void FileJob::customEvent( QCustomEvent* e )
{
	switch ( e->type() )
	{
		case EVENT_JOB_STARTED:
			emit mtStarted( m_id, ((JobStartedEvent*)e)->m_dstPath, srcPath() );
			break;
		case EVENT_JOB_FINISHED_OK:
			emit mtFinishedOK( m_id, ((JobFinishedOKEvent*)e)->m_dstPath, srcPath() );
			break;
		case EVENT_JOB_FINISHED_WITH_ERROR:
			emit mtFinishedWithError( m_id, ((JobFinishedWithErrorEvent*)e)->m_errorMsg, srcPath() );
			break;
		default:
			break;
	}

	Job::customEvent( e );
}
