/***************************************************************************
 *   Copyright (C) 2005 by Sergio Pistone                                  *
 *   sergio_pistone@yahoo.com.ar                                           *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "job.h"
#include "jobevents.h"
#include "worker.h"

#include <extratags.h>
#include <taglibplugins.h>
#include <taglib_encodingtypefiletyperesolver.h>

#include <logger.h>
#include <processhandler.h>
#include <commondefs.h>

#include <kapplication.h>
#include <kurl.h>
#include <klocale.h>

#include <qfileinfo.h>
#include <qdir.h>
#include <qregexp.h>

using namespace TransKode;

// NOTE: if needed, ALWAYS lock the worker mutex before locking the job

Job::Job( const AppConfig& cfg, const QString& profile ):
	m_id( (long)this ),
	m_config( cfg ),
	m_logger( Logger::instance() ),
	m_mutex( false ),
	m_process( 0 ),
	m_paused( false ),
	m_pauseCondition(),
	m_stopped( false ),
	m_progress( -1 ),
	m_profile( String::deepCopy( profile ) ),
	m_dstPath(),
	m_srcMetadata(),
	m_userMetadata()
{
	m_options = m_config.profileOptions( profile );
	if ( ! m_config.profileOptionsAreValid( m_options ) )
	{
		if ( m_options != 0 )
			delete m_options;
		m_options = 0;
	}
}

Job::~Job()
{
	if ( m_options != 0 )
		delete m_options;

	System::delPotentialFile( m_dstPath );
}

long Job::id() const
{
	return m_id;
}

int Job::progress() const
{
	QMutexLocker locker( &m_mutex );

	return m_progress;
}

bool Job::isDecoding() const
{
	QMutexLocker locker( &m_mutex );

	return m_progress >= 0 && m_progress <= 49;
}

bool Job::isEncoding() const
{
	QMutexLocker locker( &m_mutex );

	return m_progress >= 50 && m_progress <= 99;
}

bool Job::isRunning() const
{
	QMutexLocker locker( &m_mutex );

	return m_progress >= 0 && m_progress <= 99;
}

void Job::clearFinished()
{
	QMutexLocker locker( &m_mutex );

	if ( m_progress == 100 /*finished ok*/ || m_progress == -1 /*finished with errors*/)
		m_progress = -1;
}

bool Job::isFinishedOK() const
{
	QMutexLocker locker( &m_mutex );

	return m_progress == 100;
}

bool Job::isFinishedWithError() const
{
	QMutexLocker locker( &m_mutex );

	return m_progress == -2;
}

bool Job::isCancelRequested() const
{
	QMutexLocker locker( &m_mutex );

	return m_stopped;
}

bool Job::hasProfileError() const
{
	QMutexLocker locker( &m_mutex );

	return m_options == 0;
}

QString Job::profile() const
{
	QMutexLocker locker( &m_mutex );

	return String::deepCopy( m_profile );
}

// returns true if the job was modified
bool Job::setProfile( const QString& profile )
{
	bool emitDstPathChanged = false;
	QString signalDstPath;

	{ // QMutexLocker scope start
		QMutexLocker locker( &m_mutex );

		if ( m_progress >= 0 && m_progress <= 99 ) // isRunning
			return false;

		m_progress = -1; // mark job as not started

		EncodingProfile* newOptions = m_config.profileOptions( profile );

		m_profile = String::deepCopy( profile );

		if ( ! m_config.profileOptionsAreValid( newOptions ) )
		{
			if ( newOptions != 0 )
				delete newOptions;
			m_options = 0;
			if ( ! m_dstPath.isEmpty() )
			{
				emitDstPathChanged = true;
				setDstPath( "" );
				signalDstPath = "";
			}
		}
		else
		{
			m_options = newOptions;
			QString newDstPath = generateDstPath();
			if ( m_dstPath != newDstPath )
			{
				emitDstPathChanged = true;
				setDstPath( newDstPath );
				signalDstPath = String::deepCopy( newDstPath );
			}
		}
	} // QMutexLocker scope end

	if ( emitDstPathChanged )
	{
		emit dstPathChanged( m_id, signalDstPath );
		delaySignal( new JobDstPathChangedEvent( signalDstPath ) );
	}

	return emitDstPathChanged;
}

bool Job::resetProfile() // same efect as setProfile( m_profile )
{
	bool emitDstPathChanged = false;
	QString signalDstPath;

	{ // QMutexLocker scope start


		if ( m_progress >= 0 && m_progress <= 99 ) // isRunning
			return false;

		m_progress = -1; // mark job as not started

		EncodingProfile* newOptions = m_config.profileOptions( m_profile );

		if ( ! m_config.profileOptionsAreValid( newOptions ) )
		{
			if ( newOptions != 0 )
				delete newOptions;
			m_options = 0;
			if ( ! m_dstPath.isEmpty() )
			{
				emitDstPathChanged = true;
				setDstPath( "" );
				signalDstPath = "";
			}
		}
		else
		{
			m_options = newOptions;
			QString newDstPath = generateDstPath();
			if ( m_dstPath != newDstPath )
			{
				emitDstPathChanged = true;
				setDstPath( newDstPath );
				signalDstPath = String::deepCopy( newDstPath );
			}
		}
	} // QMutexLocker scope end

	if ( emitDstPathChanged )
	{
		emit dstPathChanged( m_id, signalDstPath );
		delaySignal( new JobDstPathChangedEvent( signalDstPath ) );
	}

	return emitDstPathChanged;
}

// call this function with the object locked
Metadata Job::resolveVisibleMetadata() const
{
	Metadata md;

	for ( int idx = 0; idx < Metadata::SIZE; ++idx )
	{
		md[idx] = String::deepCopy(
			m_userMetadata[idx].isNull() ?
				String::evalTextTransformation( m_srcMetadata[idx], m_config.tagTransformation( (Metadata::Tag)idx ) ) :
				m_userMetadata[idx]
		);
	}

	return md;
}

// call this function with the object locked
Metadata Job::resolvePathGenerationMetadata() const
{
	Metadata md = resolveVisibleMetadata();

	for ( int idx = 0; idx < Metadata::SIZE; ++idx )
	{
		if ( md[idx].isEmpty() )
			md[idx] = String::evalTextTransformation(
				m_config.tagDefault( (Metadata::Tag)idx ),
				m_config.tagTransformation( (Metadata::Tag)idx )
			);
	}

	return md;
}

Metadata Job::metadata() const
{
	QMutexLocker locker( &m_mutex );

	return resolveVisibleMetadata();
}

// returns true if the job was modified
bool Job::setMetadata( const Metadata& tags, bool skipNull )
{
	bool emitDstPathChanged = false;
	bool tagsChanged = false;
	QString signalDstPath;

	{ // QMutexLocker scope start
		QMutexLocker locker( &m_mutex );

		if ( m_progress >= 0 && m_progress <= 99 ) // isRunning
			return false;

		m_progress = -1; // mark job as not started

		EncodingProfile* newOptions = m_config.profileOptions( m_profile );

		bool nowProfileValid = m_config.profileOptionsAreValid( newOptions );

		for ( int idx = 0; idx < Metadata::SIZE; ++idx )
		{
			QString tag( tags[idx] );

			if ( skipNull && tag.isNull() )
				continue;

			if ( m_userMetadata[idx] != tag )
			{
				m_userMetadata[idx] = String::deepCopy( tag );
				tagsChanged = true;
			}
		}

		if ( ! nowProfileValid )
		{
			if ( newOptions )
				delete newOptions;
			m_options = 0;
			if ( ! m_dstPath.isEmpty() )
			{
				emitDstPathChanged = true;
				setDstPath( "" );
				signalDstPath = "";
			}
		}
		else
		{
			m_options = newOptions;
			QString newDstPath = generateDstPath();
			if ( m_dstPath != newDstPath )
			{
				emitDstPathChanged = true;
				setDstPath( newDstPath );
				signalDstPath = String::deepCopy( newDstPath );
			}
		}
	} // QMutexLocker scope end

	if ( emitDstPathChanged )
	{
		emit dstPathChanged( m_id, signalDstPath );
		delaySignal( new JobDstPathChangedEvent( signalDstPath ) );
	}

	return emitDstPathChanged || tagsChanged;
}

QString Job::dstPath() const
{
	QMutexLocker locker( &m_mutex );

	return m_dstPath;
}

// method only used internally with the lock on m_mutex already acquired
void Job::setDstPath( const QString& dstPath )
{
	System::delPotentialFile( m_dstPath );
	System::addPotentialFile( dstPath );

	m_dstPath = dstPath;
}

void Job::stop()
{
	QMutexLocker locker( &m_mutex );

	if ( m_progress >= 0 && m_progress <= 99 ) // is in progress
	{
		if ( m_paused ) // resume if paused
		{
			m_paused = false;
			m_pauseCondition.wakeAll();
			if ( m_process )
				m_process->resume();
		}

		m_stopped = true;
		if ( m_process )
			m_process->terminate( 100 );
	}
}

void Job::pause()
{
	QMutexLocker locker( &m_mutex );

	if ( m_progress >= 0 && m_progress <= 99 ) // is in progress
	{
		m_paused = true;

		if ( m_process )
			m_process->pause();
	}
}

void Job::resume()
{
	QMutexLocker locker( &m_mutex );

	if ( m_progress >= 0 && m_progress <= 99 ) // is in progress
	{
		m_paused = false;
		m_pauseCondition.wakeAll();

		if ( m_process )
			m_process->resume();
	}
}

QString Job::escapeValue( const QString& value, bool escapeSlashes )
{
	QString ret( value );
	ret.replace( "}", "\\}" ).replace( "{", "\\{" );
	return escapeSlashes ? ret.replace( "/", "_" ) : ret;
}

QString Job::unescapeValue( const QString& value )
{
	return QString( value ).replace( "\\}", "}" ).replace( "\\{", "{" );
}

// call this function with the object locked
void Job::evaluateExpressions( QString& dstPath )
{
	if ( ! m_options->bypass() )
	{
		QString dstExt = m_options->extension();
		dstPath.replace( "%{dst_ext}", escapeValue( dstExt ) );
	}

	Metadata mdForPath = resolvePathGenerationMetadata();
	for ( int idx = 0; idx < Metadata::SIZE; ++idx )
		mdForPath[idx] = escapeValue( mdForPath[idx] );

	dstPath.replace( "%{artist}",	mdForPath[Metadata::artist] );
	dstPath.replace( "%{album}",	mdForPath[Metadata::album] );
	dstPath.replace( "%{year}",		mdForPath[Metadata::year] );
	dstPath.replace( "%{track#}",	mdForPath[Metadata::track] );
	dstPath.replace( "%{track##}",	String::padNumber( mdForPath[Metadata::track], 2 ) );
	dstPath.replace( "%{title}",	mdForPath[Metadata::title] );
	dstPath.replace( "%{comment}",	mdForPath[Metadata::comment] );
	dstPath.replace( "%{genre}",	mdForPath[Metadata::genre] );
	dstPath.replace( "%{composer}",	mdForPath[Metadata::composer] );
	dstPath.replace( "%{discnum}",	mdForPath[Metadata::discnum] );
}

// method only used internally with the lock on m_mutex already acquired
QString Job::generateDstPath()
{
	if ( m_options == 0 )
		return "";

	bool overwrite = m_config.generalOption( GeneralOpt::overwrite ) == "true";
	bool caseInsensitive = m_config.generalOption( GeneralOpt::case_insensitive ) == "true";
	QString replaceSpecialChars = m_config.generalOption( GeneralOpt::replace_chars );

	QString dstPath( m_options->namingScheme() );
	evaluateExpressions( dstPath );
	dstPath = String::evalTextTransformationExpression( dstPath );
	unescapeValue( dstPath );

	if ( ! replaceSpecialChars.isEmpty() )
		for ( unsigned int idx = 0; idx < replaceSpecialChars.length() ; ++idx )
			dstPath.replace( replaceSpecialChars[idx] , "_" );

	if ( dstPath.length() == 0 )
		return "";

	if ( dstPath[0] != '/' )
		dstPath = QDir::currentDirPath() + "/" + dstPath;

	if ( ! overwrite )
		return System::nonConflictivePath( dstPath, false, caseInsensitive, true, m_dstPath );
	else
	{
		// make sure not to collide with dirs or symlinks
		return System::nonConflictivePath( dstPath, true /*onlyDir*/, caseInsensitive, false /*potentialFiles*/, QString::null );
	}
}

void Job::writeMetadata( const QString& encodeType )
{
	EncodingTypeFileTypeResolver* fileTypeResolver = TagLibPlugins::encodingTypeFileTypeResolver();

	if ( fileTypeResolver != 0 )
	{
		fileTypeResolver->lock();
		// If outputPath doesn't have an extension taglib will fail at determining the type (since, by default,
		// it's all it checks). This can be solved in some (most?) cases 'cause the file type can be determined
		// by looking at the encoding method that was used to generate the file.
		fileTypeResolver->setEncodeType( encodeType );
	}

	// TODO should use local8Bit() or to utf8()?
	TagLib::FileRef fileRef( m_dstPath.local8Bit() );

	if ( fileTypeResolver != 0 )
	{
		fileTypeResolver->setEncodeType( QString::null );
		fileTypeResolver->unlock();
	}

	if ( ! fileRef.isNull() )
	{
		Metadata metadataForFile = metadata();

		fileRef.tag()->setArtist( QStringToTString( metadataForFile[Metadata::artist] ) );
		fileRef.tag()->setAlbum( QStringToTString( metadataForFile[Metadata::album] ) );
		fileRef.tag()->setTitle( QStringToTString( metadataForFile[Metadata::title] ) );
		fileRef.tag()->setGenre( QStringToTString( metadataForFile[Metadata::genre] ) );
		fileRef.tag()->setComment( QStringToTString( metadataForFile[Metadata::comment] ) );
		fileRef.tag()->setYear( metadataForFile[Metadata::year].toInt() );
		fileRef.tag()->setTrack( metadataForFile[Metadata::track].toInt() );
		ExtraTags::setComposer( fileRef.file(), QStringToTString( metadataForFile[Metadata::composer] ) );
		ExtraTags::setDiscNumber( fileRef.file(), metadataForFile[Metadata::discnum].toInt() );

		fileRef.save();
	}
}

void Job::setProgress( int prog )
{
	{ // QMutexLocker scope start
		QMutexLocker locker( &m_mutex );

		m_progress = prog;
	} // QMutexLocker scope end

	emit progress( m_id, prog );
	delaySignal( new JobProgressEvent( prog ) );
}

void Job::setDecodeProgress( int prog )
{
	setProgress( (int)(prog/2.0) );
}

void Job::setEncodeProgress( int prog )
{
	setProgress( 50 + (int)(prog/2.0) );
}

QString Job::getTempFilePath( const QString& extension, bool caseSensitive ) const
{
	return System::nonConflictivePath(
		QDir::cleanDirPath( m_config.generalOption( GeneralOpt::temp_dir ) + QDir::separator() +
			"transkodejob-" + QString::number( m_id ) + "." + extension ),
		false,
		! caseSensitive
	);
}

void Job::normalExit( const QString& dstPath )
{
	{ // QMutexLocker scope start
		QMutexLocker locker( &m_mutex );

		m_progress = 100; // mark job as finished ok
		m_stopped = false;
	} // QMutexLocker scope end

	emit progress( m_id, 100 );
	delaySignal( new JobProgressEvent( 100 ) );

	emit finishedOK( m_id, dstPath );
	delaySignal( new JobFinishedOKEvent( dstPath ) );
}

void Job::abnormalExit( const QString& errorMsg )
{
	{ // QMutexLocker scope start
		QMutexLocker locker( &m_mutex );

		m_progress = -2; // mark job as finished with error
		m_stopped = false;
	} // QMutexLocker scope end

	emit finishedWithError( m_id, errorMsg );
	delaySignal( new JobFinishedWithErrorEvent( errorMsg ) );
}

void Job::run()
{
	QString dstPath;

	{ // QMutexLocker scope start
		QMutexLocker locker( &m_mutex );

		m_progress = 0; // marks jobs as started
		dstPath = String::deepCopy( m_dstPath );
	} // QMutexLocker scope end

	emit started( m_id, dstPath );
	delaySignal( new JobStartedEvent( dstPath ) );

	emit progress( m_id, 0 );
	delaySignal( new JobProgressEvent( 0 ) );

	bool bpass = m_options->bypass();

	m_logger.log( bpass ? "bypass started" : "transcoding started" );
	m_logger.log( "output: " + dstPath );

	if ( m_options == 0 )
	{
		m_logger.log( "error: invalid profile" );
		abnormalExit( "invalid profile" );
		return;
	}

	// m_options can't be changed unless isRunning returns false so it's safe to use it

	QString errorMsg;
	if ( bpass ? bypass( dstPath, errorMsg ) : transcode( dstPath, errorMsg ) )
	{
		writeMetadata( m_options->encodeType() );

		m_logger.log( bpass ? "copy finished succesfully" : "transcoding finished succesfully" );

		normalExit( dstPath );
	}
	else
	{
		m_logger.log( bpass ? "copy finished abnormally" : "transcoding finished abnormally" );

		abnormalExit( errorMsg );
	}
}

bool Job::bypass( const QString& dstPath, QString& errorMsg )
{
	QString newDir = QFileInfo( dstPath ).dirPath();
	QStringList createdDirs;
	if ( ! System::recursiveMakeDir( newDir, &createdDirs ) )
	{
		for ( QStringList::Iterator it = createdDirs.begin(); it != createdDirs.end(); ++it )
			QDir().rmdir( *it );

		m_logger.log( i18n( "error: can't create output directory (%1)" ).arg( newDir ) );

		errorMsg = i18n( "error creating output directory" );

		return false;
	}

	QString srcPath = fetchSourceFile();
	if ( srcPath.isEmpty() )
	{
		for ( QStringList::Iterator it = createdDirs.begin(); it != createdDirs.end(); ++it )
			QDir().rmdir( *it );

		m_logger.log( i18n( "error: no source file received" ) );

		errorMsg = i18n( "error retrieving input file" );

		return false;
	}

	if ( ! System::copy( srcPath, dstPath ) )
	{
		removeFetchedSourceFile( srcPath );

		for ( QStringList::Iterator it = createdDirs.begin(); it != createdDirs.end(); ++it )
			QDir().rmdir( *it );

		m_logger.log( i18n( "error: can't copy file (%1 to %2)" ).arg( srcPath ).arg( dstPath ) );

		errorMsg = i18n( "error copying file" );

		return false;
	}

	removeFetchedSourceFile( srcPath );

	return true;
}

bool Job::transcode( const QString& dstPath, QString& errorMsg )
{
	bool caseInsensitive = m_config.generalOption( GeneralOpt::case_insensitive ) == "true";

	QString srcPath = fetchSourceFile();

	// the path where the decompressed input file will be temporarily stored
	QString tempPath = getTempFilePath( "wav", ! caseInsensitive );

	if ( decodeToWav( srcPath, tempPath ) )
	{
		removeFetchedSourceFile( srcPath );
		System::remove( tempPath );

		m_logger.log( i18n( "error: can't decode input" ) );

		errorMsg = i18n( "error decoding input" );

		return false;
	}

	removeFetchedSourceFile( srcPath );

	QString newDir = QFileInfo( dstPath ).dirPath();
	QStringList createdDirs;
	if ( ! System::recursiveMakeDir( newDir, &createdDirs ) )
	{
		System::remove( tempPath );

		for ( QStringList::Iterator it = createdDirs.begin(); it != createdDirs.end(); ++it )
			QDir().rmdir( *it );

		m_logger.log( i18n( "error: can't create output directory (%1)" ).arg( newDir ) );

		errorMsg = i18n( "error creating output directory" ).arg( newDir );

		return false;
	}

	setProgress( 50 );

	if ( encodeFromWav( tempPath, dstPath ) )
	{
		System::remove( tempPath );

		for ( QStringList::Iterator it = createdDirs.begin(); it != createdDirs.end(); ++it )
			QDir().rmdir( *it );

		m_logger.log( i18n( "error: can't encode file" ) );

		errorMsg = i18n( "error encoding file" );

		return false;
	}

	System::remove( tempPath );

	return true;
}

int Job::decodeToWav( const QString& srcPath, const QString& dstPath )
{
	m_logger.log( "decoding " + srcPath + " to " + dstPath );

	QString srcExt = QFileInfo( srcPath ).extension( false ).lower();

	int retCode = -1;
	bool normalExit = false;

	QStringList decoderNames = PluginsManager::decoderPluginNames();
	for ( QStringList::Iterator it = decoderNames.begin(), end = decoderNames.end(); it != end; ++it )
	{
		const DecoderPlugin* plugin = PluginsManager::decoderPlugin( *it );
		if ( plugin == 0 )
			continue;
		if ( ! plugin->canDecodeExtension( srcExt ) )
			continue;

		bool rtProgress = m_config.generalOption( GeneralOpt::rt_progress ) == "true";

		{ // QMutexLocker scope start
			QMutexLocker locker( &m_mutex );

			while ( m_paused )
				m_pauseCondition.wait( &m_mutex );

			m_process = new ProcessHandler(
				m_config.generalOption( GeneralOpt::proc_priority ).toInt(),
				rtProgress ? QProcess::Stdout|QProcess::Stderr|QProcess::DupStderr : 0
			);
		} // QMutexLocker scope end

		plugin->setupArgs( *m_process, srcPath, dstPath );

		OutputParserProxy* outputParserProxy = 0;
		if ( rtProgress )
		{
			ProgressParser* progressParser = plugin->newProgressParser();
			if ( progressParser )
			{
				// NOTE: progressParser will be deleted by the proxy
				outputParserProxy = new OutputParserProxy( m_process, progressParser );
				connect( progressParser, SIGNAL( progress(int) ), this, SLOT( setDecodeProgress(int) ) );
			}
		}

		retCode = -1;
		normalExit = false;

		m_logger.log( "process args: " + m_process->arguments().join( ":" ) );
		m_logger.log( "decoding process started" );

		if ( m_process->start() )
		{
			m_process->waitProcess(); // blocks until the process ends
			normalExit = m_process->normalExit();
			retCode = m_process->exitStatus();
		}

		if ( outputParserProxy != 0 )
			delete outputParserProxy;

		{ // QMutexLocker scope start
			QMutexLocker locker( &m_mutex );

			delete m_process;
			m_process = 0;
		} // QMutexLocker scope end

		m_logger.log( "process deleted" );

		bool error = false;
		if ( (error |= ! normalExit) )
			m_logger.log( QString( "decoding process finished abnormally" ) );
		else if ( (error |= retCode != 0) ) // only usefull if normalExit is true
			m_logger.log( QString( "decoding process exited with error (%1)" ).arg( retCode ) );
		else
		{
			QFileInfo fileInfo( dstPath );
			if ( (error |= ! fileInfo.exists()) )
				m_logger.log( "decoding process error (output file not found!)" );
			else if ( (error |= fileInfo.size() == 0) )
				m_logger.log( "decoding process error (empty output file found!)" );
		}

		if ( error )
		{
			if ( isCancelRequested() )
				return 1;
		}
		else
		{
			m_logger.log( "decoding process finished successfully" );
			return 0;
		}
	}

	return 1;
}

int Job::encodeFromWav( const QString& srcPath, const QString& dstPath )
{
	m_logger.log( "encoding " + srcPath + " to " + dstPath );

	QString switches = m_options->switches();

	const EncoderPlugin* plugin = PluginsManager::encoderPlugin( m_options->encodeType() );
	if ( plugin == 0 ? true : ! plugin->isLocallySupported() )
	{
		m_logger.log( "error: encoding type (" + m_options->encodeType() + ") not supported" );
		return -1;
	}

	bool caseInsensitive = m_config.generalOption( GeneralOpt::case_insensitive ) == "true";

	QString existingPath = System::conflictivePath( dstPath, caseInsensitive );
	QString backupPath = System::nonConflictivePath( existingPath + ".bak", false, caseInsensitive );
	if ( ! existingPath.isEmpty() ) // if dstPath file exist make a backup
	{
		m_logger.log( "backing up " + existingPath + " to " + backupPath );
		System::move( existingPath, backupPath );
	}

	Metadata metadataForFile = metadata();

	bool rtProgress = m_config.generalOption( GeneralOpt::rt_progress ) == "true";

	{ // QMutexLocker scope start
		QMutexLocker locker( &m_mutex );

		while ( m_paused )
			m_pauseCondition.wait( &m_mutex );

		m_process = new ProcessHandler(
			m_config.generalOption( GeneralOpt::proc_priority ).toInt(),
			rtProgress ? QProcess::Stdout|QProcess::Stderr|QProcess::DupStderr : 0
		);
	} // QMutexLocker scope end

	plugin->setupArgs( *m_process, srcPath, dstPath, switches, metadataForFile );

	OutputParserProxy* outputParserProxy = 0;
	if ( rtProgress )
	{
		ProgressParser* progressParser = plugin->newProgressParser();
		if ( progressParser )
		{
			// NOTE: progressParser will be deleted by the proxy
			outputParserProxy = new OutputParserProxy( m_process, progressParser );
			connect( progressParser, SIGNAL( progress(int) ), this, SLOT( setEncodeProgress(int) ) );
		}
	}

	m_logger.log( "process args: " + m_process->arguments().join( ":" ) );
	m_logger.log( "encoding process started" );

	int retCode = 1;
	bool normalExit = false;

	if ( m_process->start() )
	{
		m_process->waitProcess();
		normalExit = m_process->normalExit();
		retCode = m_process->exitStatus();
	}

	if ( outputParserProxy != 0 )
		delete outputParserProxy;

	{ // QMutexLocker scope start
		QMutexLocker locker( &m_mutex );

		delete m_process;
		m_process = 0;
	} // QMutexLocker scope end

	if ( ! normalExit || retCode || ! QFile::exists( dstPath ) )
	{
		if ( ! normalExit )
			m_logger.log( QString( "encoding process finished abnormally" ) );
		else if ( retCode )
			m_logger.log( QString( "encoding process exited with error (%1)" ).arg( retCode ) );
		else
			m_logger.log( "encoding process error (output file not found)" );

		System::remove( dstPath );
		if ( ! existingPath.isEmpty() )
		{
			m_logger.log( "restoring backup from " + backupPath + " to " + existingPath );
			System::move( backupPath, existingPath );
		}

		return 1;
	}
	else
	{
		m_logger.log( "encoding process finished successfully" );

		if ( ! existingPath.isEmpty() && m_config.generalOption( GeneralOpt::overwrite ) == "true" )
		{
			m_logger.log( "removing backup" );
			System::remove( backupPath );
		}

		return 0;
	}
}

void Job::delaySignal( QCustomEvent* event )
{
	kapp->postEvent( this, event );
}

void Job::customEvent( QCustomEvent* e )
{
	switch ( e->type() )
	{
		case EVENT_JOB_DST_PATH_CHANGED:
			emit mtDstPathChanged( m_id, ((JobDstPathChangedEvent*)e)->m_dstPath );
			break;
		case EVENT_JOB_STARTED:
 			emit mtStarted( m_id, ((JobStartedEvent*)e)->m_dstPath );
			break;
		case EVENT_JOB_PROGRESS:
			emit mtProgress( m_id, ((JobProgressEvent*)e)->m_progress );
			break;
		case EVENT_JOB_FINISHED_OK:
 			emit mtFinishedOK( m_id, ((JobFinishedOKEvent*)e)->m_dstPath );
			break;
		case EVENT_JOB_FINISHED_WITH_ERROR:
 			emit mtFinishedWithError( m_id, ((JobFinishedWithErrorEvent*)e)->m_errorMsg );
			break;
		default:
			QObject::event( e );
			break;
	}
}
