#ifndef TRANSKODE_JOB_H
#define TRANSKODE_JOB_H

/***************************************************************************
 *   Copyright (C) 2005 by Sergio Pistone                                  *
 *   sergio_pistone@yahoo.com.ar                                           *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
	#include <config.h>
#endif

#include <appconfig.h>
#include <logger.h>
#include <processhandler.h>
#include <progressparser.h>
#include <commondefs.h>

#include <qmutex.h>
#include <qwaitcondition.h>

namespace TransKode
{
	class JobQueue;
	class Worker;
	class EncodingTypeFileTypeResolver;

	class Job : public QObject
	{
		Q_OBJECT

		public:

			Job( const AppConfig& config, const QString& profile=QString::null );
			virtual ~Job();

			long id() const;

			int progress() const;

			void clearFinished();

			bool isDecoding() const;
			bool isEncoding() const;
			bool isRunning() const;

			bool isFinishedOK() const;
			bool isFinishedWithError() const;

			bool hasProfileError() const;
			QString profile() const;
			bool setProfile( const QString& profile );
			bool resetProfile();

			QString dstPath() const;

			Metadata metadata() const;
			bool setMetadata( const Metadata& tags, bool skipNull );

			void run();
			void stop();	// not guaranteed to success
			void pause();	// not guaranteed to success
			void resume();	// a paused job is guaranteed to resume

		signals:

			/** Warning! this signals MAY NOT be emitted from the main thread (see qt docs on threads) */
			void dstPathChanged( long jobID, const QString& dstPath );

			// processing state
			void started( long jobID, const QString& dstPath );
			void progress( long jobID, int progress );
			void finishedOK( long jobID, const QString& dstPath );
			void finishedWithError( long jobID, const QString& errorMsg );

			/** This signals ARE emitted from the main thread */
			void mtDstPathChanged( long jobID, const QString& dstPath );

			// processing state
			void mtStarted( long jobID, const QString& dstPath );
			void mtProgress( long jobID, int progress );
			void mtFinishedOK( long jobID, const QString& dstPath );
			void mtFinishedWithError( long jobID, const QString& errorMsg );

		protected:

			bool isCancelRequested() const;

			QString getTempFilePath( const QString& extension, bool caseSensitive ) const;

			// resolves the visible metadata from the file originated in the file and the one set manually
			Metadata resolveVisibleMetadata() const;
			// resolves the metadata for building path names
			Metadata resolvePathGenerationMetadata() const;

			// clean up everything after a successful transcoding / emits the appropriate signals
			// reimplement to emit signals customized for the specific job type
			virtual void normalExit( const QString& dstPath );

			// clean up everything after an unsuccessful transcoding / emits the appropriate signals
			// reimplement to emit signals customized for the specific job type
			virtual void abnormalExit( const QString& errorMsg );

			// copy he input to the output, return true if job completes normally (success) or false otherwise (error/cancel)
			bool bypass( const QString& dstPath, QString& errorMsg );
			// do the actual transcoding work, return true if job completes normally (success) or false otherwise (error/cancel)
			bool transcode( const QString& dstPath, QString& errorMsg );

			// get's the source path.
			// for a file job it just returns the pathname, but for CDs or URLs it must rip the cd track or stream first
			virtual QString fetchSourceFile() = 0;
			virtual void removeFetchedSourceFile( const QString& srcPath ) = 0;

			// do the decoding from the input source (specific to every job type)
			int decodeToWav( const QString& srcPath, const QString& dstPath );
			// do the encoding from the decoded source
			int encodeFromWav( const QString& srcPath, const QString& dstPath );


			QString escapeValue( const QString& value, bool escapeSlashes=true );
			QString unescapeValue( const QString& value );
			virtual void evaluateExpressions( QString& dstPath );
			QString generateDstPath(); // call this function with the object locked

			void setDstPath( const QString& dstPath );

			void writeMetadata( const QString& encodeType );

			void delaySignal( QCustomEvent* event );
			virtual void customEvent( QCustomEvent* );

		protected slots:

			void setProgress( int progress );
			void setEncodeProgress( int progress );
			void setDecodeProgress( int progress );

		protected:
			const long m_id;

			const AppConfig& m_config;
			Logger& m_logger;

			mutable QMutex m_mutex;

			ProcessHandler* m_process;

			bool m_paused;
			QWaitCondition m_pauseCondition;

			bool m_stopped;

			int m_progress;		// -1: not started
								// >=0: started
								// >=0 && <=99: transcoding
								// >=0 && <=49: decoding
								// >=50 && <=99: encoding
								// =100: done
								// -2: ended with errors


			QString m_profile;
			EncodingProfile* m_options;

			QString m_dstPath;

			Metadata m_srcMetadata;
			Metadata m_userMetadata;

			static EncodingTypeFileTypeResolver s_fileTypeResolver;
	};
}

#endif
