/***************************************************************************
 *   Copyright (C) 2005 by Sergio Pistone                                  *
 *   sergio_pistone@yahoo.com.ar                                           *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "jobqueue.h"
#include "jobqueueevents.h"
#include "worker.h"

#include <kapplication.h>

using namespace TransKode;

JobQueue::JobQueue():
	m_id( (long)this ),
	m_mutex( true ),
	m_paused( false ),
	m_jobsList(),
	m_workers(),
	m_workersIdle(),
	m_okCount( 0 ),
	m_errorCount( 0 ),
	m_cancelledCount( 0 )
{
	m_jobsList.setAutoDelete( false );	// jobs are managed by their creators...
	m_workers.setAutoDelete( true );	// ...but workers are managed by the queue
}

JobQueue::~JobQueue()
{
	kdDebug() << "stopping workers" << endl;

	for ( Worker* worker = m_workers.first(); worker != 0; worker = m_workers.next() )
		worker->stop();

	kdDebug() << "waiting workers to end" << endl;

	for ( Worker* worker = m_workers.first(); worker != 0; worker = m_workers.next() )
		worker->wait();

	kdDebug() << "workers finished" << endl;
}

long JobQueue::id() const
{
	return m_id;
}

unsigned JobQueue::jobsCount() const
{
	QMutexLocker locker( &m_mutex );

	return m_jobsList.count();
}

bool JobQueue::isPaused()
{
	QMutexLocker locker( &m_mutex );

	return m_paused;
}

void JobQueue::pause()
{
	QMutexLocker locker( &m_mutex );

	m_paused = true;

	for ( Worker* worker = m_workers.first(); worker != 0; worker = m_workers.next() )
		worker->pause();
}

void JobQueue::resume()
{
	QMutexLocker locker( &m_mutex );

	m_paused = false;

	for ( Worker* worker = m_workers.first(); worker != 0; worker = m_workers.next() )
		worker->resume();
}

unsigned JobQueue::workersCount() const
{
	QMutexLocker locker( &m_mutex );

	return m_workers.count();
}

bool JobQueue::allWorkersIdle() const
{
	QMutexLocker locker( &m_mutex );

	for ( QMap<long,bool>::ConstIterator it = m_workersIdle.begin(); it != m_workersIdle.end(); ++it )
		if ( ! it.data() ) // there's a thread working
			return false;
	return true;
}

bool JobQueue::allWorkersWorking() const
{
	QMutexLocker locker( &m_mutex );

	for ( QMap<long,bool>::ConstIterator it = m_workersIdle.begin(); it != m_workersIdle.end(); ++it )
		if ( it.data() ) // there's a thread idle
			return false;
	return true;
}

void JobQueue::enqueue( Job* job )
{
	if ( job == 0 )
		return;

	m_mutex.lock();

	bool emitQueueStarted = m_jobsList.isEmpty();

	m_jobsList.append( job );

	m_mutex.unlock();

	if ( emitQueueStarted )
	{
		emit queueStarted( m_id, job->id() );
		delaySignal( new QueueJobStartedEvent( job->id() ) );
	}

	emit enqueued( m_id, job->id() );
	delaySignal( new EnqueuedEvent( job->id() ) );
}

Job* JobQueue::dequeue()
{
	m_mutex.lock();

	Job* job = m_jobsList.getFirst();
	if ( job != 0 ) // there was a job left in the queue
	{
		m_jobsList.removeFirst();

		connect( job, SIGNAL( finishedOK(long,const QString&,const QString&) ),
				 this, SLOT( setJobFinishedOK(long) ) );

		connect( job, SIGNAL( finishedWithError(long,const QString&,const QString&) ),
				 this, SLOT( setJobFinishedWithError(long) ) );

		bool emitQueueDepleted = m_jobsList.isEmpty();

		m_mutex.unlock();

		if ( emitQueueDepleted )
		{
			emit queueDepleted( m_id, job->id() );
			delaySignal( new QueueDepletedEvent( job->id() ) );
		}

		emit dequeued( m_id, job->id() );
		delaySignal( new DequeuedEvent( job->id() ) );
	}
	else
		m_mutex.unlock();

	return job;
}

bool JobQueue::remove( Job* job )
{
	QMutexLocker locker( &m_mutex );

	bool ret = m_jobsList.removeRef( job );
	if ( ret )
		m_cancelledCount++;

	return ret;
}

bool JobQueue::remove( long jobID )
{
	QMutexLocker locker( &m_mutex );

	for ( Job* job = m_jobsList.first(); job != 0; job = m_jobsList.next() )
	{
		if ( job->id() == jobID )
		{
			m_cancelledCount++;
			return m_jobsList.remove();
		}
	}

	return false;
}

void JobQueue::removeAll()
{
	QMutexLocker locker( &m_mutex );

	m_cancelledCount += m_jobsList.count();

	m_jobsList.clear();
}

void JobQueue::registerWorker( Worker* worker )
{
	QMutexLocker locker( &m_mutex );

	m_workersIdle[worker->id()] = m_jobsList.count() == 0;
	m_workers.append( worker );

	worker->start();
}

void JobQueue::setWorkerIdle( long workerID )
{
	m_mutex.lock();

	bool allWorkersWereIdle = allWorkersIdle();

	m_workersIdle[workerID] = true;

	bool allWorkersAreIdle = allWorkersIdle();

	// only emit signal if state has _changed_ to allWorkersIdle:
	bool emitWorkersIdle = allWorkersAreIdle && ! allWorkersWereIdle;

	WorkersIdleEvent* event = 0;

	if ( emitWorkersIdle )
	{
		event = new WorkersIdleEvent( m_okCount, m_errorCount, m_cancelledCount );

		m_okCount = 0;
		m_errorCount = 0;
		m_cancelledCount = 0;
	}

	m_mutex.unlock();

	if ( emitWorkersIdle )
	{
		emit workersIdle( m_id );
		emit workersIdle( m_id, event->m_okCount, event->m_errorCount, event->m_cancelledCount );
		delaySignal( event );
	}
}

void JobQueue::setWorkerBusy( long workerID )
{
	m_mutex.lock();

	// allWorkerWereIdle implies a change of state (because at least workerID is now busy)
	bool emitWorkersBusy = allWorkersIdle();

	m_workersIdle[workerID] = false;

	m_mutex.unlock();

	if ( emitWorkersBusy )
	{
		emit workersBusy( m_id );
		delaySignal( new WorkersBusyEvent() );
	}
}

void JobQueue::setJobFinishedOK( long jobID )
{
	QMutexLocker locker( &m_mutex );

	m_okCount++;

	disconnect( (Job*)jobID, 0, this, 0 );
}

void JobQueue::setJobFinishedWithError( long jobID )
{
	QMutexLocker locker( &m_mutex );

	m_errorCount++;

	disconnect( (Job*)jobID, 0, this, 0 );
}

void JobQueue::delaySignal( QCustomEvent* event )
{
	kapp->postEvent( this, event );
}

void JobQueue::customEvent( QCustomEvent* e )
{
	switch ( e->type() )
	{
		case ENQUEUEDEVENT:
			emit mtEnqueued( m_id, ((EnqueuedEvent*)e)->m_id );
			break;
		case DEQUEUEDEVENT:
			emit mtDequeued( m_id, ((DequeuedEvent*)e)->m_id );
			break;
		case QUEUEEVENT_JOB_STARTED:
			emit mtQueueStarted( m_id, ((QueueJobStartedEvent*)e)->m_id );
			break;
		case QUEUEDEPLETEDEVENT:
			emit mtQueueDepleted( m_id, ((QueueDepletedEvent*)e)->m_id );
			break;
		case WORKERSBUSYEVENT:
			emit mtWorkersBusy( m_id );
			break;
		case WORKERSIDLEEVENT:
		{
			WorkersIdleEvent* wie = (WorkersIdleEvent*)e;
			emit mtWorkersIdle( m_id );
			emit mtWorkersIdle( m_id, wie->m_okCount, wie->m_errorCount, wie->m_cancelledCount );
			break;
		}
		default:
			QObject::event( e );
			break;
	}
}
