#ifndef TRANSKODE_JOBQUEUE_H
#define TRANSKODE_JOBQUEUE_H

/***************************************************************************
 *   Copyright (C) 2005 by Sergio Pistone                                  *
 *   sergio_pistone@yahoo.com.ar                                           *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
	#include <config.h>
#endif

#include "job.h"

#include <qobject.h>
#include <qmutex.h>
#include <qptrlist.h>
#include <qstringlist.h>

namespace TransKode
{
	class JobQueue : public QObject
	{
		Q_OBJECT

		friend class Worker;

		public:

			JobQueue();
			virtual ~JobQueue();

			long id() const;

			unsigned jobsCount() const;

			bool isPaused();
			void pause();
			void resume();

			unsigned workersCount() const;
			bool allWorkersIdle() const;
			bool allWorkersWorking() const;

			void enqueue( Job* job );

			bool remove( Job* job );
			bool remove( long jobID );
			void removeAll();

		signals:

			/**
			* Warning! this signals MAY NOT be emitted from the main thread (check qt docs on threads)
			*/
			void enqueued( long queueID, long jobID );
			void dequeued( long queueID, long jobID );

			void queueStarted( long queueID, long jobID );
			void queueDepleted( long queueID, long jobID );

			void workersBusy( long queueID );
			void workersIdle( long queueID );
			void workersIdle( long queueID, int okCount, int errorCount, int cancelledCount );

			/**
			* This signals ARE emitted from the main thread
			*/
			void mtEnqueued( long queueID, long jobID );
			void mtDequeued( long queueID, long jobID );

			void mtQueueStarted( long queueID, long jobID );
			void mtQueueDepleted( long queueID, long jobID );

			void mtWorkersBusy( long queueID );
			void mtWorkersIdle( long queueID );
			void mtWorkersIdle( long queueID, int okCount, int errorCount, int cancelledCount );

		protected:

			Job* dequeue(); // used for workers to get next job

			void registerWorker( Worker* worker );

			void delaySignal( QCustomEvent* event );
			virtual void customEvent( QCustomEvent* );

		protected slots:

			void setWorkerIdle( long workerID );
			void setWorkerBusy( long workerID );

			void setJobFinishedOK( long jobID );
			void setJobFinishedWithError( long jobID );

		protected:

			const long m_id;

			mutable QMutex m_mutex;

			bool m_paused;

			QPtrList<Job> m_jobsList;

			QPtrList<Worker> m_workers;
			QMap<long,bool> m_workersIdle;

			int m_okCount;
			int m_errorCount;
			int m_cancelledCount;
	};
}

#endif
