/***************************************************************************
 *   Copyright (C) 2005 by Sergio Pistone                                  *
 *   sergio_pistone@yahoo.com.ar                                           *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "worker.h"

#include <kapplication.h>

using namespace TransKode;

Worker::Worker( JobQueue& jobsQueue ):
	QThread(),
	m_id( (long)this ),
	m_jobsQueue( jobsQueue ),
	m_mutex( false ),
	m_stopped( false ),
	m_paused( false ),
	m_pauseCondition(),
	m_currentJob( 0 )
{
	jobsQueue.registerWorker( this );
}

Worker::~Worker()
{
	stop();
	wait();
}

long Worker::id() const
{
	return m_id;
}

bool Worker::isPaused() const
{
	QMutexLocker locker( &m_mutex );

	return m_paused;
}

void Worker::pause()
{
	QMutexLocker locker( &m_mutex );

	if ( ! m_paused )
	{
		kdDebug() << "worker " << m_id << " paused" << endl;

		m_paused = true;

		if ( m_currentJob )
			m_currentJob->pause(); // may or may not pause the subprocess
	}
}

void Worker::resume()
{
	QMutexLocker locker( &m_mutex );

	if ( m_paused )
	{
		kdDebug() << "worker " << m_id << " resumed" << endl;

		m_paused = false;
		m_pauseCondition.wakeAll(); // wake worker thread if it was sleeping

		if ( m_currentJob )
			m_currentJob->resume(); // will resume the subprocess if it was paused
	}
}

bool Worker::isStopped() const
{
	QMutexLocker locker( &m_mutex );

	return m_stopped;
}

void Worker::stop()
{
	QMutexLocker locker( &m_mutex );

	if ( m_paused ) // resume thread if paused
	{
		m_paused = false;
		m_pauseCondition.wakeAll();
	}

	m_stopped = true;

	if ( m_currentJob )
		m_currentJob->stop();
}

bool Worker::isProcessing() const
{
	QMutexLocker locker( &m_mutex );

	return m_currentJob != 0;
}

void Worker::run()
{
	kdDebug() << "worker " << m_id << " running" << endl;

	bool isIdle = true;

	while ( ! isStopped() )
	{
		m_mutex.lock();

		for ( ; ; )
		{
			while ( m_paused && ! m_stopped )
				m_pauseCondition.wait( &m_mutex );

			m_currentJob = m_stopped ? 0 : m_jobsQueue.dequeue();

			if ( ! m_currentJob )
				break;

			m_mutex.unlock();

			if ( isIdle )
			{
				isIdle = false;
				m_jobsQueue.setWorkerBusy( m_id );
			}

			m_currentJob->run();

			m_mutex.lock();
		}

		m_mutex.unlock();

		isIdle = true;
		m_jobsQueue.setWorkerIdle( m_id );

		QThread::sleep( 2 ); // avoid busy wait when the queue is empty
	}

	kdDebug() << "worker " << m_id << " finished" << endl;
}
