/***************************************************************************
 *   Copyright (C) 2007 by Pete Chapman                                    *
 *   tot5tri02@sneakemail.com                                              *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "extratags.h"

#include <qptrlist.h>
#include <qstring.h>

#include <taglib/textidentificationframe.h>

#include <taglib/apetag.h>
#include <taglib/id3v2tag.h>
#include <taglib/xiphcomment.h>
#include <wma/wmatag.h>

#include <taglib/flacfile.h>
#include <taglib/mpegfile.h>
#include <taglib/oggflacfile.h>
#include <taglib/vorbisfile.h>
#include <wma/wmafile.h>

namespace
{
	typedef TagLib::String TString;

	class TaggerBase
	{
		public:
			virtual ~TaggerBase() {}
			virtual TString getComposer() = 0;
			virtual TString getDiscNumber() = 0;
			virtual void setComposer( const TString& ) = 0;
			virtual void setDiscNumber( const TString& ) = 0;
	};

	class ID3v2Tagger : public TaggerBase
	{
		public:
			ID3v2Tagger( TagLib::ID3v2::Tag* tag ) : m_tag( tag ) {}

			virtual TString getComposer()   { return getID3v2TextFrame( "TCOM" ); }
			virtual TString getDiscNumber() { return getID3v2TextFrame( "TPOS" ); }
			virtual void setComposer( const TString& composer )     { setID3v2TextFrame( "TCOM", composer ); }
			virtual void setDiscNumber( const TString& discNumber ) { setID3v2TextFrame( "TPOS", discNumber ); }

		private:
			TString getID3v2TextFrame( const char* type )
			{
				const TagLib::ID3v2::FrameList& frames = m_tag->frameList( type );
				const TagLib::ID3v2::TextIdentificationFrame* tid;
				if ( ! frames.isEmpty() &&
					 (tid = dynamic_cast<const TagLib::ID3v2::TextIdentificationFrame*>(frames.front())) )
					return tid->toString();
				return TString();
			}

			void setID3v2TextFrame( const char* type, const TString& value )
			{
				m_tag->removeFrames( type );
				if ( ! value.isEmpty() )
				{
					TagLib::ID3v2::TextIdentificationFrame* tid =
						new TagLib::ID3v2::TextIdentificationFrame( type, TagLib::String::UTF8 );
					tid->setText( value );
					m_tag->addFrame( tid );
				}
			}

			TagLib::ID3v2::Tag* m_tag;
	};

	class XiphTagger : public TaggerBase
	{
		public:
			XiphTagger( TagLib::Ogg::XiphComment* tag ) : m_tag( tag ) {}

			virtual void setComposer( const TString& composer )     { setXiphField( QString( "COMPOSER" ), composer ); }
			virtual void setDiscNumber( const TString& discNumber ) { setXiphField( QString( "DISCNUMBER" ), discNumber ); }
			virtual TString getComposer()   { return getXiphField( QString( "COMPOSER" ) ); }
			virtual TString getDiscNumber() { return getXiphField( QString( "DISCNUMBER" ) ); }

		private:
			TString getXiphField( const TString& type )
			{
				const TagLib::Ogg::FieldListMap& flmap = m_tag->fieldListMap();
				TagLib::Ogg::FieldListMap::ConstIterator it = flmap.find( type );
				if ( it != flmap.end() && ! it->second.isEmpty() )
					return it->second.front();
				return TString();
			}

			TString getXiphField( const QString& type )
			{
				TString value;
				if ( ! (value = getXiphField( QStringToTString( type ) )).isEmpty() )
					return value;
				if ( ! (value = getXiphField( QStringToTString( type.upper() ) )).isEmpty() )
					return value;
				return getXiphField( QStringToTString( type.lower() ) );
			}

			void setXiphField( const QString& type, const TString& value )
			{
				// remove upper and lower case versions of the same key
				m_tag->removeField( QStringToTString( type.lower() ) );
				m_tag->removeField( QStringToTString( type.upper() ) );
				if ( ! value.isEmpty() )
					m_tag->addField( QStringToTString( type ), value, true );
			}

			TagLib::Ogg::XiphComment* m_tag;
	};


	class APETagger : public TaggerBase
	{
		public:
			APETagger( TagLib::APE::Tag* tag ) : m_tag( tag ) {}

			virtual void setComposer( const TString& composer )     { setAPEField( QString( "COMPOSER" ), composer ); }
			virtual void setDiscNumber( const TString& discNumber ) { setAPEField( QString( "DISCNUMBER" ), discNumber ); }
			virtual TString getComposer()   { return getAPEField( QString( "COMPOSER" ) ); }
			virtual TString getDiscNumber() { return getAPEField( QString( "DISCNUMBER" ) ); }

		private:
			TString getAPEField( const TString& type )
			{
				const TagLib::APE::ItemListMap& itmap = m_tag->itemListMap();
				TagLib::APE::ItemListMap::ConstIterator it = itmap.find( type );
				if ( it != itmap.end() && it->second.type() == TagLib::APE::Item::Text )
					return it->second.toString();
				return TString();
			}

			TString getAPEField( const QString& type )
			{
				TString value;
				if ( ! (value = getAPEField( QStringToTString( type ) )).isEmpty() )
					return value;
				if ( ! (value = getAPEField( QStringToTString( type.upper() ) )).isEmpty() )
					return value;
				return getAPEField( QStringToTString( type.lower() ) );
			}

			void setAPEField( const QString& type, const TString& value )
			{
				// remove upper and lower case versions of the same key
				m_tag->removeItem( QStringToTString( type.lower() ) );
				m_tag->removeItem( QStringToTString( type.upper() ) );
				if ( ! value.isEmpty() )
				{
					TagLib::APE::Item item( QStringToTString( type ), value );
					item.setType( TagLib::APE::Item::Text );
					m_tag->setItem( item.key(), item );
				}
			}

			TagLib::APE::Tag* m_tag;
	};

	class WMATagger : public TaggerBase
	{
		public:
			WMATagger( TagLib::WMA::Tag* tag ) : m_tag( tag ) {}

			virtual void setComposer( const TString& composer )     { setWMAField( "WM/Composer", composer ); }
			virtual void setDiscNumber( const TString& discNumber ) { setWMAField( "WM/DiscNumber", discNumber ); }
			virtual TString getComposer()   { return getWMAField( "WM/Composer" ); }
			virtual TString getDiscNumber() { return getWMAField( "WM/DiscNumber" ); }

		private:
			TString getWMAField( const TagLib::ByteVector& type )
			{
				const TagLib::WMA::AttributeMap attmap = m_tag->attributeMap();
				TagLib::WMA::AttributeMap::ConstIterator it = attmap.find( type );
				if ( it != attmap.end() && it->second.type() == TagLib::WMA::Attribute::UnicodeType )
					return it->second.toString();
				return TString();
			}

			void setWMAField( const TagLib::ByteVector& type, const TString& value )
			{
				m_tag->removeItem( type );
				if ( ! value.isEmpty() )
					m_tag->setAttribute( type, TagLib::WMA::Attribute( type, value ) );
			}

			TagLib::WMA::Tag* m_tag;
	};



	typedef QPtrList<TaggerBase> TaggerList;

	// Some concrete instances of TagLib::File can store more than one type of
	// TagLib::Tag. We need to be able to set and query them all, not just the
	// one returned by File::tag().  Additionally, the undocumented
	// TagLib::MPEGTag and TagLib::FLAC::Tag objects returned by their respective
	// tag() methods do not support the XiphComment or ID3v2 interfaces necessary
	// to access the Composer and Disc Number fields.
	void TaggersFromFile( TagLib::File* file, TaggerList& taggers )
	{
		taggers.clear();
		taggers.setAutoDelete( true );

		// NOTE: no point in handling non extensible tag types which don't support disc number nor composer:
		// ID3v1, RealMediaTag, AudibleTag

		// deal first with file types that can have multiple kinds of tags and for wich we support more than one
		if ( TagLib::FLAC::File* flacFile = dynamic_cast<TagLib::FLAC::File*>(file) )
		{
			if ( flacFile->xiphComment() )
				taggers.append( new XiphTagger( flacFile->xiphComment() ) );
			if ( flacFile->ID3v2Tag() )
				taggers.append( new ID3v2Tagger( flacFile->ID3v2Tag() ) );
		}
		else if ( TagLib::MPEG::File* mpegFile = dynamic_cast<TagLib::MPEG::File*>(file) )
		{
			if ( mpegFile->ID3v2Tag() )
				taggers.append( new ID3v2Tagger( mpegFile->ID3v2Tag() ) );
			if ( mpegFile->APETag() )
				taggers.append( new APETagger( mpegFile->APETag() ) );
		}
		else // now deal with files for which we support only one tag type
		{
			if ( ! file->tag() )
				return;

			if ( TagLib::ID3v2::Tag* id3v2Tag = dynamic_cast<TagLib::ID3v2::Tag*>(file->tag()) )
			{
				taggers.append( new ID3v2Tagger( id3v2Tag ) );
			}
			else if ( TagLib::Ogg::XiphComment* xiphComment = dynamic_cast<TagLib::Ogg::XiphComment*>(file->tag()) )
			{
				taggers.append( new XiphTagger( xiphComment ) );
			}
			else if ( TagLib::APE::Tag* APETag = dynamic_cast<TagLib::APE::Tag*>(file->tag()) )
			{
				taggers.append( new APETagger( APETag ) );
			}
			else if ( TagLib::WMA::Tag* WMATag = dynamic_cast<TagLib::WMA::Tag*>(file->tag()) )
			{
				taggers.append( new WMATagger( WMATag ) );
			}
		}
	}
}

namespace ExtraTags
{
	TString getComposer( TagLib::File* file )
	{
		TaggerList taggers;
		TaggersFromFile( file, taggers );
		TString composer;
		for ( taggers.first(); taggers.current() != 0 && composer.isEmpty(); taggers.next() )
			composer = taggers.current()->getComposer();
		return composer;
	}

	void setComposer( TagLib::File* file, const TString& composer )
	{
		TaggerList taggers;
		TaggersFromFile( file, taggers );
		for ( taggers.first(); taggers.current() != 0; taggers.next() )
			taggers.current()->setComposer( composer );
	}

	TagLib::uint getDiscNumber( TagLib::File* file )
	{
		TaggerList taggers;
		TaggersFromFile( file, taggers );
		TString discNumber;
		for ( taggers.first(); taggers.current() != 0 && discNumber.isEmpty(); taggers.next() )
			discNumber = taggers.current()->getDiscNumber();

		int ret = discNumber.toInt();
		return ret > 0 ? ret : 0;
	}

	void setDiscNumber( TagLib::File* file, const TagLib::uint discNumber )
	{
		TaggerList taggers;
		TaggersFromFile( file, taggers );
		for ( taggers.first(); taggers.current() != 0; taggers.next() )
			taggers.current()->setDiscNumber( discNumber > 0 ? TString::number( discNumber ) : TString() );
	}
}
