/***************************************************************************
 *   Copyright (C) 2005 by Sergio Pistone                                  *
 *   sergio_pistone@yahoo.com.ar                                           *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "flacencoderdialog.h"
#include <commondefs.h>

#include <qregexp.h>
#include <qcheckbox.h>
#include <qradiobutton.h>
#include <qbuttongroup.h>
#include <klineedit.h>
#include <kcombobox.h>
#include <kdecspinbox.h>

FLACEncoderDialog::FLACEncoderDialog( QWidget* parent, const char* name ):
	FLACEncoderDialogBase( parent, name ) {}

FLACEncoderDialog::~FLACEncoderDialog() {}

void FLACEncoderDialog::setSwitches( const QString& switches )
{
	QRegExp regExp;

	regExp.setPattern( "(^|[ \\t])(-|--compression-level-)([0-8])([ \\t]|$)" );
	if ( regExp.search( switches ) != -1 )
	{
		profileCheckBox->setChecked( true );
		profileComboBox->setCurrentItem( regExp.cap( 3 ).toInt() );
	}
	else
	{
		regExp.setPattern( "(^|[ \\t])--(fast|best)([ \\t]|$)" );
		if ( regExp.search( switches ) != -1 )
		{
			profileCheckBox->setChecked( true );
			if ( regExp.cap( 2 ) == "fast" )
				profileComboBox->setCurrentItem( 0 );
			else // best
				profileComboBox->setCurrentItem( 8 );
		}
	}

	regExp.setPattern( "(^|[ \\t])(-e|--exhaustive-model-search)([ \\t]|$)" );
	if ( regExp.search( switches ) != -1 )
		exhaustiveModelSearchCheckBox->setChecked( true );

	regExp.setPattern( "(^|[ \\t])(-l[ \\t]+|--max-lpc-order=)([0-9]+)([ \\t]|$)" );
	if ( regExp.search( switches ) != -1 )
	{
		maxLPCOrderCheckBox->setChecked( true );
		maxLPCOrderSpinBox->setValue( regExp.cap( 3 ).toInt() );
	}

	regExp.setPattern( "(^|[ \\t])(-q[ \\t]+|--qlp-coeff-precision=)([0-5])([ \\t]|$)" );
	if ( regExp.search( switches ) != -1 )
	{
		quantizedLPCPrecisionCheckBox->setChecked( true );
		quantizedLPCPrecisionComboBox->setCurrentItem( regExp.cap( 3 ).toInt()+1 );
	}
	else
	{
		regExp.setPattern( "(^|[ \\t])(-p|--qlp-coeff-precision-search)([ \\t]|$)" );
		if ( regExp.search( switches ) != -1 )
		{
			quantizedLPCPrecisionCheckBox->setChecked( true );
			quantizedLPCPrecisionComboBox->setCurrentItem( 0 );
		}
	}

	regExp.setPattern( "(^|[ \\t])(-r[ \\t]+|--rice-partition-order=)([0-9]+,|)([0-9]+)([ \\t]|$)" );
	if ( regExp.search( switches ) != -1 )
	{
		maxResidualPOCheckBox->setChecked( true );
		maxResidualPOSpinBox->setValue( regExp.cap( 4 ).toInt() );
		if ( ! regExp.cap( 3 ).isEmpty() )
		{
			minResidualPOCheckBox->setChecked( true );
			minResidualPOSpinBox->setValue( regExp.cap( 3 ).left( regExp.cap( 3 ).length()-1 ).toInt() );
		}
	}

	regExp.setPattern( "(^|[ \\t])--lax([ \\t]|$)" );
	if ( regExp.search( switches ) != -1 )
	{
		regExp.setPattern( "(^|[ \\t])(b[ \\t]+|--blocksize=)([0-9]+)([ \\t]|$)" );
		if ( regExp.search( switches ) != -1 )
		{
			blockSizeCheckBox->setChecked( true );
			laxBlockSizeRadioButton->setChecked( true );
			laxBlockSizeSpinBox->setValue( regExp.cap( 3 ).toInt() );
		}
	}
	else
	{
		regExp.setPattern( "(^|[ \\t])(b[ \\t]+|--blocksize=)(192|576|1152|2304|4608|256|512|1024|2048|4096|8192|16384|32768)([ \\t]|$)" );
		if ( regExp.search( switches ) != -1 )
		{
			blockSizeCheckBox->setChecked( true );
			strictBlockSizeRadioButton->setChecked( true );
			switch ( regExp.cap( 3 ).toInt() )
			{
				case 192:	strictBlockSizeComboBox->setCurrentItem( 0 );	break;
				case 576:	strictBlockSizeComboBox->setCurrentItem( 1 );	break;
				case 1152:	strictBlockSizeComboBox->setCurrentItem( 2 );	break;
				case 2304:	strictBlockSizeComboBox->setCurrentItem( 3 );	break;
				case 4608:	strictBlockSizeComboBox->setCurrentItem( 4 );	break;
				case 256:	strictBlockSizeComboBox->setCurrentItem( 5 );	break;
				case 512:	strictBlockSizeComboBox->setCurrentItem( 6 );	break;
				case 1024:	strictBlockSizeComboBox->setCurrentItem( 7 );	break;
				case 2048:	strictBlockSizeComboBox->setCurrentItem( 8 );	break;
				case 4096:	strictBlockSizeComboBox->setCurrentItem( 9 );	break;
				case 8192:	strictBlockSizeComboBox->setCurrentItem( 10 );	break;
				case 16384:	strictBlockSizeComboBox->setCurrentItem( 11 );	break;
				case 32768:	strictBlockSizeComboBox->setCurrentItem( 12 );	break;
			}
		}
	}

	regExp.setPattern( "(^|[ \\t])(-m|--mid-side)([ \\t]|$)" );
	if ( regExp.search( switches ) != -1 )
	{
		stereoModeCheckBox->setChecked( true );
		stereoModeComboBox->setCurrentItem( 0 );
	}
	else
	{
		regExp.setPattern( "(^|[ \\t])(-M|--adaptive-mid-side)([ \\t]|$)" );
		if ( regExp.search( switches ) != -1 )
		{
			stereoModeCheckBox->setChecked( true );
			stereoModeComboBox->setCurrentItem( 1 );
		}
	}

	// MISC

	regExp.setPattern( "(^|[ \\t])--ogg([ \\t]|$)" );
	if ( regExp.search( switches ) != -1 )
		oggFLACCheckBox->setChecked( true );

	regExp.setPattern( "(^|[ \\t])(-V|--verify)([ \\t]|$)" );
	if ( regExp.search( switches ) != -1 )
		verifyFileCheckBox->setChecked( true );



	// METADATA

	regExp.setPattern( "(^|[ \\t])--replay-gain([ \\t]|$)" );
	if ( regExp.search( switches ) != -1 )
		replayGainCheckBox->setChecked( true );

	regExp.setPattern( "(^|[ \\t])--cuesheet=([^ \\t]+)([ \\t]|$)" );
	if ( regExp.search( switches ) != -1 )
	{
		cueSheetCheckBox->setChecked( true );
		cueSheetLineEdit->setText( regExp.cap( 2 ) );

		regExp.setPattern( "(^|[ \\t])--no-cued-seekpoints([ \\t]|$)" );
		if ( regExp.search( switches ) != -1 )
			noCueSheetPointsCheckBox->setChecked( true );
	}

	regExp.setPattern( "(^|[ \\t])(-P[ \\t]+|--padding=)([0-9]+)([ \\t]|$)" );
	if ( regExp.search( switches ) != -1 )
	{
		paddingCheckBox->setChecked( true );
		paddingSpinBox->setValue( regExp.cap( 3 ).toInt() );
	}
	else
	{
		regExp.setPattern( "(^|[ \\t])--no-padding([ \\t]|$)" );
		if ( regExp.search( switches ) != -1 )
		{
			paddingCheckBox->setChecked( true );
			paddingSpinBox->setValue( 0 );
		}
	}


	regExp.setPattern( "(^|[ \\t])(-S[ \\t]+|--seekpoint=)([0-9]+\\.?[0-9]*|[0-9]*\\.?[0-9]+)s([ \\t]|$)" );
	if ( regExp.search( switches ) != -1 )
	{
		seekPointsButtonGroup->setChecked( true );
		seekPointsSecsRadioButton->setChecked( true );
		seekPointsSecsDecSpinBox->setDecimalValue( regExp.cap( 3 ).toDouble() );
	}
	else
	{
		regExp.setPattern( "(^|[ \\t])(-S[ \\t]+|--seekpoint=)([0-9]+)x([ \\t]|$)" );
		if ( regExp.search( switches ) != -1 )
		{
			seekPointsButtonGroup->setChecked( true );
			seekPointsCountRadioButton->setChecked( true );
			seekPointsCountSpinBox->setValue( regExp.cap( 3 ).toInt() );
		}
	}

	regExp.setPattern( "(^|[ \\t])(--sector-align)([ \\t]|$)" );
	if ( regExp.search( switches ) != -1 )
		sectorAlignCheckBox->setChecked( true );
	else
	{
		regExp.setPattern( "(^|[ \\t])--skip=([0-9]+)([ \\t]|$)" );
		if ( regExp.search( switches ) != -1 )
		{
			startButtonGroup->setChecked( true );
			startSamplesRadioButton->setChecked( true );
			startSamplesSpinBox->setValue( regExp.cap( 2 ).toInt() );
		}
		else
		{
			regExp.setPattern( "(^|[ \\t])--skip=([0-9]+):([0-5]?[0-9]|[0-5]?[0-9]\\.[0-9]?[0-9]?[0-9]?)([ \\t]|$)" );
			if ( regExp.search( switches ) != -1 )
			{
				startButtonGroup->setChecked( true );
				startTimeRadioButton->setChecked( true );

				int min = regExp.cap( 2 ).toInt();
				float f_sec = regExp.cap( 3 ).toFloat();
				int sec = (int)f_sec;
				int msec = (int)((f_sec - sec) * 1000);

				startMinsSpinBox->setValue( min );
				startSecsSpinBox->setValue( sec );
				startMsecsSpinBox->setValue( msec );
			}
		}

		regExp.setPattern( "(^|[ \\t])--until=([+-]?)([0-9]+)([ \\t]|$)" );
		if ( regExp.search( switches ) != -1 )
		{
			durationButtonGroup->setChecked( true );
			durationSamplesRadioButton->setChecked( true );
			durationSamplesSpinBox->setValue( regExp.cap( 3 ).toInt() );

			if ( regExp.cap( 2 ) == "-" )
				relativeToEndCheckBox->setChecked( true );
		}
		else
		{
			regExp.setPattern( "(^|[ \\t])--until=([+-]?)([0-9]+):([0-5]?[0-9]|[0-5]?[0-9]\\.[0-9]?[0-9]?[0-9]?)([ \\t]|$)" );
			if ( regExp.search( switches ) != -1 )
			{
				durationButtonGroup->setChecked( true );
				durationTimeRadioButton->setChecked( true );

				int min = regExp.cap( 3 ).toInt();
				float f_sec = regExp.cap( 4 ).toFloat();
				int sec = (int)f_sec;
				int msec = (int)((f_sec - sec) * 1000);

				durationMinsSpinBox->setValue( min );
				durationSecsSpinBox->setValue( sec );
				durationMsecsSpinBox->setValue( msec );

				if ( regExp.cap( 2 ) == "-" )
					relativeToEndCheckBox->setChecked( true );
			}
		}
	}
}

QString FLACEncoderDialog::switches()
{
	QString result;

	if ( profileCheckBox->isChecked() )
		result = "--compression-level-" + QString::number( profileComboBox->currentItem() );

	if ( exhaustiveModelSearchCheckBox->isChecked() )
		result += " --exhaustive-model-search";

	if ( maxLPCOrderCheckBox->isChecked() )
		result += " --max-lpc-order=" + QString::number( maxLPCOrderSpinBox->value() );

	if ( quantizedLPCPrecisionCheckBox->isChecked() )
	{
		int value = quantizedLPCPrecisionComboBox->currentItem();
		if ( value == 0 )
			result += " --qlp-coeff-precision-search";
		else
			result += " --qlp-coeff-precision=" + QString::number( value-1 );
	}

	if ( maxResidualPOCheckBox->isChecked() )
	{
		if ( minResidualPOCheckBox->isChecked() )
			result += " --rice-partition-order=" + QString::number( minResidualPOSpinBox->value() ) + "," + QString::number( maxResidualPOSpinBox->value() );
		else
			result += " --rice-partition-order=" + QString::number( maxResidualPOSpinBox->value() );
	}

	if ( blockSizeCheckBox->isChecked() )
	{
		if ( laxBlockSizeRadioButton->isChecked() )
			result += " --lax --blocksize=" + QString::number( laxBlockSizeSpinBox->value() );
		else
			result += " --blocksize=" + strictBlockSizeComboBox->currentText();
	}

	if ( stereoModeCheckBox->isChecked() )
	{
		if ( stereoModeComboBox->currentItem() == 0 )
			result += " --mid-side";
		else
			result += " --adaptive-mid-side";
	}


	if ( oggFLACCheckBox->isChecked() )
		result += " --ogg";

	if ( verifyFileCheckBox->isChecked() )
		result += " --verify";

	if ( replayGainCheckBox->isChecked() )
		result += " --replay-gain";

	if ( cueSheetCheckBox->isChecked() && ! cueSheetLineEdit->text().isEmpty() )
	{
		result += " --cuesheet=" + String::quoteForShell( cueSheetLineEdit->text() );

		if ( noCueSheetPointsCheckBox->isChecked() )
			result += " --no-cued-seekpoints";
	}

	if ( paddingCheckBox->isChecked() )
	{
		if ( paddingSpinBox->value() == 0 )
			result += " --no-padding";
		else
			result += " --padding=" + QString::number( paddingSpinBox->value() );
	}

	if ( seekPointsButtonGroup->isChecked() )
	{
		if ( seekPointsSecsRadioButton->isChecked() )
			result += " --seekpoint=" + QString::number( seekPointsSecsDecSpinBox->decimalValue(), 'f', 3 ) + "s";
		else // seekPointsCountRadioButton->isChecked()
			result += " --seekpoint=" + QString::number( seekPointsCountSpinBox->value() ) + "x";
	}


	if ( sectorAlignCheckBox->isChecked() )
		result += " --sector-align";
	else
	{
		if ( startButtonGroup->isChecked() )
		{
			if ( startSamplesRadioButton->isChecked() )
				result += " --skip=" + QString::number( startSamplesSpinBox->value() );
			else // startTimeRadioButton->isChecked()
				result += " --skip=" +	QString::number( startMinsSpinBox->value() ) + ":" +
										QString::number( (double)(startSecsSpinBox->value()) +
														 (double)(startMsecsSpinBox->value())/1000, 'f', 3 );
		}

		if ( durationButtonGroup->isChecked() )
		{
			if ( durationSamplesRadioButton->isChecked() )
				result += " --until=" +	QString( relativeToEndCheckBox->isChecked() ? "-" : " ") +
										QString::number( durationSamplesSpinBox->value() );
			else // durationTimeRadioButton->isChecked()
				result += " --until=" +	QString( relativeToEndCheckBox->isChecked() ? "-" : "" ) +
										QString::number( durationMinsSpinBox->value() ) + ":" +
										QString::number( (double)(durationSecsSpinBox->value()) +
														 (double)(durationMsecsSpinBox->value())/1000, 'f', 3 );
		}
	}

	return result.stripWhiteSpace();
}
