/***************************************************************************
 *   Copyright (C) 2005 by Sergio Pistone                                  *
 *   sergio_pistone@yahoo.com.ar                                           *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
	#include <config.h>
#endif

#include "processhandler.h"
#include <commondefs.h>

#include <qtimer.h>
#include <qregexp.h>

ProcessHandler::ProcessHandler( int priority, int comm, QObject* parent, const char* name ):
	QProcess( parent, name ),
	m_priority( priority ),
	m_paused( false ),
	m_shellParsing( false ),
	m_waitCondition(),
	m_mutex(),
	m_finished( false )
{
	setCommunication( comm );
	connect( this, SIGNAL( processExited() ), this, SLOT( onProcessExited() ) );
}

ProcessHandler::ProcessHandler( const QString& arg0, int priority, int comm, QObject* parent, const char* name ):
	QProcess( arg0, parent, name ),
	m_priority( priority ),
	m_paused( false ),
	m_shellParsing( false ),
	m_waitCondition(),
	m_mutex(),
	m_finished( false )
{
	setCommunication( comm );
}

ProcessHandler::ProcessHandler( const QStringList& args, int priority, int comm, QObject* parent, const char* name ):
	QProcess( args, parent, name ),
	m_priority( priority ),
	m_paused( false ),
	m_shellParsing( false ),
	m_waitCondition(),
	m_mutex(),
	m_finished( false )
{
	setCommunication( comm );
}

ProcessHandler::~ProcessHandler()
{
	kill();
}

bool ProcessHandler::start( QStringList* env )
{
	{ // Begin mutex locker scope
		QMutexLocker locker( &m_mutex );

		m_finished = false;
		m_waitCondition.wakeAll(); // wake up anyone who could be waiting an old process
	} // End mutex locker scopt

	bool ret = QProcess::start( env );

	if ( ret )
	{
		if ( ! setPriority( m_priority, true ) )
			m_priority = 0;
	}
	else
		m_priority = 0;

	return ret;
}

void ProcessHandler::terminate( unsigned long msecs )
{
	if ( msecs > 0 )
	{
		tryTerminate();
		QTimer::singleShot( msecs, this, SLOT( kill() ) );
	}
	else
		kill();
}

void ProcessHandler::onProcessExited()
{
	QMutexLocker locker( &m_mutex );

	m_finished = true;
	m_waitCondition.wakeAll();
}

bool ProcessHandler::waitProcess( unsigned long msecs )
{
	QMutexLocker locker( &m_mutex );

	if ( m_finished )
		return true;

	m_waitCondition.wait( &m_mutex, msecs );

	return m_finished;
}

int ProcessHandler::priority() const
{
	return m_priority;
}

bool ProcessHandler::setPriority( int priority )
{
	return setPriority( priority, false );
}

bool ProcessHandler::setPriority( int priority, bool force )
{
#ifdef Q_OS_UNIX
	if ( ! isRunning() )
	{
		m_priority = priority;
		return false;
	}

	if ( force || m_priority != priority )
	{
		QString cmd( QString( "renice %1 -p %2" ).arg( priority ).arg( processIdentifier() ) );
		kdDebug() << "executing " << cmd << endl;
		if ( ::system( cmd ) == 0 )
			m_priority = priority;
	}
#else
#endif
	return m_priority == priority;
}

bool ProcessHandler::paused() const
{
	return m_paused;
}

bool ProcessHandler::pause()
{
#ifdef Q_OS_UNIX
	if ( ! m_paused )
	{
		QString cmd( QString( "kill -s STOP %2" ).arg( processIdentifier() ) );
		kdDebug() << "executing " << cmd << endl;
		if ( ::system( cmd ) == 0 )
			m_paused = true;
	}
#else
#endif
	return m_paused;
}

bool ProcessHandler::resume()
{
#ifdef Q_OS_UNIX
	if ( m_paused )
	{
		QString cmd( QString( "kill -s CONT %2" ).arg( processIdentifier() ) );
		kdDebug() << "executing " << cmd << endl;
		if ( ::system( cmd ) == 0 )
			m_paused = false;
	}
#else
#endif
	return ! m_paused;
}

void ProcessHandler::setShellParsingMode( bool value )
{
	m_shellParsing = value;
}

bool ProcessHandler::getShellParsingMode()
{
	return m_shellParsing;
}


void ProcessHandler::setArguments( const QStringList& args )
{
	clearArguments();
	addArguments( args );
}

void ProcessHandler::addArgument( const QString& arg )
{
	if ( m_shellParsing )
	{
		QStringList args = parseShellArgs( arg );
		for ( QStringList::ConstIterator it = args.begin(); it != args.end(); ++it )
			QProcess::addArgument( *it );
	}
	else
		QProcess::addArgument( arg );
}

void ProcessHandler::addArguments( const QStringList& args )
{
	for ( QStringList::ConstIterator it = args.begin(); it != args.end(); ++it )
		addArgument( *it );
}

ProcessHandler& ProcessHandler::operator<<( const QStringList& args )
{
	addArguments( args );
	return *this;
}

ProcessHandler& ProcessHandler::operator<<( const char* arg )
{
	addArgument( arg );
	return *this;
}

ProcessHandler& ProcessHandler::operator<<( const QByteArray& arg )
{
	return operator<<( arg.data() );
}

ProcessHandler& ProcessHandler::operator<<( const QString& arg )
{
	addArgument( arg );
	return *this;
}


QStringList ProcessHandler::parseShellArgs( const QString& args )
{
	#define FIND_ARG_START	0
	#define FIND_ARG_END	1

	static const QRegExp regExp( "\\\\([^\\\\])" );
	bool escaped = false;
	bool action = FIND_ARG_START;
	QChar argStartChar = '\0';
	unsigned argStartIdx = 0;
	unsigned argEndIdx = 0;

	QStringList ret;

	for( unsigned idx = 0; idx < args.length(); ++idx )
	{
		bool findArgEnd = action == FIND_ARG_END;

		const QChar c = args.at( idx );
		if ( action == FIND_ARG_START )
		{
			if (! escaped && ( c == '"' || c == '\'' ))
			{
				argStartChar = c;
				argStartIdx = idx;
				action = FIND_ARG_END;
			}
			else if ( (escaped && c == ' ') || (! escaped && c != ' ') )
			{
				argStartChar = '\0';
				argStartIdx = idx;
				action = FIND_ARG_END;
				if ( idx == args.length()-1 )
					findArgEnd = true;
			}
		}

		if ( findArgEnd )
		{
			if ( /*argStartChar != ' ' &&*/ ! escaped && c == argStartChar )
			{
				action = FIND_ARG_START;
				QString arg = args.mid( argStartIdx + 1, idx - argStartIdx - 1 );
				if ( c == '"' )
					arg.replace( "\\\\", "\\" );
				if ( argEndIdx+1 == argStartIdx && ret.size() > 0 ) // join with last parameter
				{
					arg = ret.last() + arg;
					ret.pop_back();
				}
				ret.append( arg );
				argEndIdx = idx;
			}
			else if ( argStartChar == '\0' && ( (! escaped && c == ' ') || (idx == args.length()-1) ) )
			{
				action = FIND_ARG_START;
				QString arg = args.mid( argStartIdx, idx - argStartIdx - (c == ' ' ? 0 : -1) );
				arg.replace( "\\ ", " " );
				arg.replace( regExp /* QRegExp( "\\\\([^\\\\])" ) */, "\\1" );
				arg.replace( "\\\\", "\\" );
				if ( argEndIdx+1 == argStartIdx && ret.size() > 0 ) // join with last parameter
				{
					arg = ret.last() + arg;
					ret.pop_back();
				}
				ret.append( arg );
				argEndIdx = idx - 1;
			}
		}

		if ( escaped )
			escaped = false;
		else
			escaped = c == '\\';
	}

	return ret;
}
