/***************************************************************************
 *   Copyright (C) 2005 by Sergio Pistone                                  *
 *   sergio_pistone@yahoo.com.ar                                           *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "processoutput.h"
#include <commondefs.h>

OutputParser::OutputParser( QObject* parent, const char* name ):
	QObject( parent, name ) {};

OutputParser::~OutputParser() {}



OutputGrabber::OutputGrabber( QProcess* process ):
	m_process( process )
{
	if ( m_process == 0 )
		return;

	int comm = m_process->communication();
	if ( comm & QProcess::Stdout )
		connect( m_process, SIGNAL( readyReadStdout() ), this, SLOT( saveStdout() ) );
	if ( (comm & QProcess::Stderr) && ! (comm & QProcess::DupStderr) )
		connect( m_process, SIGNAL( readyReadStderr() ), this, SLOT( saveStderr() ) );
}

OutputGrabber::~OutputGrabber() { disconnect(); }

QByteArray OutputGrabber::getOutput()
{
	m_mutex.lock();

	// makes a deep copy of m_output
	QByteArray ret = m_output.copy();

	m_mutex.unlock();

	return ret;
}

QString OutputGrabber::getOutputAsString()
{
	m_mutex.lock();

	QString ret = QString::fromLocal8Bit( m_output.data(), m_output.size() );

	m_mutex.unlock();

	return ret;
}

QByteArray OutputGrabber::takeOutput()
{
	m_mutex.lock();

	QByteArray ret = m_output;
	m_output = QByteArray();

	m_mutex.unlock();

	return ret;
}

QString OutputGrabber::takeOutputAsString()
{
	m_mutex.lock();

	QString ret = QString::fromLocal8Bit( m_output.data(), m_output.size() );
	m_output = QByteArray();

	m_mutex.unlock();

	return ret;
}

void OutputGrabber::saveStdout()
{
	QByteArray newOutput = m_process->readStdout();

	m_mutex.lock();

	int oldSize = m_output.size();
	int newSize = oldSize + newOutput.size();

	m_output.resize( newSize );
	char* m_outputData = m_output.data();
	char* newOutputData = newOutput.data();

	for ( int idx1 = oldSize, idx2 = 0; idx1 < newSize; ++idx1, ++idx2 )
		m_outputData[idx1] = newOutputData[idx2];

	m_mutex.unlock();
}

void OutputGrabber::saveStderr()
{
	QByteArray newOutput = m_process->readStderr();

	m_mutex.lock();

	int oldSize = m_output.size();
	int newSize = oldSize + newOutput.size();

	m_output.resize( newSize );
	char* m_outputData = m_output.data();
	char* newOutputData = newOutput.data();

	for ( int idx1 = oldSize, idx2 = 0; idx1 < newSize; ++idx1, ++idx2 )
		m_outputData[idx1] = newOutputData[idx2];

	m_mutex.unlock();
}


OutputParserProxy::OutputParserProxy( QProcess* process, OutputParser* parser, unsigned pollInterval ):
	QThread(),
	m_parser( parser ),
	m_pollInterval( pollInterval ),
	m_outputGrabber( process ),
	m_stop( false ),
	m_mutex( false )
{
	if ( process == 0 || m_parser == 0 )
		return;

	start();
}

void OutputParserProxy::run()
{
	while ( ! stop() )
	{
		QString output = m_outputGrabber.takeOutput();
		if ( ! output.isEmpty() )
			m_parser->parseOutput( output );

		QThread::msleep( m_pollInterval );
	}

	QString output = m_outputGrabber.takeOutputAsString();
	if ( ! output.isEmpty() )
		m_parser->parseOutput( output );
}

bool OutputParserProxy::stop()
{
	QMutexLocker locker( &m_mutex );

	return m_stop;
}

OutputParserProxy::~OutputParserProxy()
{
	if ( m_parser != 0 ) // if parser == 0 the thread was never started
	{
		m_outputGrabber.disconnect();

		m_mutex.lock();

		m_stop = true;

		m_mutex.unlock();

		wait(); // wait for thread to end

		m_parser->disconnect();

		//delete m_parser;
		m_parser->deleteLater();
	}
}
