#ifndef TRANSKODE_PROGRESSPARSER_H
#define TRANSKODE_PROGRESSPARSER_H

/***************************************************************************
 *   Copyright (C) 2005 by Sergio Pistone                                  *
 *   sergio_pistone@yahoo.com.ar                                           *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
	#include <config.h>
#endif

#include "processoutput.h"
#include <qregexp.h>

class ProgressParser : public OutputParser
{
	Q_OBJECT

	signals:
		void progress( int progress );

	protected:
		ProgressParser();
		virtual ~ProgressParser();

		virtual void parseOutput( const QString& output );
		virtual int parseProgress( const QString& output ) = 0;

		int m_progress;
};


class RegExpProgressParser : public ProgressParser
{
	public:
		typedef enum {
			int_percentage,		// an integer from 0 to 100
			float_percentage,	// a float from 0.0 to 100.0
			float_normalized,	// a float from 0.0 to 1.0
			int_fraction,		// an integers fraction from 0/total to total/total
			float_fraction		// a floats fraction from 0.0/total(f) to total/total(f)
		} Type;

		RegExpProgressParser( QString regExp, Type type ):
			m_regExp( regExp ),
			m_type( type ) {}
		virtual ~RegExpProgressParser() {}

	protected:

		int parseProgress( const QString& line )
		{
			if ( m_regExp.search( line ) == -1 )
				return -1;

			switch ( m_type )
			{
				case int_percentage:
					return m_regExp.cap( 1 ).toInt();
				case float_percentage:
					return (int)(m_regExp.cap( 1 ).toFloat()+0.5);
				case float_normalized:
					return (int)((m_regExp.cap( 1 ).toFloat()*100.0)+0.5);
				case int_fraction:
					return (int)((100.0*m_regExp.cap( 1 ).toInt()/m_regExp.cap( 2 ).toInt())+0.5);
				case float_fraction:
					return (int)((100.0*m_regExp.cap( 1 ).toFloat()/m_regExp.cap( 2 ).toFloat())+0.5);
				default:
					return -1;
			}
		}

		QRegExp m_regExp;
		Type m_type;
};

#endif
