/***************************************************************************
 *   Copyright (C) 2007 by Sergio Pistone                                  *
 *   sergio_pistone@yahoo.com.ar                                           *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "twolameencoderdialog.h"

#include <qregexp.h>
#include <qradiobutton.h>
#include <qcheckbox.h>
#include <kcombobox.h>
#include <kdecspinbox.h>

TwoLAMEEncoderDialog::TwoLAMEEncoderDialog( QWidget* parent, const char* name ):
	TwoLAMEEncoderDialogBase( parent, name ) {}

TwoLAMEEncoderDialog::~TwoLAMEEncoderDialog() {}

void TwoLAMEEncoderDialog::setSwitches( const QString& switches )
{
	// ENCODING SETTINGS

	QRegExp regExp( "(^|[ \\t])(-v[ \\t]+|--vbr-level[ \\t]+|--vbr-level=)([0-9]+\\.?[0-9]*|-?[0-9]*\\.?[0-9]+)([ \\t]|$)" );
	if ( regExp.search( switches ) != -1 )
	{
		vbrRadioButton->setChecked( true );
		toggleVariableBitRate( true );
		qualityCheckBox->setChecked( true );
		qualityDecSpinBox->setDecimalValue( regExp.cap( 3 ).toDouble() );
	}
	else
	{
		regExp.setPattern( "(^|[ \\t])(-v|--vbr)([ \\t]|$)" );
		if ( regExp.search( switches ) != -1 )
		{
			vbrRadioButton->setChecked( true );
			toggleVariableBitRate( true );
		}
		else
		{
			cbrRadioButton->setChecked( true );
			toggleConstantBitRate( true );
			regExp.setPattern( "(^|[ \\t])(-b[ \\t]+|--bitrate[ \\t]+|--bitrate=)(32|40|48|56|64|80|96|112|128|160|192|224|256|320)([ \\t]|$)" );
			if ( regExp.search( switches ) != -1 )
			{
				bitrateCheckBox->setChecked( true );
				int bitrateIdx = 0;
				switch ( regExp.cap( 3 ).toInt() )
				{
					case 32:	bitrateIdx = 0;		break;
					case 40:	bitrateIdx = 1;		break;
					case 48:	bitrateIdx = 2;		break;
					case 56:	bitrateIdx = 3;		break;
					case 64:	bitrateIdx = 4;		break;
					case 80:	bitrateIdx = 5;		break;
					case 96:	bitrateIdx = 6;		break;
					case 112:	bitrateIdx = 7;		break;
					case 128:	bitrateIdx = 8;		break;
					case 160:	bitrateIdx = 9;		break;
					case 192:	bitrateIdx = 10;	break;
					case 224:	bitrateIdx = 11;	break;
					case 256:	bitrateIdx = 12;	break;
					case 320:	bitrateIdx = 13;	break;
				}
				bitrateComboBox->setCurrentItem( bitrateIdx );
			}
		}
	}

	regExp.setPattern( "(^|[ \\t])(-m[ \\t]+|--mode[ \\t]+|--mode=)(a|s|d|j|m)([ \\t]|$)" );
	if ( regExp.search( switches ) != -1 )
	{
		stereoModeCheckBox->setChecked( true );
		if ( regExp.cap( 3 ) == "a" )
			stereoModeComboBox->setCurrentItem( 0 );
		else if ( regExp.cap( 3 ) == "s" )
			stereoModeComboBox->setCurrentItem( 1 );
		else if ( regExp.cap( 3 ) == "d" )
			stereoModeComboBox->setCurrentItem( 2 );
		else if ( regExp.cap( 3 ) == "j" )
			stereoModeComboBox->setCurrentItem( 3 );
		else if ( regExp.cap( 3 ) == "m" )
			stereoModeComboBox->setCurrentItem( 4 );
	}
	else
	{
		regExp.setPattern( "(^|[ \\t])(-a|--downmix)([ \\t]|$)" );
		if ( regExp.search( switches ) != -1 ) // set mono stereo mode
		{
			stereoModeCheckBox->setChecked( true );
			stereoModeComboBox->setCurrentItem( 4 );
		}
	}


	// PSYCHOACOUSTIC MODEL SETTINGS

	regExp.setPattern( "(^|[ \\t])(-P[ \\t]+|--psyc-mode[ \\t]+|--psyc-mode=)(-1|0|1|2|3|4)([ \\t]|$)" );
	if ( regExp.search( switches ) != -1 )
	{
		psyModelCheckBox->setChecked( true );
		psyModelComboBox->setCurrentItem( regExp.cap( 3 ).toInt() + 1 );
	}

	regExp.setPattern( "(^|[ \\t])(-l[ \\t]+|--ath[ \\t]+|--ath=)([0-9]+\\.?[0-9]*|-?[0-9]*\\.?[0-9]+)([ \\t]|$)" );
	if ( regExp.search( switches ) != -1 )
	{
		athLevelCheckBox->setChecked( true );
		athLevelDecSpinBox->setDecimalValue( regExp.cap( 3 ).toDouble() );
	}

	regExp.setPattern( "(^|[ \\t])(-q[ \\t]+|--quick[ \\t]+|--quick=)([0-9]+)([ \\t]|$)" );
	if ( regExp.search( switches ) != -1 )
	{
		quickModeCheckBox->setChecked( true );
		quickModeSpinBox->setValue( regExp.cap( 3 ).toInt() );
	}


	// MISCELLANEOUS SETTINGS

	regExp.setPattern( "(^|[ \\t])(-c|--copyright)([ \\t]|$)" );
	if ( regExp.search( switches ) != -1 )
		copyrightCheckBox->setChecked( true );

	regExp.setPattern( "(^|[ \\t])(--original)([ \\t]|$)" );
	if ( regExp.search( switches ) != -1 )
		originalCheckBox->setChecked( true );

	regExp.setPattern( "(^|[ \\t])(-p|--protect)([ \\t]|$)" );
	if ( regExp.search( switches ) != -1 )
		crcCheckBox->setChecked( true );

	regExp.setPattern( "(^|[ \\t])(-d|--padding)([ \\t]|$)" );
	if ( regExp.search( switches ) != -1 )
		paddingCheckBox->setChecked( true );

	regExp.setPattern( "(^|[ \\t])(-E|--energy)([ \\t]|$)" );
	if ( regExp.search( switches ) != -1 )
		energyCheckBox->setChecked( true );

	regExp.setPattern( "(^|[ \\t])(-R[ \\t]+|--reserve[ \\t]+|--reserve=)([0-9]+)([ \\t]|$)" );
	if ( regExp.search( switches ) != -1 )
	{
		reserveBitsCheckBox->setChecked( true );
		reserveBitsSpinBox->setValue( regExp.cap( 3 ).toInt() );
	}


	// AUDIO PROCESSING SETTINGS

	regExp.setPattern( "(^|[ \\t])--scale[ \\t]+([0-9]+\\.?[0-9]*|[0-9]*\\.?[0-9]+)([ \\t]|$)" );
	if ( regExp.search( switches ) != -1 )
	{
		scaleBothCheckBox->setChecked( true );
		scaleBothDecSpinBox->setDecimalValue( regExp.cap( 2 ).toDouble() );
	}

	regExp.setPattern( "(^|[ \\t])--scale-l[ \\t]+([0-9]+\\.?[0-9]*|[0-9]*\\.?[0-9]+)([ \\t]|$)" );
	if ( regExp.search( switches ) != -1 )
	{
		scaleLeftCheckBox->setChecked( true );
		scaleLeftDecSpinBox->setDecimalValue( regExp.cap( 2 ).toDouble() );
	}

	regExp.setPattern( "(^|[ \\t])--scale-r[ \\t]+([0-9]+\\.?[0-9]*|[0-9]*\\.?[0-9]+)([ \\t]|$)" );
	if ( regExp.search( switches ) != -1 )
	{
		scaleRightCheckBox->setChecked( true );
		scaleRightDecSpinBox->setDecimalValue( regExp.cap( 2 ).toDouble() );
	}

	regExp.setPattern( "(^|[ \\t])(-g|--swap-channels)([ \\t]|$)" );
	if ( regExp.search( switches ) != -1 )
		swapChannelsCheckBox->setChecked( true );
}


QString TwoLAMEEncoderDialog::switches()
{
	QString result;

	// ENCODING SETTINGS

	if ( vbrRadioButton->isChecked() )
	{
		result += "-v";
		if ( qualityCheckBox->isChecked() )
			result += " " + QString::number( qualityDecSpinBox->decimalValue(), 'f', 1 );
	}
	else if ( bitrateCheckBox->isChecked() )
		result += "-b " + bitrateComboBox->currentText();

	if ( stereoModeCheckBox->isChecked() )
	{
		QString stereoMode;
		switch ( stereoModeComboBox->currentItem() )
		{
			case 0:		stereoMode = "a";		break;
			case 1:		stereoMode = "s";		break;
			case 2:		stereoMode = "d";		break;
			case 3:		stereoMode = "j";		break;
			case 4:		stereoMode = "m";		break;
			default:	stereoMode = "j";		break;
		}
		result += " --mode " + stereoMode;
	}


	// PSYCHOACOUSTIC MODEL SETTINGS

	if ( psyModelCheckBox->isChecked() )
		result += " --psyc-mode " + QString::number( psyModelComboBox->currentItem() - 1 );

	if ( athLevelCheckBox->isChecked() )
		result += " --ath " + QString::number( athLevelDecSpinBox->decimalValue(), 'f', 1 );

	if ( quickModeCheckBox->isChecked() )
		result += " --quick " + QString::number( quickModeSpinBox->value() );


	// MISCELLANEOUS SETTINGS

	if ( copyrightCheckBox->isChecked() )
		result += " --copyright";

	if ( originalCheckBox->isChecked() )
		result += " --original";

	if ( crcCheckBox->isChecked() )
		result += " --protect";

	if ( paddingCheckBox->isChecked() )
		result += " --padding";

	if ( energyCheckBox->isChecked() )
		result += " --energy";

	if ( reserveBitsCheckBox->isChecked() )
		result += " --reserve " + QString::number( reserveBitsSpinBox->value() );


	// AUDIO PROCESSING SETTINGS

	if ( scaleBothCheckBox->isChecked() )
		result += " --scale " + QString::number( scaleBothDecSpinBox->decimalValue(), 'f', 6 );

	if ( scaleLeftCheckBox->isChecked() )
		result += " --scale-l " + QString::number( scaleLeftDecSpinBox->decimalValue(), 'f', 6 );

	if ( scaleRightCheckBox->isChecked() )
		result += " --scale-r " + QString::number( scaleRightDecSpinBox->decimalValue(), 'f', 6 );

	if ( swapChannelsCheckBox->isChecked() )
		result += " --swap-channels";


	return result.simplifyWhiteSpace();
}
