/***************************************************************************
 *   Copyright (C) 2006 by Sergio Pistone                                  *
 *   sergio_pistone@yahoo.com.ar                                           *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/


#include "mainwindow.h"
#include "listview.h"
#include "listviewitem.h"

#include <qfileinfo.h>
#include <qpainter.h>
#include <qheader.h>

#include <kurl.h>
#include <kurldrag.h>
#include <kpopupmenu.h>
#include <klocale.h>
#include <kapplication.h>
#include <kconfigbase.h>
#include <kmessagebox.h>

using namespace TransKode;

ListView::ListView(
	QWidget *parent,
	const AppConfig& config,
	JobQueue& queue
):
	KListView( parent, "ListView" ),
	m_config( config ),
	m_queue( queue ),
	m_horizMargin( 1 ),
	m_startMsgRichText( 0 )
{
	addColumn( i18n( "Source Path" ) );
	addColumn( i18n( "Source Directory" ) );
	addColumn( i18n( "Source Filename" ) );
	addColumn( i18n( "Destination Path" ) );
	addColumn( i18n( "Destination Directory" ) );
	addColumn( i18n( "Destination Filename" ) );
	addColumn( i18n( "Profile" ) );
	addColumn( i18n( "Status" ) );

	for ( int idx = 0; idx < columns(); ++idx )
		setColumnWidthMode( idx, QListView::Manual ),

	setColumnAlignment( profile, Qt::AlignCenter );
	setColumnAlignment( status, Qt::AlignCenter );

	setShowSortIndicator( true );
	setAllColumnsShowFocus( true );
	setSelectionMode( QListView::Extended );
	setItemsMovable( true );

	setDropVisualizer( true );
	setDropVisualizerWidth( 2 );
	setItemHorizontalMargin( 3 );

	setAcceptDrops( true );
	setDragEnabled( true );
	connect( this, SIGNAL( dropped(QDropEvent*,QListViewItem*) ), this, SLOT( onDropEvent(QDropEvent*,QListViewItem*) ) );

	header()->installEventFilter( this );
	disconnect( header(), SIGNAL( sectionHandleDoubleClicked(int) ), this, 0 );
	connect( header(), SIGNAL( clicked(int) ), this, SLOT( slotSetSortColumn(int) ) );
	connect( header(), SIGNAL( sectionHandleDoubleClicked(int) ), this, SLOT( slotAdjustColumnWidth(int) ) );
	connect( header(), SIGNAL( sizeChange(int,int,int) ), SLOT( triggerUpdate() ) );


    m_startMsgRichText = new QSimpleRichText( i18n(
		"<div align=center>"
		"<h3>Jobs List</h3>"
		"Add files by clicking on the \"Add Files\" icon or by dropping them here.<br/>"
		"Then right click on the items to select the encoding profile."
		"</div>"
		), QApplication::font()
	);

}

ListViewItem* ListView::add( const QString& file, ListViewItem* after )
{
	ListViewItem* item = new ListViewItem( this, m_config, file );
	item->moveItem( after );

	return item;
}

ListViewItem* ListView::add( const QStringList& files, ListViewItem* after )
{
	for ( QStringList::ConstIterator it = files.begin(); it != files.end(); ++it )
		after = add( *it, after );

	return after;
}

bool ListView::removeSelected()
{
	QPtrList<ListViewItem> selectedItems;
	for ( QListViewItemIterator it( this, QListViewItemIterator::Selected ) ; it.current() != 0; ++it )
		selectedItems.append( (ListViewItem*)it.current() );

	if ( ! selectedItems.count() || KMessageBox::questionYesNo( 0, i18n( "Remove selected jobs?" ), i18n( "Confirm action" ) + " - transKode" ) != KMessageBox::Yes )
		return false;

	for ( QPtrListIterator<ListViewItem> it( selectedItems ); it.current() != 0; ++it )
		(*it)->remove();

	return true;
}

bool ListView::removeAll()
{
	if ( ! childCount() || KMessageBox::questionYesNo( 0, i18n( "Remove all jobs?" ), i18n( "Confirm action" ) + " - transKode" ) != KMessageBox::Yes )
		return false;

	QPtrList<ListViewItem> items;
	for ( QListViewItemIterator it( this ) ; it.current() != 0; ++it )
		items.append( (ListViewItem*)it.current() );

	for ( QPtrListIterator<ListViewItem> it( items ); it.current() != 0; ++it )
		(*it)->remove();

	return true;
}

void ListView::removeFinished()
{
	QPtrList<ListViewItem> finishedItems;
	for ( QListViewItemIterator it( this ) ; it.current() != 0; ++it )
		if ( ((ListViewItem*)it.current())->isFinishedOK() )
			finishedItems.append( (ListViewItem*)it.current() );

	for ( QPtrListIterator<ListViewItem> it( finishedItems ); it.current() != 0; ++it )
		(*it)->remove();
}

void ListView::enqueueSelected()
{
	for ( QListViewItemIterator it( this, QListViewItemIterator::Selected ) ; it.current() != 0; ++it )
	{
		ListViewItem* item = (ListViewItem*)it.current();
		if ( item->isQueueable() )
			item->enqueue( &m_queue );
	}
}

bool ListView::dequeueSelected()
{
	bool itemsToCancel = false;
	for ( QListViewItemIterator it( this, QListViewItemIterator::Selected ) ; it.current() != 0; ++it )
	{
		if ( ((ListViewItem*)it.current())->isCancellable() )
		{
			itemsToCancel = true;
			break;
		}
	}

	if ( itemsToCancel )
	{
		if ( KMessageBox::questionYesNo( 0, i18n( "Cancel transcoding of selected files?" ), i18n( "Confirm action" ) + " - transKode" ) != KMessageBox::Yes )
			return false;

		for ( QListViewItemIterator it( this, QListViewItemIterator::Selected ) ; it.current() != 0; ++it )
			((ListViewItem*)it.current())->stop();
	}

	return true;
}

void ListView::selectAll()
{
	KListView::selectAll( true );
}

void ListView::refreshSelected()
{
	for ( QListViewItemIterator it( this, QListViewItemIterator::Selected ) ; it.current() != 0; ++it )
		((ListViewItem*)it.current())->resetProfile();
}


// bool hasFinishedItems()
// {
// 	for ( QListViewItemIterator it( this, QListViewItemIterator::Selected ) ; it.current() != 0; ++it )
// 	{
// 		if ( it.current()->rtti() == TRANSKODE )
// 			if ( ((ListViewItem*)it.current())->isDone() )
// 				return true;
// 	}
//
// 	return false;
// }


void ListView::slotSetSortColumn( int column )
{
	setSortColumn( column );

	m_sortColumns.remove( column );
	m_sortColumns.prepend( column );
}

void ListView::slotAdjustColumnWidth( int column )
{
	adjustColumn( column );
	setColumnWidth( column, columnWidth( column ) + 2*m_horizMargin );
}

void ListView::onDropEvent( QDropEvent* event, QListViewItem* after )
{
	KURL::List urls;
	QString text;
	if ( KURLDrag::decode( event, urls ) && ! urls.isEmpty() )
	{
		setSorting( -1 );

		QFileInfo fileInfo;
		ListViewItem* item = 0;
		for ( KURL::List::ConstIterator it = urls.begin(); it != urls.end(); ++it )
		{
			if ( (*it).protocol() != "file" )
				continue;

			fileInfo.setFile( (*it).path() );
			if ( fileInfo.isFile() )
				item = add( (*it).path(), item );
			else if ( fileInfo.isDir() )
			{
				QStringList nestedFiles;
				MainWindow::filesInDir( (*it).path(), nestedFiles );
				for ( QStringList::Iterator nestedIt = nestedFiles.begin(); nestedIt != nestedFiles.end(); ++nestedIt )
					item = add( *nestedIt, item );
			}
		}
	}
	else if ( QTextDrag::decode( event, text ) ) // dragging within the list
	{
		setSorting( -1 );
		m_sortColumns.clear();

		QListViewItem* current = currentItem();

		QListViewItem* item;
		QStringList list = QStringList::split( "\n", text );
		// How dangerous is this kind of manipulation?... probably pretty nasty
		for ( QStringList::Iterator it = list.begin(); it != list.end(); ++it )
		{
			item = static_cast<QListViewItem*>((void *)(*it).toLong( 0 ));
			//item->moveItem( after ); // this one doesn't work when after is 0
			moveItem( item, 0, after );
			after = item;
		}

		setCurrentItem( current );
	}
}

bool ListView::acceptDrag( QDropEvent* event ) const
{
	return KURLDrag::canDecode( event ) || ( QTextDrag::canDecode( event ) && event->source() == this );
}

QDragObject* ListView::dragObject()
{
	QString text;

	for ( QListViewItemIterator it( this ); it.current(); ++it )
		if ( it.current()->isSelected() )
			text += QString::number( (long int)it.current() ) + "\n";

	return new QTextDrag( text, this );
}

// Method (mostly) taken from Amarok
bool ListView::eventFilter( QObject* object, QEvent* event )
{
	if ( object == header() && event->type() == QEvent::MouseButtonPress )
	{
		QMouseEvent* me = static_cast<QMouseEvent*>(event);
		if ( me->button() == Qt::RightButton )
		{
			enum { HIDE = 1000, CUSTOM };

			const int columnUnderMouse = header()->sectionAt( me->pos().x() );

			KPopupMenu popup;
			popup.setCheckable( true );
			popup.insertItem( i18n( "Hide This Column" ), HIDE ); //TODO
			popup.setItemEnabled( HIDE, columnUnderMouse != -1 );

			for ( int i = 0; i < columns(); ++i )
			{
				popup.insertItem( columnText( i ), i, i + 1 );
				popup.setItemChecked( i, columnWidth( i ) != 0 );
			}

			//do last so it doesn't get the first id
			popup.insertTitle( "transKode " + i18n( "Columns" ), /*id*/ -1, /*index*/ 1 );

			int col = popup.exec( me->globalPos() );

			if ( col == HIDE )
			{
				hideColumn( columnUnderMouse );
				QResizeEvent e( size(), QSize() );
				viewportResizeEvent( &e );
			}
			else if ( col != -1 )
			{
				//TODO can result in massively wide column appearing!
				if ( columnWidth( col ) == 0 )
				{
					slotAdjustColumnWidth( col );
					header()->setResizeEnabled( true, col );
				}
				else
				{
					hideColumn( col );
					header()->setResizeEnabled( false, col );
				}
			}

			return true; //eat event
		}
	}

	return KListView::eventFilter( object, event ); //allow the header to process this
}


void ListView::contextMenuEvent( QContextMenuEvent* e )
{
	QPtrList<ListViewItem> selectedItems;
	for ( QListViewItemIterator it( this, QListViewItemIterator::Selected ) ; it.current() != 0; ++it )
		if ( ! ((ListViewItem*)it.current())->isEnqueued() )
			selectedItems.append( ((ListViewItem*)it.current()) );

	if ( selectedItems.count() <= 0 )
		return;

	KPopupMenu popup;

	QStringList profiles = m_config.validProfiles();
	QStringList showProfiles;
	popup.insertTitle( i18n( "Set Profile" ), -1 );
	if ( profiles.count() > 0 )
	{
		int id = 0;
		for ( QStringList::Iterator it = profiles.begin(); it != profiles.end(); ++it )
		{
			EncodingProfile* options = m_config.profileOptions( *it );
			if ( options != 0 )
			{
				if ( options->show() )
				{
					popup.insertItem( *it, id++ );
					showProfiles.append( *it );
				}
				delete options;
			}
		}
	}
	else
	{
		popup.insertItem( i18n( "No valid profiles available" ), 0 );
		popup.setItemEnabled( 0, false );
	}

	int id = popup.exec( e->globalPos() );
	if ( id >= 0 )
	{
		for ( QPtrList<ListViewItem>::Iterator it = selectedItems.begin(); it != selectedItems.end(); ++it )
			(*it)->setProfile( showProfiles[id] );

		emit selectionChanged();
		emit currentChanged( currentItem() );
	}
}


void ListView::viewportPaintEvent( QPaintEvent* e )
{
	KListView::viewportPaintEvent( e );

	if ( childCount() == 0 )
	{
		QPainter p( viewport() );

		m_startMsgRichText->setWidth( width() - 50 );

		const uint w = m_startMsgRichText->width() + 20;
		const uint h = m_startMsgRichText->height() + 20;

		p.setBrush( colorGroup().background() );
		p.drawRoundRect( 15, 15, w, h, (8*200)/w, (8*200)/h );
		m_startMsgRichText->draw( &p, 20, 20, QRect(), colorGroup() );
	}
}

// needed for correct redraw of m_startMsgRichText help
void ListView::viewportResizeEvent( QResizeEvent* )
{
	triggerUpdate();
}

void ListView::setItemHorizontalMargin( uint margin )
{
	m_horizMargin = margin;
}

uint ListView::itemHorizontalMargin()
{
	return m_horizMargin;
}

void ListView::loadConfig()
{
	KConfigGroupSaver saver( kapp->config(), "ListView Settings" );

	restoreLayout( kapp->config(), "ListView Settings" );

	for ( int idx = 0; idx < columns(); ++idx )
		if ( columnWidth( idx ) == 0 )
			header()->setResizeEnabled( false, idx );

	m_sortColumns = kapp->config()->readIntListEntry( "SortColumns" );
}

void ListView::saveConfig()
{
	KConfigGroupSaver saver( kapp->config(), "ListView Settings" );

	saveLayout( kapp->config(), "ListView Settings" );

	kapp->config()->writeEntry( "SortColumns", m_sortColumns );
}


#include "listview.moc"
