/***************************************************************************
 *   Copyright (C) 2006 by Sergio Pistone                                  *
 *   sergio_pistone@yahoo.com.ar                                           *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "listview.h"
#include "listviewitem.h"

#include <qpainter.h>
#include <qheader.h>

using namespace TransKode;

ListViewItem::ListViewItem( QListView* parent, const AppConfig& config, const QString& srcPath ):
	QObject( 0, "ListViewItem" ),
	KListViewItem( parent ),
	m_job( new FileJob( config, srcPath ) ),
	m_queue( 0 ),
	m_delayedDeletion( false )
{
	QFileInfo fileInfo( srcPath );
	setText( ListView::src_path, fileInfo.filePath() );
	setText( ListView::src_dir, fileInfo.dirPath() );
	setText( ListView::src_file, fileInfo.fileName() );

	setText( ListView::status, i18n( "waiting profile" ) );

	// update the state of the item from the job status signals
	connect( m_job, SIGNAL( mtDstPathChanged(long,const QString&) ), this, SLOT( setDstPath(long,const QString&) ) );
	connect( m_job, SIGNAL( mtStarted(long,const QString&,const QString&) ), this, SLOT( setStarted() ) );
	connect( m_job, SIGNAL( mtProgress(long,int) ), this, SLOT( setProgress(long,int) ) );
	connect( m_job, SIGNAL( mtFinishedOK(long,const QString&) ), this, SLOT( setFinishedOK() ) );
	connect( m_job, SIGNAL( mtFinishedWithError(long,const QString&) ), this, SLOT(setFinishedWithError(long,const QString&)) );
}

ListViewItem::~ListViewItem()
{
	delete m_job;
}

FileJob* ListViewItem::job() const
{
	return m_job;
}

void ListViewItem::enqueue( JobQueue* queue )
{
	if ( isQueueable() )
	{
		m_queue = queue;
		m_job->clearFinished();
		m_queue->enqueue( m_job );
		setText( ListView::status, i18n( "queued" ) );
	}
}

void ListViewItem::setProfile( const QString& profile )
{
	if ( m_job->profile().isEmpty() || isQueueable() )
	{
		m_job->setProfile( profile );
		setText( ListView::profile, profile );
		setText( ListView::status, m_job->hasProfileError() ? i18n( "profile error" ) : i18n( "profile set" ) );
	}
}

void ListViewItem::resetProfile()
{
	if ( isQueueable() )
	{
		m_job->resetProfile();
		setText( ListView::profile, m_job->profile() );
		setText( ListView::status, m_job->hasProfileError() ? i18n( "profile error" ) : i18n( "profile set" ) );
	}
}

void ListViewItem::stop()
{
	if ( m_queue != 0 )
	{
		bool ret = m_queue->remove( m_job );
		m_queue = 0;
		if ( ret ) // job was on the queue and is not anymore
			setCancelled();
	}

	// job was not on the queue (otherwise it would have been removed)
	// if it was running it will be stoped
	if ( m_job->isRunning() )
		m_job->stop();
}

void ListViewItem::remove()
{
	if ( m_queue != 0 )
		m_queue->remove( m_job );

	// job is not on the queue (if it was in it, it has been removed by now)

	m_job->stop(); // job signaled to end

	if ( ! m_job->isRunning() ) // job is not in progress (nor about to be), it's safe to delete it
// 		deleteLater();
		delete this;
	else // job is in progress (or about to be), so we wait for the cancelled signal to trigger the removal
		m_delayedDeletion = true;
}


bool ListViewItem::isQueueable() const
{
	return m_queue == 0 && ! m_job->hasProfileError();
}

bool ListViewItem::isEnqueued() const
{
	return m_queue != 0 && ! m_job->isRunning();
}

bool ListViewItem::isCancellable() const
{
	return m_queue != 0;
}

bool ListViewItem::isRefresheable() const
{
	return isQueueable();
}

bool ListViewItem::isRunning() const
{
	return m_job->isRunning();
}

bool ListViewItem::isFinishedOK() const
{
	return m_job->isFinishedOK();
}

bool ListViewItem::isFinishedWithError() const
{
	return m_job->isFinishedWithError();
}

void ListViewItem::setDstPath( long, const QString& dstPath )
{
	if ( dstPath.isEmpty() )
	{
		setText( ListView::dst_path, "" );
		setText( ListView::dst_dir, "" );
		setText( ListView::dst_file, "" );
	}
	else
	{
		QFileInfo fileInfo( dstPath );
		setText( ListView::dst_path, fileInfo.filePath() );
		setText( ListView::dst_dir, fileInfo.dirPath() );
		setText( ListView::dst_file, fileInfo.fileName() );
	}
}

void ListViewItem::setStarted()
{
	setProgress( 0, 0 );
}

void ListViewItem::setProgress( long, int progress )
{
	setText( ListView::status, QString::number( progress ) + "%" );
}

void ListViewItem::setFinishedOK()
{
	m_queue = 0;
	setProgress( 0, 100 );
	((ListView*)listView())->emit jobFinishedOK( m_job->id() );
}

void ListViewItem::setFinishedWithError( long, const QString& errorMsg )
{
	m_queue = 0;
	setText( ListView::status, errorMsg );
	((ListView*)listView())->emit jobFinishedWithError( m_job->id() );
}

void ListViewItem::setCancelled()
{
	if ( m_delayedDeletion )
		delete this;
	else
		setText( ListView::status, i18n( "profile set" ) );
	((ListView*)listView())->emit jobCancelled( m_job->id() );
}



int ListViewItem::compare( QListViewItem* item, int, bool ascending ) const
{
	ListView* list = (ListView*)listView();

	int ret = 0;
	for ( unsigned idx = 0; idx < list->m_sortColumns.size(); ++idx )
		if ( (ret = compareAux( item, idx, ascending )) )
			break;

	return ret;
}

int ListViewItem::compareAux( QListViewItem* item, int column, bool ascending ) const
{
	if ( item->rtti() != TRANSKODE )
		return KListViewItem::compare( item, column, ascending );

// 	if ( column == ListView::status )
// 	{
// 		if ( m_job->progress() >= 0 )
// 		{
// 			if ( ((ListViewItem*)item)->m_job->progress() >= 0 )
// 				return m_job->progress() - ((ListViewItem*)item)->m_job->progress();
// 			else
// 				return 1;
// 		}
// 		else if ( ((ListViewItem*)item)->m_job->progress() >= 0 )
// 			return -1;
// 	}

	return KListViewItem::compare( item, column, ascending );
}

// reimplemented to draw progress bar
void ListViewItem::paintCell( QPainter* p, const QColorGroup &cg, int column, int width, int align )
{
	QColorGroup auxcg( cg );
	QColor c = auxcg.text();

	if ( isFinishedWithError() )
		auxcg.setColor( QColorGroup::Text, Qt::red );
	else if ( isFinishedOK() )
		auxcg.setColor( QColorGroup::Text, Qt::gray );

	int h = height();
// 	int margin = ((ListView*)listView())->itemHorizontalMargin();
// 	int w = width - 2*margin;

	// clean background
	p->fillRect( 0, 0, width, h, isSelected() ? auxcg.highlight() : auxcg.base() );
	p->setPen( isSelected() ? auxcg.highlightedText() : auxcg.text() );

	bool running = isRunning();
	if ( p->font().bold() != running )
	{
		QFont font( p->font() );
		font.setBold( running );
		p->setFont( font );
	}

	switch ( column )
	{
		case ListView::src_path:
		case ListView::src_dir:
		case ListView::src_file:
		case ListView::dst_path:
		case ListView::dst_dir:
		case ListView::dst_file:
		{
/*			if ( p->boundingRect( margin, 0, w, h, align|Qt::SingleLine|Qt::ExpandTabs, text( column ) ).width() > w )
			{
				align ^= Qt::AlignLeft; // disable left align
				align |= Qt::AlignRight; // enable right align
			}*/
			break;
		}
		case ListView::status:
		case ListView::profile:
		default:
			break;
	}

	KListViewItem::paintCell( p, auxcg, column, width, align );
}

#include "listviewitem.moc"
