/***************************************************************************
 *   Copyright (C) 2006 by Sergio Pistone                                  *
 *   sergio_pistone@yahoo.com.ar                                           *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "mainwindow.h"
#include "listviewitem.h"

#include <qlayout.h>
#include <qcursor.h>
#include <qtooltip.h>
#include <qevent.h>

#include <kapplication.h>
#include <kfiledialog.h>
#include <kglobal.h>
#include <kstandarddirs.h>
#include <klocale.h>
#include <kaction.h>
#include <kstdaction.h>
#include <kshortcut.h>
#include <kstdaccel.h>
#include <kstatusbar.h>
#include <kconfig.h>
#include <kurl.h>
#include <kmessagebox.h>
#include <kpopupmenu.h>
#include <klistviewsearchline.h>

using namespace TransKode;

MainWindow::MainWindow():
	KMainWindow( 0, "MainWindow" ),
	m_logger( Logger::instance() ),
	m_config( ::locate( "appdata", "defaultsrc" ) ),
	m_queue(),
	m_configDialog( ::locate( "appdata", "defaultsrc" ), false, this ),
	m_systemTray( new KSystemTray( this ) ),
	m_lastDir( QDir::homeDirPath() )
{
	QSplitter* splitter = new QSplitter( this );
	splitter->setOrientation( Qt::Vertical );

	m_listView = new ListView( splitter, m_config, m_queue );
	m_metadataWidget = new MetadataWidget( splitter );

	splitter->setCollapsible( m_listView, false );
	QValueList<int> heights;
	heights << 2000 << 50;
	splitter->setSizes( heights );

// 	splitter->setResizeMode( m_metadataWidget, QSplitter::KeepSize );

	setCentralWidget( splitter ); // tell the KMainWindow that this is indeed the main widget
	setupActions(); // then, setup our actions
// 	statusBar()->show(); // a status bar
	toolBar()->show(); // and a tool bar

	setupGUI();

	updateProcessingState();

	// For some strange reason this #1) has to be plugged into something
	// for it to work and #2) has to be plugged after setupGUI execution
	m_searchManagerAction->plug( new KPopupMenu( this ) );

	m_config.load( ::locate( "appdata", "transkoderc" ) );
	m_configDialog.setConfig( m_config );

	m_logger.setFile( m_config.generalOption( GeneralOpt::log ), true );
	m_logger.setMsgPrefix( "[transkode] " );

	m_systemTray->setPixmap( KSystemTray::loadIcon( "transkode" ) );
	if ( m_config.generalOption( GeneralOpt::show_tray ) == "true" )
		m_systemTray->setShown( true );

	connect( m_systemTray, SIGNAL( quitSelected() ), this, SLOT( showConfirmQuit() ) );
	connect( &m_configDialog, SIGNAL( configSaved() ), this, SLOT( updateConfigFromDialog() ) ) ;

	connect( m_listView, SIGNAL( selectionChanged() ), this, SLOT( updateActionsState() ) );
	connect( m_listView, SIGNAL( currentChanged(QListViewItem*) ), this, SLOT( updateStatusBar() ) );

	connect( m_listView, SIGNAL( jobFinishedOK(long) ), this, SLOT( updateActionsState() ) );
// 	connect( m_listView, SIGNAL( jobFinishedOK(long) ), this, SLOT( updateProcessingState() ) );
// 	connect( m_listView, SIGNAL( jobFinishedWithError(long) ), this, SLOT( updateProcessingState() ) );
// 	connect( m_listView, SIGNAL( jobCancelled(long) ), this, SLOT( updateProcessingState() ) );

	connect( m_listView, SIGNAL( selectionChanged() ), this, SLOT( loadSelectedItemsMetadata() ) );
	connect( m_metadataWidget, SIGNAL( metadataSetted() ), this, SLOT( saveSelectedItemsMetadata() ) );

	connect( &m_queue, SIGNAL( workersBusy(long) ), this, SLOT( updateProcessingState() ) );
	connect( &m_queue, SIGNAL( workersIdle(long) ), this, SLOT( updateProcessingState() ) );
	connect( &m_queue, SIGNAL( queueStarted(long,long) ), this, SLOT( updateProcessingState() ) );
	connect( &m_queue, SIGNAL( queueDepleted(long,long) ), this, SLOT( updateProcessingState() ) );

	kdDebug() << "starting workers" << endl;

	int threads = m_config.generalOption( GeneralOpt::threads ).toInt();
	if ( threads < 1 || threads > 8 )
		threads = 2;
	for ( int idx = 0; idx < threads; ++idx )
		new Worker( m_queue );

	loadConfig();
}

MainWindow::~MainWindow()
{
}

void MainWindow::setupActions()
{
	KStdAction::quit( this, SLOT( showConfirmQuit() ), actionCollection() );
	KStdAction::preferences( this, SLOT( showPreferences() ), actionCollection() );

	KAction* trayConfigure = new KAction(
		i18n( "&Configure..." ),
 		"configure",					// Action icon
		KShortcut( "" ),				// Action accelerator
		this,							// Action receiver
		SLOT( showPreferences() ),		// Action slot
		actionCollection(),				// Action collection manager
		"tray_configure" );				// Action name/identifier
	trayConfigure->plug( m_systemTray->contextMenu(), 1 );

	// Create a toolBar named 'searchToolBar' and add a search widget to it
	m_searchAction = new KWidgetAction(
		new KListViewSearchLineWidget( m_listView, toolBar( "searchToolBar" ) ),
		i18n( "Search" ),				// Action text
 		"location_edit",				// Action icon
		this,							// Action receiver
		0,								// Action slot
		actionCollection(),				// Action collection manager
		"search" );						// Action name/identifier
	m_searchAction->plug( toolBar( "searchToolBar" ) );

	// Create an action asociated to the toolBar created above so we can enable/disable the filter
	m_searchManagerAction = new KToggleToolBarAction(
		"searchToolBar",
		"Toggle Search",
		actionCollection(),
		"search_manager" );
	connect( m_searchManagerAction, SIGNAL( toggled(bool) ), this, SLOT( toggleToolBarFilterEnabled(bool) ) );

	new KAction(
		i18n( "&Add Files" ),			// Action text
 		"add",							// Action icon
		KShortcut( "Insert" ),			// Action accelerator
		this,							// Action receiver
		SLOT( add() ),					// Action slot
		actionCollection(),				// Action collection manager
		"add" );						// Action name/identifier

	new KAction(
		i18n( "&Remove Files" ),		// Action text
 		"button_cancel",				// Action icon
		KShortcut( "Delete" ),			// Action accelerator
		this,							// Action receiver
		SLOT( removeSelected() ),		// Action slot
		actionCollection(),				// Action collection manager
		"remove" );						// Action name/identifier

	m_selectAllAction = new KAction(
		i18n( "&Select All" ),			// Action text
 		"player_playlist",				// Action icon
		KStdAccel::SelectAll,			// Action accelerator
		this,							// Action receiver
		SLOT( selectAll() ),			// Action slot
		actionCollection(),				// Action collection manager
		"select_all" );					// Action name/identifier
	m_selectAllAction->setEnabled( false );

	m_invertSelectionAction = new KAction(
		i18n( "&Invert Selection" ),	// Action text
		KShortcut( "Ctrl+I" ),			// Action accelerator
		m_listView,						// Action receiver
		SLOT( invertSelection() ),		// Action slot
		actionCollection(),				// Action collection manager
		"invert_selection" );			// Action name/identifier
	m_invertSelectionAction->setEnabled( false );

	m_clearAllAction = new KAction(
		i18n( "&Clear" ),				// Action text
		"view_remove",					// Action icon
		KShortcut( "Ctrl+Delete" ),		// Action accelerator
		this,							// Action receiver
		SLOT( removeAll() ),			// Action slot
		actionCollection(),				// Action collection manager
		"clear" );						// Action name/identifier
	m_clearAllAction->setEnabled( false );

	m_clearFinishedAction = new KAction(
		i18n( "Clear &Finished" ),		// Action text
		"button_ok",					// Action icon
		KShortcut( "Ctrl+F" ),			// Action accelerator
		this,							// Action receiver
		SLOT( removeFinished() ),		// Action slot
		actionCollection(),				// Action collection manager
		"clear_finished" );				// Action name/identifier
	m_clearFinishedAction->setEnabled( false );

	m_refreshFilesAction = new KAction(
		i18n( "&Refresh" ),				// Action text
		"reload"	,					// Action icon
		KShortcut( "Ctrl+R" ),			// Action accelerator
		this,							// Action receiver
		SLOT( refreshSelected() ),		// Action slot
		actionCollection(),				// Action collection manager
		"refresh" );					// Action name/identifier
	m_refreshFilesAction->setEnabled( false );

	m_enqueueFilesAction = new KAction(
		i18n( "En&queue" ),				// Action text
 		"player_play",					// Action icon
		KShortcut( "Ctrl+Q" ),			// Action accelerator
		this,							// Action receiver
		SLOT( enqueueSelected() ),		// Action slot
		actionCollection(),				// Action collection manager
		"queue" );						// Action name/identifier
	m_enqueueFilesAction->setEnabled( false );

	m_dequeueFilesAction = new KAction(
		i18n( "&Cancel" ),				// Action text
 		"player_stop",					// Action icon
		KShortcut( "Ctrl+D" ),			// Action accelerator
		this,							// Action receiver
		SLOT( dequeueSelected() ),		// Action slot
		actionCollection(),				// Action collection manager
		"cancel" );						// Action name/identifier
	m_dequeueFilesAction->setEnabled( false );

	m_pauseEncodingAction = new KToggleAction(
		i18n( "&Pause" ),				// Action text
		"player_pause",					// Action icon
		KShortcut( "Ctrl+P" ),			// Action accelerator
		this,							// Action receiver
		SLOT( togglePaused() ),	// Action slot
		actionCollection(),				// Action collection manager
		"pause" );						// Action name/identifier
	m_pauseEncodingAction->plug( m_systemTray->contextMenu(), 1 );
}


void MainWindow::updateProcessingState()
{
	bool isIdle = m_queue.allWorkersIdle();
	bool isPaused = m_pauseEncodingAction->isChecked();
	unsigned queuedJobs = m_queue.jobsCount();

	if ( isIdle && ! queuedJobs )
	{
		QToolTip::add( m_systemTray, "transKode" );
		setCaption( "" );
	}
	else if ( ! isIdle && ! isPaused )
	{
		QString caption = i18n( "Transcoding" );
		QToolTip::add( m_systemTray, "transKode - " + caption );
		setCaption( caption );
	}
	else
	{
		QString caption = i18n( "Paused" );
		QToolTip::add( m_systemTray, "transKode - " + caption );
		setCaption( caption );
	}
}

void MainWindow::updateActionsState()
{
	bool hasItems = m_listView->childCount() > 0;
	bool hasQueueableItems = false;
	bool hasEnqueuedItems = false;
	bool hasRunningItems = false;
	bool hasRefresheableItems = false;
	bool hasFinishedOKItems = false;

	if ( hasItems )
	{
		ListViewItem* item = 0;
		for ( QListViewItemIterator it( m_listView ) ; it.current() != 0; ++it )
		{
			item = (ListViewItem*)it.current();

			if ( item->isSelected() )
			{
				if ( item->isRunning() ) hasRunningItems = true;
				if ( item->isEnqueued() ) hasEnqueuedItems = true;
				if ( item->isQueueable() ) hasQueueableItems = true;
				if ( item->isRefresheable() ) hasRefresheableItems = true;
			}
			if ( item->isFinishedOK() ) hasFinishedOKItems = true;

			if ( hasEnqueuedItems && hasQueueableItems && hasRefresheableItems && hasFinishedOKItems )
				break;
		}
	}

	m_enqueueFilesAction->setEnabled( hasQueueableItems );
	m_dequeueFilesAction->setEnabled( hasEnqueuedItems || hasRunningItems );
	m_refreshFilesAction->setEnabled( hasRefresheableItems );
	m_clearFinishedAction->setEnabled( hasFinishedOKItems );
	m_selectAllAction->setEnabled( hasItems );
	m_clearAllAction->setEnabled( hasItems );
}

void MainWindow::loadSelectedItemsMetadata()
{
	MetadataList metadatas;

	ListViewItem* item = 0;
	for ( QListViewItemIterator it( m_listView, QListViewItemIterator::Selected ) ; it.current() != 0; ++it )
	{
		item = (ListViewItem*)it.current();
		if ( item->isCancellable() )
			continue;

		metadatas.append( item->job()->metadata() );
	}

	if ( metadatas.empty() )
		m_metadataWidget->setEmptyMetadata();
	else
		m_metadataWidget->setMetadata( metadatas );
}

void MainWindow::saveSelectedItemsMetadata()
{
	Metadata md = m_metadataWidget->metadata();

	ListViewItem* item = 0;
	for ( QListViewItemIterator it( m_listView, QListViewItemIterator::Selected ) ; it.current() != 0; ++it )
	{
		item = (ListViewItem*)it.current();
		if ( item->isCancellable() )
			continue;

		item->job()->setMetadata( md, true );
	}
}

void MainWindow::updateStatusBar()
{
	QString message;

	ListViewItem* currentItem = (ListViewItem*)m_listView->currentItem();

	if ( currentItem == 0 )
		message = "";
	else
	{
		const FileJob* job = currentItem->job();

		if ( job->profile().isEmpty() )
			message = job->srcPath();
		else
		{
			EncodingProfile* opts = m_config.profileOptions( job->profile() );
			if ( opts != 0 )
			{
				message = job->srcPath() + " -> " + job->dstPath() + " | " + job->profile() + " [" + opts->switches() + "]";
				delete opts;
			}
			else
				message = job->srcPath();
		}
	}

	statusBar()->message( message );
}

void MainWindow::togglePaused()
{
	if ( m_queue.isPaused() )
	{
		m_queue.resume();

		m_pauseEncodingAction->setText( i18n( "Pause" ) );
		m_pauseEncodingAction->setToolTip( i18n( "Pause" ) );
	}
	else
	{
		m_queue.pause();

		m_pauseEncodingAction->setText( i18n( "Resume" ) );
		m_pauseEncodingAction->setToolTip( i18n( "Resume" ) );
	}

	updateProcessingState();
}


void MainWindow::filesInDir( const QString& directory, QStringList& files /*where files are returned*/ )
{
	static const QStringList extensions = buildSupportedFormatsExtensions();

	QDir dir( directory );
	QFileInfo dirInfo( dir.absPath() );

	QString dirPath = dir.absPath();
	QString dirPathParent = dirInfo.dirPath( true );

	const QFileInfoList* filesInfo = dir.entryInfoList( QDir::Files|QDir::Dirs|QDir::Readable|QDir::Hidden|QDir::NoSymLinks );
	for ( QFileInfoListIterator it( *filesInfo ); it.current() != 0; ++it )
	{
		QString curPath = it.current()->absFilePath();
		if ( curPath == dirPath || curPath == dirPathParent )
			continue;

		if ( it.current()->isFile() )
		{
			if ( extensions.contains( it.current()->extension( false ).lower() ) )
				files.append( it.current()->absFilePath() );
		}
		else if ( it.current()->isDir() )
			filesInDir( it.current()->absFilePath(), files );
	}
}

QMap<QString,QStringList> MainWindow::supportedFormatsMap()
{
	// NOTE: place all extensions here on lowercase

	QMap<QString,QStringList> ret;
	ret["3GPP Multimedia File"] = "3gp";
	ret["Advanced Audio Coding"] = "aac";
	ret["Advanced Streaming Format"] = QStringList::split( " ", "asf asx" );
	ret["Bonk Audio"] = "bonk";
	ret["Digital Theater System"] = "dts";
	ret["DVD-Video Object"] = "vob";
	ret["Dolby Digital"] = "ac3";
	ret["Free Lossless Audio Codec"] = QStringList::split( " ", "flac ogg" );
	ret["Flash Video"] = "flv";
	ret["Lossless Audio"] = "la";
	ret["Lossless Predictive Audio Compression"] = "pac";
	ret["MPEG-1 Layer 1 Audio"] = "mp1";
	ret["MPEG-1 Layer 2 Audio"] = "mp2";
	ret["MPEG-1 Layer 3 Audio"] = "mp3";
	ret["MPEG-1 System Stream"] = QStringList::split( " ", "mpg mpeg" );
	ret["MPEG-4"] = QStringList::split( " ", "mp4 m4a m4b" );
	ret["mp3PRO"] = "mp3";
	ret["Matroska Audio"] = "mka";
	ret["Monkey's Audio"] = QStringList::split( " ", "ape mac" );
	ret["Musepack"] = QStringList::split( " ", "mpc mpp mp+" );
	ret["Ogg Vorbis"] = "ogg";
	ret["OptimFROG"] = "ofr";
	ret["QuickTime Movie"] = QStringList::split( " ", "mov qt" );
	ret["Real Audio"] = QStringList::split( " ", "ra ram" );
	ret["Real Media"] = QStringList::split( " ", "rv rm rmvb" );
	ret["Shorten Audio"] = "shn";
	ret["Speex"] = "spx";
	ret["True Audio"] = "tta";
	ret["WavPack"] = "wv";
	ret["Windows Media Audio"] = "wma";
	ret["Windows Media Video"] = "wmv";
	ret["Waveform Audio Format"] = "wav";
	ret["Audio Video Interleave"] = "avi";
	return ret;
}

QStringList MainWindow::buildSupportedFormatsExtensions()
{
	QStringList extensions;
	QMap<QString,QStringList> supportedFormatsMap = MainWindow::supportedFormatsMap();
	for ( QMap<QString,QStringList>::Iterator it = supportedFormatsMap.begin(); it != supportedFormatsMap.end(); ++it )
		for ( QStringList::Iterator extIt = it.data().begin(); extIt != it.data().end(); ++extIt )
			if ( ! extensions.contains( *extIt ) )
				extensions.append( *extIt );
	return extensions;
}

QString MainWindow::buildSupportedFormatsFilter()
{
	QString filter;
	QString allSupported;
	QMap<QString,QStringList> supportedFormatsMap = MainWindow::supportedFormatsMap();
	for ( QMap<QString,QStringList>::Iterator it = supportedFormatsMap.begin(); it != supportedFormatsMap.end(); ++it )
	{
		QString extensions;
		for ( QStringList::Iterator extIt = it.data().begin(); extIt != it.data().end(); ++extIt )
			extensions += "*." + *extIt + " *." + (*extIt).upper();
		allSupported += " " + extensions;
		filter += "\n" + extensions + "|" + i18n( it.key() );
	}

	filter = allSupported + "|" + i18n( "All Supported Files" ) + filter;

	return filter.stripWhiteSpace();
}

void MainWindow::add()
{
	static QString filter( buildSupportedFormatsFilter() );

	QStringList files = KFileDialog::getOpenFileNames( m_lastDir, filter, this, i18n( "Add files" ) );
	if ( ! files.isEmpty() )
	{
		QFileInfo fileInfo( files[0] );
		m_lastDir = fileInfo.dirPath( true );

		ListViewItem* item = 0;
		for ( QStringList::Iterator it = files.begin(); it != files.end(); ++it )
		{
			fileInfo.setFile( *it );
			if ( fileInfo.isFile() )
				item = m_listView->add( *it, item );
			else if ( fileInfo.isDir() )
			{
				QStringList nestedFiles;
				filesInDir( *it, nestedFiles );
				for ( QStringList::Iterator nestedIt = nestedFiles.begin(); nestedIt != nestedFiles.end(); ++nestedIt )
					item = m_listView->add( *nestedIt, item );
			}
		}

		updateActionsState();
	}
}

void MainWindow::removeSelected()
{
	m_listView->removeSelected();
	updateActionsState();

	if ( m_listView->childCount() == 0 )
		m_metadataWidget->setEmptyMetadata();
}

void MainWindow::removeFinished()
{
	m_listView->removeFinished();
	updateActionsState();

	if ( m_listView->childCount() == 0 )
		m_metadataWidget->setEmptyMetadata();
}

void MainWindow::removeAll()
{
	m_listView->removeAll();
	updateActionsState();

	m_metadataWidget->setEmptyMetadata();
}

void MainWindow::enqueueSelected()
{
	m_listView->enqueueSelected();
	updateActionsState();
}

void MainWindow::dequeueSelected()
{
	m_listView->dequeueSelected();
	updateActionsState();
}

void MainWindow::selectAll()
{
	m_listView->selectAll();
// 	updateActionsState();
}

void MainWindow::refreshSelected()
{
	m_listView->refreshSelected();
	updateActionsState();
}



void MainWindow::showPreferences()
{
// 	QSize size = m_configDialog.size();
// 	m_configDialog.setInitialSize( size );
	m_configDialog.show();
}

void MainWindow::showConfirmQuit()
{
	if ( ! m_queue.allWorkersIdle() )
		if ( KMessageBox::Yes != KMessageBox::questionYesNo(
				0,
				i18n( "There are pending jobs. Really quit?" ),
				i18n( "Confirm action" ) + " - transKode" ) )
			return;

	saveConfig();
	kapp->quit();
}

void MainWindow::toggleToolBarFilterEnabled( bool enabled )
{
	KListViewSearchLineWidget* searchLineWidget = (KListViewSearchLineWidget*)m_searchAction->widget();
	if ( ! searchLineWidget )
		return;

	KListViewSearchLine* searchLine = searchLineWidget->searchLine();
	if ( ! searchLine )
		return;

	// disable filtering when toolbar is hidden
	searchLine->updateSearch( enabled ? QString::null : "" );
}

void MainWindow::updateConfigFromDialog()
{
	m_config = m_configDialog.getConfig();
	m_config.save( ::locateLocal( "appdata", "transkoderc" ) );

	m_logger.setFile( m_config.generalOption( GeneralOpt::log ), true );

	bool showTray = m_config.generalOption( GeneralOpt::show_tray ) == "true";

	if ( showTray )
		m_systemTray->show();
	else
	{
		m_systemTray->hide();
		show();
	}
}


bool MainWindow::queryClose()
{
	if ( m_config.generalOption( GeneralOpt::show_tray ) == "true" )
	{
		hide();
		return false;
	}
	else
	{
		if ( ! m_queue.allWorkersIdle() )
			if ( KMessageBox::Yes != KMessageBox::questionYesNo(
					0,
					i18n( "There are pending jobs. Really quit?" ),
					i18n( "Confirm action - " ) + "transKode" ) )
				return false;

		saveConfig();
		return true;
	}
}

void MainWindow::saveConfig()
{
	KConfigGroupSaver saver( kapp->config(), "MainWindow Settings" );

	kapp->config()->writePathEntry( "LastDir", m_lastDir );

	m_listView->saveConfig();
}

void MainWindow::loadConfig()
{
	KConfigGroupSaver saver( kapp->config(), "MainWindow Settings" );

	m_lastDir = kapp->config()->readPathEntry( "LastDir", QDir::homeDirPath() );

	m_listView->loadConfig();
}

#include "mainwindow.moc"
