/***************************************************************************
 *   Copyright (C) 2006 by Sergio Pistone                                  *
 *   sergio_pistone@yahoo.com.ar                                           *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "metadatawidget.h"

#include <qcheckbox.h>
#include <klineedit.h>
#include <knuminput.h>
#include <kcombobox.h>
#include <kpushbutton.h>
#include <ktextedit.h>

using namespace TransKode;

MetadataWidget::MetadataWidget( QWidget* parent, const char* name ):
	MetadataWidgetBase( parent, name ),
	m_metadata()
{
	connect( titleLineEdit, SIGNAL( textChanged(const QString&) ), this, SLOT( setButtonsEnabled() ) );
	connect( artistLineEdit, SIGNAL( textChanged(const QString&) ), this, SLOT( setButtonsEnabled() ) );
	connect( albumLineEdit, SIGNAL( textChanged(const QString&) ), this, SLOT( setButtonsEnabled() ) );
	connect( discNumberSpinBox, SIGNAL( valueChanged(const QString&) ), this, SLOT( setButtonsEnabled() ) );
	connect( trackSpinBox, SIGNAL( valueChanged(const QString&) ), this, SLOT( setButtonsEnabled() ) );
	connect( yearSpinBox, SIGNAL( valueChanged(const QString&) ), this, SLOT( setButtonsEnabled() ) );
	connect( composerLineEdit, SIGNAL( textChanged(const QString&) ), this, SLOT( setButtonsEnabled() ) );
	connect( genreComboBox, SIGNAL( textChanged(const QString&) ), this, SLOT( setButtonsEnabled() ) );
	connect( commentTextEdit, SIGNAL( textChanged() ), this, SLOT( setButtonsEnabled() ) );

	connect( setButton, SIGNAL( clicked() ), this, SLOT( set() ) );
	connect( resetButton, SIGNAL( clicked() ), this, SLOT( reset() ) );

	connect( commentCheckBox, SIGNAL( toggled(bool) ), this, SLOT( setCommentTextEditEditable(bool) ) );
	setCommentTextEditEditable( false );

	artistLineEdit->completionObject()->setOrder( KCompletion::Sorted );
	albumLineEdit->completionObject()->setOrder( KCompletion::Sorted );
	titleLineEdit->completionObject()->setOrder( KCompletion::Sorted );
	composerLineEdit->completionObject()->setOrder( KCompletion::Sorted );

	setEmptyMetadata();
}

MetadataWidget::~MetadataWidget()
{
}

void MetadataWidget::setGenresList( const QStringList& genres )
{
	genreComboBox->clear();
	genreComboBox->insertStringList( genres );
}

Metadata MetadataWidget::metadata()
{
	return m_metadata;
}

void MetadataWidget::setEmptyMetadata()
{
	MetadataList metadatas;
	setMetadata( metadatas );
}

void MetadataWidget::setMetadata( const Metadata& metadata )
{
	MetadataList metadatas;
	metadatas.append( metadata );
	setMetadata( metadatas );
}

void MetadataWidget::setMetadata( const MetadataList& metadatas )
{
	KCompletion* artistComp = artistLineEdit->completionObject();
	KCompletion* albumComp = albumLineEdit->completionObject();
	KCompletion* titleComp = titleLineEdit->completionObject();
	KCompletion* composerComp = composerLineEdit->completionObject();

	artistComp->clear();
	albumComp->clear();
	titleComp->clear();
	composerComp->clear();

	if ( metadatas.empty() )
	{
		setEnabled( false );
		setCheckBoxsVisible( false );
		for ( int idx = 0; idx < Metadata::SIZE; ++idx )
			m_metadata[idx] = QString::null;
	}
	else if ( metadatas.size() == 1 )
	{
		setEnabled( true );
		setCheckBoxsVisible( false );
		Metadata metadata = metadatas.first();
		for ( int idx = 0; idx < Metadata::SIZE; ++idx )
			m_metadata[idx] = metadata[idx].isNull() ? "" : metadata[idx];
	}
	else // if ( metadatas.size() > 1 )
	{
		setEnabled( true );
		setCheckBoxsVisible( true );

		MetadataList::const_iterator it = metadatas.begin();

		for ( int idx = 0; idx < Metadata::SIZE; ++idx )
			m_metadata[idx] = (*it)[idx];

		if ( ! m_metadata[Metadata::artist].isEmpty() )
			artistComp->addItem( m_metadata[Metadata::artist] );
		if ( ! m_metadata[Metadata::album].isEmpty() )
			albumComp->addItem( m_metadata[Metadata::album] );
		if ( ! m_metadata[Metadata::title].isEmpty() )
			titleComp->addItem( m_metadata[Metadata::title] );
		if ( ! m_metadata[Metadata::composer].isEmpty() )
			composerComp->addItem( m_metadata[Metadata::composer] );

		for ( ; it != metadatas.end(); ++it )
		{
			const Metadata& metadata = *it;
			for ( int idx = 0; idx < Metadata::SIZE; ++idx )
			{
				if ( m_metadata[idx] != metadata[idx] )
				{
					setCheckBoxChecked( (Metadata::Tag)idx, false );
					m_metadata[idx] = QString::null;
				}
			}

			if ( ! metadata[Metadata::artist].isEmpty() )
				artistComp->addItem( metadata[Metadata::artist] );
			if ( ! metadata[Metadata::album].isEmpty() )
				albumComp->addItem( metadata[Metadata::album] );
			if ( ! metadata[Metadata::title].isEmpty() )
				titleComp->addItem( metadata[Metadata::title] );
			if ( ! metadata[Metadata::composer].isEmpty() )
				composerComp->addItem( metadata[Metadata::composer] );
		}
	}

	reset();
}

void MetadataWidget::set()
{
	if ( artistCheckBox->isChecked() )
		m_metadata[Metadata::artist] = artistLineEdit->text();

	if ( yearCheckBox->isChecked() )
		m_metadata[Metadata::year] = yearSpinBox->value() > 0 ? QString::number( yearSpinBox->value() ) : "";

	if ( albumCheckBox->isChecked() )
		m_metadata[Metadata::album] = albumLineEdit->text();

	if ( discNumberCheckBox->isChecked() )
		m_metadata[Metadata::discnum] = discNumberSpinBox->value() > 0 ? QString::number( discNumberSpinBox->value() ) : "";

	if ( trackCheckBox->isChecked() )
		m_metadata[Metadata::track] = trackSpinBox->value() > 0 ? QString::number( trackSpinBox->value() ) : "";

	if ( titleCheckBox->isChecked() )
		m_metadata[Metadata::title] = titleLineEdit->text();

	if ( genreCheckBox->isChecked() )
		m_metadata[Metadata::genre] = genreComboBox->lineEdit()->text();

	if ( composerCheckBox->isChecked() )
		m_metadata[Metadata::comment] = commentTextEdit->text();

	if ( commentCheckBox->isChecked() )
		m_metadata[Metadata::composer] = composerLineEdit->text();

	setButtonsDisabled();

	emit metadataSetted();
}

void MetadataWidget::reset()
{
	artistLineEdit->setText( m_metadata[Metadata::artist] );
	setCheckBoxChecked( Metadata::artist, ! m_metadata[Metadata::artist].isNull() );

	yearSpinBox->setValue( m_metadata[Metadata::year].toInt() );
	setCheckBoxChecked( Metadata::year, ! m_metadata[Metadata::year].isNull() );

	albumLineEdit->setText( m_metadata[Metadata::album] );
	setCheckBoxChecked( Metadata::album, ! m_metadata[Metadata::album].isNull() );

	discNumberSpinBox->setValue( m_metadata[Metadata::discnum].toInt() );
	setCheckBoxChecked( Metadata::discnum, ! m_metadata[Metadata::discnum].isNull() );

	trackSpinBox->setValue( m_metadata[Metadata::track].toInt() );
	setCheckBoxChecked( Metadata::track, ! m_metadata[Metadata::track].isNull() );

	titleLineEdit->setText( m_metadata[Metadata::title] );
	setCheckBoxChecked( Metadata::title, ! m_metadata[Metadata::title].isNull() );

	genreComboBox->lineEdit()->setText( m_metadata[Metadata::genre] );
	setCheckBoxChecked( Metadata::genre, ! m_metadata[Metadata::genre].isNull() );

	composerLineEdit->setText( m_metadata[Metadata::composer] );
	setCheckBoxChecked( Metadata::composer, ! m_metadata[Metadata::composer].isNull() );

	commentTextEdit->setText( m_metadata[Metadata::comment] );
	setCheckBoxChecked( Metadata::comment, ! m_metadata[Metadata::comment].isNull() );

	setButtonsDisabled();
}

void MetadataWidget::setCommentTextEditEditable( bool editable )
{
	commentTextEdit->setReadOnly( ! editable );
}

void MetadataWidget::setCheckBoxsVisible( bool visible )
{
	if ( visible )
	{
		artistCheckBox->show();
		yearCheckBox->show();
		albumCheckBox->show();
		discNumberCheckBox->show();
		trackCheckBox->show();
		titleCheckBox->show();
		genreCheckBox->show();
		composerCheckBox->show();
		commentCheckBox->show();
	}
	else
	{
		artistCheckBox->hide();
		yearCheckBox->hide();
		albumCheckBox->hide();
		discNumberCheckBox->hide();
		trackCheckBox->hide();
		titleCheckBox->hide();
		genreCheckBox->hide();
		composerCheckBox->hide();
		commentCheckBox->hide();
	}
}

void MetadataWidget::setCheckBoxChecked( Metadata::Tag tag, bool checked )
{
	switch ( tag )
	{
		case Metadata::artist:
			artistCheckBox->setChecked( checked );
			artistLineEdit->setEnabled( checked );
			break;
		case Metadata::year:
			yearCheckBox->setChecked( checked );
			yearSpinBox->setEnabled( checked );
			break;
		case Metadata::album:
			albumCheckBox->setChecked( checked );
			albumLineEdit->setEnabled( checked );
			break;
		case Metadata::discnum:
			discNumberCheckBox->setChecked( checked );
			discNumberSpinBox->setEnabled( checked );
			break;
		case Metadata::track:
			trackCheckBox->setChecked( checked );
			trackSpinBox->setEnabled( checked );
			break;
		case Metadata::title:
			titleCheckBox->setChecked( checked );
			titleLineEdit->setEnabled( checked );
			break;
		case Metadata::genre:
			genreCheckBox->setChecked( checked );
			genreComboBox->lineEdit()->setEnabled( checked );
			break;
		case Metadata::composer:
			composerCheckBox->setChecked( checked );
			composerLineEdit->setEnabled( checked );
			break;
		case Metadata::comment:
			commentCheckBox->setChecked( checked );
			commentTextEdit->setEnabled( checked );
			setCommentTextEditEditable( checked );
			break;
		default:
			break;
	}
}

void MetadataWidget::setCheckBoxsChecked( bool checked )
{
	for ( int idx = 0; idx < Metadata::SIZE; ++idx )
		setCheckBoxChecked( (Metadata::Tag)idx, checked );
}

void MetadataWidget::setButtonsEnabled( bool enabled )
{
	setButton->setEnabled( enabled );
	resetButton->setEnabled( enabled );
}

void MetadataWidget::setButtonsEnabled()
{
	setButtonsEnabled( true );
}

void MetadataWidget::setButtonsDisabled()
{
	setButtonsEnabled( false );
}
