/***************************************************************************
					alsasound.cpp  -  description
						 -------------------
		begin                : Wed Jul 5 2000
		copyright            : (C) 2000 by Jozef Kosoru
		email                : jozef.kosoru@pobox.sk
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "control.h"
#include <cassert>
#include <cerrno>
#include <cstdlib>
#include <cstdio>
#include <endian.h>
#include <stdint.h>
#include <sstream>
#include <iostream>
#include <ostream>
#include <klocale.h>
#include "alsasound.h"
#include "tuneroptions.h"

AlsaSound::AlsaSound(TunerOptions* pOptions) throw(std::runtime_error, std::bad_alloc)
 : m_pAudioBuffer(0)
{
  	if (pOptions->m_bUsePcmString)
	{
		m_sPcmName = pOptions->m_sPcmName;
    }
	else
	{
        std::ostringstream pcmNameStream;
        pcmNameStream << "plughw:" << pOptions->m_iCardNumber << ',' << pOptions->m_iDeviceNumber;
    	m_sPcmName = pcmNameStream.str();
	}

#ifdef _DEBUG
	std::cerr << "[AlsaSound::AlsaSound()] sPcmName: " << m_sPcmName << std::endl;
#endif

	int iErr;

	// open pcm device
	const snd_pcm_stream_t stream = SND_PCM_STREAM_CAPTURE;
	if ((iErr = snd_pcm_open(&m_pcm_handle, m_sPcmName.c_str(), stream, SND_PCM_NONBLOCK)) < 0)
	{
		throw std::runtime_error(I18N_NOOP("ALSA error: Cannot open audio device"));
	}

	// set block mode
	if ((iErr = snd_pcm_nonblock(m_pcm_handle, 0)) < 0)
	{
       	snd_pcm_close(m_pcm_handle);
		throw std::runtime_error(I18N_NOOP("ALSA error: Cannot set nonblock mode"));
	}

	// allocate the hw params structure
	snd_pcm_hw_params_t* p_hw_params;
	if ((iErr = snd_pcm_hw_params_malloc(&p_hw_params)) < 0)
	{
       	snd_pcm_close(m_pcm_handle);
		throw std::runtime_error(I18N_NOOP("ALSA error: Cannot allocate hardware parameter structure"));
	}

	// init the default configuration space
	if ((iErr = snd_pcm_hw_params_any(m_pcm_handle, p_hw_params)) < 0)
	{
       	snd_pcm_close(m_pcm_handle);
		snd_pcm_hw_params_free(p_hw_params);
		throw std::runtime_error(I18N_NOOP("ALSA error: Cannot initialize hardware parameter structure"));
	}

	// set interleaved sample acces
	if ((iErr = snd_pcm_hw_params_set_access(m_pcm_handle, p_hw_params,
											 SND_PCM_ACCESS_RW_INTERLEAVED)) < 0)
	{
       	snd_pcm_close(m_pcm_handle);
		snd_pcm_hw_params_free(p_hw_params);
		throw std::runtime_error(I18N_NOOP("ALSA error: Cannot set access type"));
	}

	// set sample format (16bit signed Little/Big Endian or 8bit signed)
	const snd_pcm_format_t iFormat = (pOptions->m_iResolution == 8)
								   ? SND_PCM_FORMAT_S8
								   : (__BYTE_ORDER == __LITTLE_ENDIAN)
											? SND_PCM_FORMAT_S16_LE
											: SND_PCM_FORMAT_S16_BE;
    if ((iErr = snd_pcm_hw_params_set_format(m_pcm_handle, p_hw_params, iFormat)) < 0)
	{
        snd_pcm_close(m_pcm_handle);
		//snd_pcm_hw_params_free(p_hw_params);
		throw std::runtime_error(I18N_NOOP("ALSA error: Cannot set sample resolution"));
	}
//    snd_pcm_close(m_pcm_handle);
//	snd_pcm_hw_params_free(p_hw_params);
//    throw  std::runtime_error("xxx");

	// set samplarate
	unsigned iSamplerate = pOptions->m_iSamplerate;
	if ((iErr = snd_pcm_hw_params_set_rate_near(m_pcm_handle, p_hw_params, &iSamplerate, 0)) < 0)
	{
       	snd_pcm_close(m_pcm_handle);
		snd_pcm_hw_params_free(p_hw_params);
		throw std::runtime_error(I18N_NOOP("ALSA error: Cannot set sample rate"));
	}
	else
	{
		// store the real samplerate
		pOptions->m_iSamplerate_real = iSamplerate;
#ifdef _DEBUG
		std::cerr << "[AlsaSound::AlsaSound()] iSamplerate: " << iSamplerate << std::endl;
#endif
	}

	// set number of channels (mono)
	if ((iErr = snd_pcm_hw_params_set_channels(m_pcm_handle, p_hw_params, 1)) < 0)
	{
       	snd_pcm_close(m_pcm_handle);
		snd_pcm_hw_params_free(p_hw_params);
		throw std::runtime_error(I18N_NOOP("ALSA error: Cannot set channel count"));
	}

#if 0
	if ((iErr = snd_pcm_hw_params_set_periods(m_pcm_handle, p_hw_params, 2, 0)) < 0)
	{
       	snd_pcm_close(m_pcm_handle);
		snd_pcm_hw_params_free(p_hw_params);
		throw std::runtime_error(I18N_NOOP("ALSA error: Cannot set period size"));
	}
#endif

	// set buffer size
	unsigned iBufferTime = pOptions->m_iBufferTime * 1000;
	if ((iErr = snd_pcm_hw_params_set_buffer_time_near(m_pcm_handle, p_hw_params,
													   &iBufferTime, 0)) < 0)
	{
       	snd_pcm_close(m_pcm_handle);
		snd_pcm_hw_params_free(p_hw_params);
		throw std::runtime_error(I18N_NOOP("ALSA error: Cannot set buffer time"));
	}

	// store the real buffer size
	if ((iErr = snd_pcm_hw_params_get_buffer_size(p_hw_params, &m_iBufferSize)) < 0)
	{
       	snd_pcm_close(m_pcm_handle);
		snd_pcm_hw_params_free(p_hw_params);
		throw std::runtime_error(I18N_NOOP("ALSA error: Cannot get buffer size"));
	}
	else
	{
#ifdef _DEBUG
		std::cerr << "[AlsaSound::AlsaSound()] m_iBufferSize (in frames):"
				  << m_iBufferSize << std::endl;
#endif
		if (pOptions->m_iResolution == 16)
		{
			pOptions->m_iBufferSize_real = m_iBufferSize * 2;
		}
		else
		{
			pOptions->m_iBufferSize_real = m_iBufferSize;
		}
	}

	// set hw configuration
	if ((iErr = snd_pcm_hw_params(m_pcm_handle, p_hw_params)) < 0)
	{
       	snd_pcm_close(m_pcm_handle);
		snd_pcm_hw_params_free(p_hw_params);
		throw std::runtime_error(I18N_NOOP("ALSA error: Cannot set audio parameters"));
	}

	// deallocate hw param structure
	snd_pcm_hw_params_free(p_hw_params);

	// create the sample buffer
	m_pAudioBuffer =  new char[pOptions->m_iBufferSize_real];
	pOptions->m_pAudioBuffer = m_pAudioBuffer;

	// prepare the PCM device for capture
	if ((iErr = snd_pcm_prepare(m_pcm_handle)) < 0)
	{
       	snd_pcm_close(m_pcm_handle);
		m_pOptions->m_pAudioBuffer = 0;
		delete [] m_pAudioBuffer;
		throw std::runtime_error(I18N_NOOP("ALSA error: Cannot prepare audio interface for use."));
	}

	m_pOptions = pOptions;
}

AlsaSound::~AlsaSound()
{
	snd_pcm_close(m_pcm_handle);

	m_pOptions->m_pAudioBuffer = 0;
	delete [] m_pAudioBuffer;

#ifdef _DEBUG
	std::cerr << "[AlsaSound::~AlsaSound()]" << std::endl;
#endif
}

void AlsaSound::fillDeviceList(TunerOptions* pOptions) throw(std::runtime_error, std::bad_alloc)
{
	int iCard = -1;

	if (snd_card_next(&iCard) < 0 || iCard < 0)
	{
		TunerOptions::DeviceEntry deviceEntry;
		deviceEntry.sCardName = "No sound card found";
		deviceEntry.iCardNumber = 0;
		deviceEntry.iDevicesCount = 1;
		return;
	}

	snd_ctl_card_info_t* pInfo;
	snd_ctl_card_info_alloca(&pInfo);

	snd_pcm_info_t* pPcminfo;
	snd_pcm_info_alloca(&pPcminfo);

	while (iCard >= 0)
	{
		char name[32];
		std::sprintf(name, "hw:%d", iCard);

		int iErr;
		snd_ctl_t* pHandle;
		if ((iErr = snd_ctl_open(&pHandle, name, 0)) < 0)
		{
			std::cerr << "ALSA error: control open (" << iCard << "): " << snd_strerror(iErr)
					  << std::endl;
			continue;
		}

		if ((iErr = snd_ctl_card_info(pHandle, pInfo)) < 0)
		{
			std::cerr << "ALSA error: control hardware info (" << iCard << "): "
					  << snd_strerror(iErr) << std::endl;

			snd_ctl_close(pHandle);
			continue;
		}

		TunerOptions::DeviceEntry deviceEntry;
		deviceEntry.sCardName = snd_ctl_card_info_get_name(pInfo);
		deviceEntry.iCardNumber = iCard;
		deviceEntry.iDevicesCount = 0;

		int iDev = -1;
		while (true)
		{
			if (snd_ctl_pcm_next_device(pHandle, &iDev) < 0)
				std::cerr << "ALSA error: snd_ctl_pcm_next_device" << std::endl;

			if (iDev < 0)
				break;

			++deviceEntry.iDevicesCount;
		}

		snd_ctl_close(pHandle);

		pOptions->m_deviceList.push_back(deviceEntry);

		if (snd_card_next(&iCard) < 0)
		{
			std::cerr << "ALSA error: snd_card_next" << std::endl;
			break;
		}
	}
}
