/***************************************************************************
					pitchdetector.cpp  -  description
						 -------------------
		begin                : Fri Jun 30 2000
		copyright            : (C) 2000 by Jozef Kosoru
		email                : jozef.kosoru@pobox.sk
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#define __STDC_LIMIT_MACROS
#include "control.h"
#include <cassert>
#include <cmath>
#include <cstdlib>
#include <iostream>
#include "pitchdetector.h"
#include "tuneroptions.h"

template<class SAMPLE>
PitchDetector<SAMPLE>::PitchDetector(int samplerate,
									 SAMPLE* buffer,
									 int bufSize,
									 SAMPLE threshold,
									 int maxFreq,
									 int minFreq,
									 float maxFreqDif,
									 float maxAmpDif,
									 float calibration,
									 int minPeaksAmount) throw (std::bad_alloc)
{
	m_samplerate = static_cast<float>(samplerate)*calibration;
	m_buffer = buffer;
	m_bufSize = bufSize;
	m_threshold = threshold;
	m_maxPeriodDif = maxFreqDif;
	m_maxAmpDif = maxAmpDif;
	m_minPeaksAmount = minPeaksAmount;
	m_minPeriod = roundToInt(samplerate / static_cast<float>(maxFreq));
	m_maxPeriod = roundToInt(samplerate / static_cast<float>(minFreq));
	m_peaksMinimum = m_bufSize/m_maxPeriod;
	m_peakBufPos = new PeakStore[bufSize / 4];
	m_peakBufNeg = new PeakStore[bufSize / 4];
}

template<class SAMPLE>
PitchDetector<SAMPLE>::PitchDetector(const TunerOptions* options)
{
	m_samplerate = static_cast<float>(options->m_iSamplerate_real) * options->m_calibration;
	m_buffer = reinterpret_cast<SAMPLE*>(options->m_pAudioBuffer);
	m_bufSize = options->m_iBufferSize_real / sizeof(SAMPLE);
	m_threshold = static_cast<SAMPLE>(options->m_threshold);
	float maxIntValue = 0;

	switch(sizeof(SAMPLE))
	{
		case 1: maxIntValue = INT8_MAX; break;
		case 2: maxIntValue = INT16_MAX; break;
		case 4: maxIntValue = INT32_MAX; break;
		case 8: maxIntValue = INT64_MAX; break;
	}

	m_threshold = static_cast<SAMPLE>(options->m_threshold * maxIntValue);
	m_maxPeriodDif = options->m_maxFreqDif;
	m_maxAmpDif = options->m_maxAmpDif;
	m_minPeaksAmount = options->m_minPeaksAmount;
	m_minPeriod = roundToInt(m_samplerate / static_cast<float>(options->m_maxFreq));
	m_maxPeriod = roundToInt(m_samplerate / static_cast<float>(options->m_minFreq));
	m_peaksMinimum = m_bufSize / m_maxPeriod;
	m_peakBufPos = new PeakStore[m_bufSize / 4];
	m_peakBufNeg = new PeakStore[m_bufSize / 4];

}

template<class SAMPLE>
inline int PitchDetector<SAMPLE>::roundToInt(const float f)
{
	const int fi = static_cast<int>(f);

	if ((f - static_cast<float>(fi)) >= 0.5)
	{
		return fi + 1;
	}
	else
	{
		return fi;
	}
}

template<class SAMPLE>
PitchDetector<SAMPLE>::~PitchDetector()
{
#ifdef _DEBUG
	std::cerr << "PitchDetector deleted...\n";
#endif
	delete [] m_peakBufPos;
	delete [] m_peakBufNeg;
}

template<class SAMPLE>
float PitchDetector<SAMPLE>::analyzeBuffer()
{
	SearchPhase status = lowSignal;
	int posPos = 0, negPos = 0;
	bool upCross = false, downCross = false;
	m_peakBufPos[0].value = m_peakBufNeg[0].value = 0;
	int cTopPos = 0, cBottomPos = 0; // defalut values are not needed, but a compiler gives warning
	int cTopPeak = 0, cBottomPeak = 0;

	// fill peak buffers
	for(int count = 0; count < m_bufSize - 1; ++count)
	{
		switch(status)
		{
			case lowSignal: // process samples between -threshold & threshold level
				if (m_buffer[count] > m_threshold && m_buffer[count + 1] > m_threshold)
				{
					status = topPeak;
					upCross = true;

					if (downCross)
					{
						if (m_peakBufNeg[negPos].value > cBottomPeak)
						{
							cBottomPeak = m_peakBufNeg[negPos].value;
							cBottomPos = negPos;
						}

						negPos++;
						downCross = false;
						m_peakBufNeg[negPos].value = 0;
					}
				}
				else if (m_buffer[count] < -m_threshold && m_buffer[count+1] < -m_threshold)
				{
					status = bottomPeak;
					downCross = true;
					if (upCross)
					{
						if (m_peakBufPos[posPos].value > cTopPeak)
						{
							cTopPeak = m_peakBufNeg[posPos].value;
							cTopPos = posPos;
						}

						posPos++;
						upCross = false;
						m_peakBufPos[posPos].value = 0;
					}
				}
				break;

			case topPeak: // process samples bigger that threshold level
				if (m_buffer[count] <= m_threshold)
				{
					status = lowSignal;
				}
				else if (m_buffer[count] > m_peakBufPos[posPos].value)
				{
					m_peakBufPos[posPos].value = m_buffer[count];
					m_peakBufPos[posPos].position = count;
				}
				break;

			case bottomPeak: // process samples smaller that -threshold level
				if (-m_buffer[count] < m_threshold)
				{
					status = lowSignal;
				}
				else if (-m_buffer[count] > m_peakBufNeg[negPos].value)
				{
					m_peakBufNeg[negPos].value = -m_buffer[count];
					m_peakBufNeg[negPos].position = count;
				}
				break;
		}
	}

	// there must be at least m_peaksMinimum samples on the positive or negative side
	if (posPos < m_peaksMinimum && negPos < m_peaksMinimum)
	{
#ifdef _DEBUG_PD
		std::cout << "unsufficient amount of peaks\n";
		std::cout << "posPos: " << posPos << " negPos: " << negPos << std::endl;
#endif
		return -1;  // unsufficient amount of peaks
	}

	// compute average peak for positive and negative values
	const int averagePos = averagePeak(m_peakBufPos, posPos);
	const int averageNeg = averagePeak(m_peakBufNeg, negPos);

	// decide which side (positive/negative) to process according an average peak
	PeakStore* peakBuff;
	int peakNum, start;

	if (averagePos >= averageNeg)
	{
		peakBuff = m_peakBufPos;
		peakNum = posPos;
		start = cTopPos;
	}
	else
	{
		peakBuff = m_peakBufNeg;
		peakNum = negPos;
		start = cBottomPos;
	}

	// choose process direction
	int stop1, stop2;
	bool rightDirect; //true = --> / false = <--
	assert(peakNum != start);

	if ((peakNum - start - 1) >= start)
	{	// direction = -->
		stop1 = peakNum;
		stop2 = -1;
		rightDirect = true;
	}
	else
	{	// direction = <--
		stop1 = -1;
		stop2 = peakNum;
		rightDirect = false;
	}

	// find base period
	int periodsFound = 0, periodsSum = 0, cPeriod, cStep = start;
	// periodsFound and periodsSum don't need defaut values, but a compiler gives warning

	while (cPeriod = giveBasePeriod(cStep, peakBuff, start, stop1, rightDirect))
	{
		int aPeriod = cPeriod, bufStep = cStep;
		periodsFound = 1;
		periodsSum = cPeriod;

		while ((aPeriod = findAnotherPeriod(aPeriod, bufStep, peakBuff, stop1, rightDirect)) > 0)
		{
			++periodsFound;
			periodsSum += aPeriod;
		}

		if(!aPeriod)
		{	// no more peaks (maybe we found the right period)
			break;
		}
		// else aPeriod=-1 -> we must try next basePeriod
	}

	if (!cPeriod || periodsFound < m_minPeaksAmount)
	{
#ifdef _DEBUG_PD
		std::cout << "no base period / there was few of suitable peaks\n";
#endif
		return -1;  // no base period / there was few of suitable peaks
	}

	// evaluate other direction
	int aPeriod = cPeriod;
	int bufStep = start;
	rightDirect =! rightDirect;

	while ((aPeriod = findAnotherPeriod(aPeriod, bufStep, peakBuff, stop2, rightDirect)) > 0)
	{
		++periodsFound;
		periodsSum += aPeriod;
	}

	if(aPeriod == -1)
	{
#ifdef _DEBUG_PD
		std::cout << "no same basePeriod on other side\n";
#endif
		return -1; // no same basePeriod on other side
	}

	const float resultPeriod = static_cast<float>(periodsSum) / static_cast<float>(periodsFound);
	return m_samplerate / resultPeriod;  // return resultant frequency
}

template<class SAMPLE>
inline int PitchDetector<SAMPLE>::giveBasePeriod(int& cStep,
												 const PeakStore* peakBuff,
												 const int start,
												 const int stop,
												 const bool rightDirect) const
{
	static int peakValue;
	static ValueRange ampRange;

	if (peakValue != static_cast<int>(peakBuff[start].value))
	{	// peak value has changed -> compute new range
		peakValue = static_cast<int>(peakBuff[start].value);
		ampRange = computeRange(peakValue, m_maxAmpDif);
	}

#ifdef _DEBUG_PD
	std::cout << "ENTER cStep " << cStep << std::endl;
#endif

	if (rightDirect)
	{
		while ((++cStep) < stop)
		{
			const PeakStore peak = peakBuff[cStep];
			const int periodSize = peak.position-peakBuff[start].position;

			if (periodSize > m_maxPeriod)
			{
				break;
			}
			else if (periodSize < m_minPeriod)
			{
#ifdef _DEBUG_PD
				std::cout << "TOO SMALL - " << periodSize << std::endl;
#endif
				continue;
			}
			else if (peak.value <= ampRange.hi && peak.value >= ampRange.lo)
			{
#ifdef _DEBUG_PD
				std::cout << ">>TRY----: " << periodSize << " value: " << peak.value
						  << std::endl;
				std::cout << "cStep: " << cStep << std::endl;
#endif
				return periodSize;
			}

#ifdef _DEBUG_PD
			std::cout << "baseFAIL: " << periodSize << " ampDif: " << ampRange.hi << "/"
					  << ampRange.lo << std::endl;
#endif
		}
	}
	else
	{
		while ((--cStep) > stop)
		{
			const PeakStore peak = peakBuff[cStep];
			const int periodSize = peakBuff[start].position - peak.position;

			if (periodSize > m_maxPeriod)
			{
				break;
			}
			else if (periodSize < m_minPeriod)
			{
#ifdef _DEBUG_PD
				std::cout << "TOO SMALL - " << periodSize << std::endl;
#endif
				continue;
			}
			else if (peak.value <= ampRange.hi && peak.value >= ampRange.lo)
			{
#ifdef _DEBUG_PD
				std::cout << ">>TRY----: " << periodSize << " value: " << peak.value
						  << std::endl;
				std::cout << "cStep: " << cStep << std::endl;
#endif
				return periodSize;
			}
#ifdef _DEBUG_PD
			std::cout << "baseFAIL: " << periodSize << " ampDif: " << ampRange.hi << "/"
					  << ampRange.lo << std::endl;
#endif
		}
	}
	return 0; // no suitable peak
}

template<class SAMPLE>
inline typename PitchDetector<SAMPLE>::ValueRange
PitchDetector<SAMPLE>::computeRange(const int num, const float difMulti)
{
	const int dif = roundToInt(static_cast<float>(num) * difMulti);
	const ValueRange range = {num - dif, num + dif};
	return range;
}

template<class SAMPLE>
inline int PitchDetector<SAMPLE>::findAnotherPeriod(const int cPeriod,
													int& bufStep,
													const PeakStore* peakBuff,
													const int stop,
													const bool rightDirect) const
{
	const PeakStore prevPeak = peakBuff[bufStep];
	const ValueRange ampRange = computeRange(static_cast<int>(prevPeak.value), m_maxAmpDif);
	const ValueRange periodRange = computeRange(cPeriod, m_maxPeriodDif);
#ifdef _DEBUG_PD
	std::cout << "find__bufStep: " << bufStep << std::endl;
#endif

	if (rightDirect)
	{
		while ((++bufStep) < stop)
		{
			const PeakStore peak = peakBuff[bufStep];
			const int periodSize = peak.position - prevPeak.position;

			if (periodSize > periodRange.hi || periodSize > m_maxPeriod || periodSize < m_minPeriod)
			{
#ifdef _DEBUG_PD
				std::cout << "periodFAIL: " << periodSize << " pRange: " << periodRange.hi << "|"
						  << periodRange.lo << std::endl;
				std::cout << "__bufStep: " << bufStep << " PEAK: " << peak.position << std::endl;
#endif
				return -1;
			}
			else if (peak.value <= ampRange.hi && peak.value >= ampRange.lo &&
					 periodSize >= periodRange.lo)
			{
				if ((bufStep + 1) < stop)
				{
					int nextPeriodSize = peakBuff[bufStep + 1].position - prevPeak.position;
					if (std::abs(cPeriod - nextPeriodSize) < abs(cPeriod - periodSize))
					{
						continue; // next peak will be better
					}
				}
				return periodSize;  // this was suitable peak
			}

#ifdef _DEBUG_PD
			std::cout << "FAIL: " << periodSize << " ampDif: " << ampRange.hi << "/"
					  << ampRange.lo << " periodDif: " << periodRange.hi << "/"
					  << periodRange.lo << std::endl;
#endif
		}
	}
	else
	{
		while ((--bufStep) > stop)
		{
			const PeakStore peak = peakBuff[bufStep];
			const int periodSize = prevPeak.position - peak.position;
			if (periodSize > periodRange.hi || periodSize > m_maxPeriod ||
				periodSize < m_minPeriod)
			{
#ifdef _DEBUG_PD
				std::cout << "periodFAIL: " << periodSize << " pRange: " << periodRange.hi << "|"
						  << periodRange.lo << std::endl;
				std::cout << "__bufStep: " << bufStep << " PEAK: " << peak.position << std::endl;
#endif
				return -1;
			}
			else if (peak.value <= ampRange.hi && peak.value >= ampRange.lo &&
					 periodSize >= periodRange.lo)
			{
				if ((bufStep-1) > stop)
				{
					int nextPeriodSize = prevPeak.position - peakBuff[bufStep-1].position;
					if(std::abs(cPeriod - nextPeriodSize) < abs(cPeriod - periodSize))
					{
						continue; // next peak will be better
					}
				}
				return periodSize;  // this was suitable peak
			}

#ifdef _DEBUG_PD
			std::cout << "FAIL: " << periodSize << " ampDif: " << ampRange.hi << "/" << ampRange.lo
					  << " periodDif: " << periodRange.hi << "/" << periodRange.lo << std::endl;
#endif
		}
	}

	return 0; // no more peaks
}

template<class SAMPLE>
inline int PitchDetector<SAMPLE>::averagePeak(const PeakStore* peaksBuf, const int bufSize)
{
	if (bufSize)
	{
		int sum = 0;
		for (int count = 0; count < bufSize; ++count)
		{
			sum += peaksBuf[count].value;
		}

		return sum / bufSize;
	}
	else
	{
		return 0;
	}
}

template class PitchDetector<int16_t>;
template class PitchDetector<int8_t>;
