/* icons.c
*
* exports icons
*
*/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "global.h"
#include "options.h"
#include "icons.h"
#include "mainwindow.h"
#include "misc.h"
#include "misc_strings.h"
#include "status_page.h"

#include <string.h>

static GdkPixbuf                       *icon_pool[ICON_NUM_ICONS];

#define load_icon(num, file)            icon_pool[num] = icons_read_icon_file(file)
#define icon_alias(num, orignum)        icon_pool[num] = icon_pool[orignum]






/* icons_get_installed_icon_themes
 *
 * returns a list of installed icon themes (i.e. directories with icons)
 *
 * the first entry returned should automatically be either the selected
 *   icon theme (if option is set) or the default icon theme path (if
 *   option is not set)
 *
 */

GSList *
icons_get_installed_icon_themes (void)
{
	const gchar    *dirname;
	const gchar    *homedir;
	GSList         *themes = NULL;
	GError         *error  = NULL;
	gchar          *hdiconpath;
	GDir           *dir;

	homedir = g_getenv ("HOME");

	if (opt_get_str(OPT_GUI_ICON_THEME))
		themes = g_slist_append (themes, g_strdup(opt_get_str(OPT_GUI_ICON_THEME)));

	/* first, check $pkgdatapath/icons/ */

	dir = g_dir_open ( APP_PIXMAPS, 0, &error );
	if (!error)
	{
		/* Note: g_dir_read_name() omits '.' and '..' entries */
		dirname = g_dir_read_name(dir);
		while (dirname)
		{
			if ( !opt_get_str(OPT_GUI_ICON_THEME) || strcmp(opt_get_str(OPT_GUI_ICON_THEME),dirname) != 0 )
				themes = g_slist_append (themes, g_strdup(dirname));

			dirname = g_dir_read_name(dir);
		}
		g_dir_close(dir);
	}
	else
	{
		g_warning("g_dir_open(%s) failed. Reason: %s\n", APP_PIXMAPS, error->message );
		g_error_free(error);
		error = NULL;
	}

	if (!homedir)
		return themes;

	hdiconpath = g_strconcat (homedir, G_DIR_SEPARATOR_S, ".", PACKAGE, G_DIR_SEPARATOR_S, "icons", NULL);
	g_return_val_if_fail ( hdiconpath != NULL, themes );

	/* Note: we DO want to test for existence here, because in 99.99% of the cases,
	 * users do not have private icon themes installed in ~/.ed2k_gui/icons/
	 * - and we really only want to display an error if it does exist and we can't open it */
	if ( !g_file_test (hdiconpath, G_FILE_TEST_EXISTS) )
	{
		g_free(hdiconpath);
		return themes;
	}

	error = NULL;
	dir = g_dir_open (hdiconpath, 0, &error);
	if (error)
	{
		status_system_error_msg (_("opendir($HOME/.ed2k_gui/icons/) failed"));
		g_printerr ("g_dir_open(%s) failed. Reason: %s\n", hdiconpath, error->message );
		g_free(hdiconpath);
		g_error_free(error);
		return themes;
	}

	/* Note: g_dir_read_name() omits '.' and '..' entries */
	dirname = g_dir_read_name(dir);
	while (dirname)
	{
		if ( g_slist_find_custom (themes, dirname, (GCompareFunc)strcmp) == NULL )
			themes = g_slist_append (themes, g_strdup(dirname));

		dirname = g_dir_read_name(dir);
	}

	g_dir_close(dir);

	g_free(hdiconpath);

	return themes;
}




static GdkPixbuf *
icons_read_icon_file (const gchar *name)
{
	GdkPixbuf   *pb = NULL;
	GdkPixbuf   *pb_scaled = NULL;
	GError      *err = NULL;
	const gchar *path;
	gchar       *icon_name;
	gint         scaling_factor;

	icon_name = g_strconcat("icons", G_DIR_SEPARATOR_S, opt_get_str(OPT_GUI_ICON_THEME),
	                        G_DIR_SEPARATOR_S, name, ".png", NULL);

	path = opt_get_opt_filename_without_instance (icon_name);

	pb = gdk_pixbuf_new_from_file (path, &err);

	g_free (icon_name);

	if (pb)
		return pb;

	if (err)
	{
		g_error_free(err);
		err = NULL;
	}

	icon_name = g_strconcat( APP_PIXMAPS, G_DIR_SEPARATOR_S, opt_get_str(OPT_GUI_ICON_THEME),
	                         G_DIR_SEPARATOR_S, name, ".png", NULL);

	pb = gdk_pixbuf_new_from_file (icon_name, &err);

	if (err)
	{
		g_print (_("WARNING: error loading icon '%s': %s\n"), name, err->message);
		g_error_free(err);
		g_free(icon_name);
		return NULL;
	}

	g_free(icon_name);

	/* Do we want to resize the pixbuf or not? */

	scaling_factor = opt_get_int(OPT_GUI_ICON_SCALING_FACTOR);

	if ( pb == NULL || scaling_factor == 100)
		return pb;

	g_return_val_if_fail ( scaling_factor > 25 && scaling_factor < 500, pb );

	pb_scaled = gdk_pixbuf_scale_simple(pb, (gdk_pixbuf_get_width(pb) * scaling_factor)/100,
	                                        (gdk_pixbuf_get_height(pb) * scaling_factor)/100,
	                                        GDK_INTERP_HYPER); /* GDK_INTERP_BILINEAR if speed is important */

	gdk_pixbuf_unref(pb); /* delete original pixbuf */

  return pb_scaled;
}


gboolean
icons_check_theme (const gchar *theme)
{
	const gchar *icon_path;
	gchar       *icon_buf;

	icon_path = opt_get_opt_filename_without_instance("icons");

	icon_buf = g_strdup_printf ("%s/%s", icon_path, theme);

	if (!g_file_test(icon_buf, G_FILE_TEST_IS_DIR))
	{
		g_free (icon_buf);

		icon_buf = g_strdup_printf ("%s/%s", APP_PIXMAPS, theme);

		if (!g_file_test(icon_buf, G_FILE_TEST_IS_DIR))
		{
			g_free (icon_buf);
			return FALSE;
		}
	}

	G_FREE(icon_buf);

	return TRUE;
}

void icons_init (void)
{

	gint i;

	for (i = 0; i < ICON_NUM_ICONS; i++)

		icon_pool[i] = NULL;

	/* Until we're done with all icons, we can use icon_alias(id, id2) to make icon ID a clone of icon ID2 */


	load_icon(ICON_NONE, "none");

	load_icon(ICON_OK, "ok");
	load_icon(ICON_CANCEL, "cancel");
	load_icon(ICON_NUKE, "nuke");
	load_icon(ICON_SHUTDOWN, "shutdown");

	load_icon(ICON_OPTION_TRUE, "option-true");
	load_icon(ICON_OPTION_FALSE, "option-false");

	load_icon(ICON_MENU_DOWNLOAD, "menu-download");
	load_icon(ICON_MENU_CANCEL, "menu-cancel");
	load_icon(ICON_MENU_DL_CANCEL, "menu-dl-cancel");
	load_icon(ICON_MENU_PAUSE, "menu-pause");
	icon_alias(ICON_MENU_PAUSE_ALL, ICON_MENU_PAUSE);
	load_icon(ICON_MENU_RESUME, "menu-resume");
	icon_alias(ICON_MENU_RESUME_ALL, ICON_MENU_RESUME); 
	load_icon(ICON_MENU_OPTIONS, "menu-options");
	load_icon(ICON_MENU_CONNECT, "menu-connect");
	load_icon(ICON_MENU_DISCONNECT, "menu-disconnect");
	load_icon(ICON_MENU_PRIO_HIGH, "menu-prio-high");
	load_icon(ICON_MENU_PRIO_MEDIUM, "menu-prio-medium");
	load_icon(ICON_MENU_PRIO_LOW, "menu-prio-low");
	load_icon(ICON_MENU_HIDE_COLUMN, "menu-hide-column");
	load_icon(ICON_MENU_UNHIDE_COLUMN, "menu-unhide-column");
	load_icon(ICON_MENU_REFRESH, "menu-refresh");
	load_icon(ICON_MENU_BLACKLIST, "menu-blacklist");
	load_icon(ICON_MENU_STATUS, "menu-status");
	load_icon(ICON_MENU_SEARCH, "menu-search");
	load_icon(ICON_MENU_PREVIEW, "menu-preview");
	load_icon(ICON_MENU_QUIT, "menu-quit");
	load_icon(ICON_MENU_EXTEND, "menu-extend");
	load_icon(ICON_MENU_HOMEPAGE, "menu-homepage");

	icon_alias(ICON_MENU_200, ICON_MENU_EXTEND);

	load_icon(ICON_MENU_THANKYOU, "menu-thankyou");

	icon_alias(ICON_SEARCH, ICON_MENU_SEARCH);
	icon_alias(ICON_EXTEND, ICON_MENU_EXTEND);


	load_icon(ICON_TAB_DOWNLOADS, "tab-downloads");
	load_icon(ICON_TAB_UPLOADS,"tab-uploads");
	load_icon(ICON_TAB_SEARCH, "tab-search");
	load_icon(ICON_TAB_SERVERS, "tab-servers");
	load_icon(ICON_TAB_STATUS, "tab-status");
	load_icon(ICON_TAB_OPTIONS,"tab-options");
	load_icon(ICON_TAB_SHARED,"tab-shared");
	load_icon(ICON_TAB_STATS, "tab-stats");
	load_icon(ICON_TAB_FORGET, "tab-forget");

	icon_alias(ICON_STATS, ICON_TAB_STATS);


	load_icon(ICON_TOOLBAR_ED2K_LOGO, "toolbar-ed2k-logo");
	load_icon(ICON_TOOLBAR_OVERNET_LOGO, "toolbar-overnet-logo");
	load_icon(ICON_TOOLBAR_NUM_CONNECTIONS,"toolbar-num-connections");
	load_icon(ICON_TOOLBAR_SERVER, "toolbar-server");
	icon_alias(ICON_TOOLBAR_DOWN_SPEED, ICON_TAB_DOWNLOADS);
	icon_alias(ICON_TOOLBAR_UP_SPEED, ICON_TAB_UPLOADS);
	load_icon(ICON_TOOLBAR_GO, "toolbar-go");
	load_icon(ICON_TOOLBAR_ABOUT, "toolbar-about");
	icon_alias(ICON_ABOUT, ICON_TOOLBAR_ABOUT);
	icon_alias(ICON_MENU_ABOUT, ICON_TOOLBAR_ABOUT);

	load_icon(ICON_SYS_KDE, "sys-kde");
	load_icon(ICON_SYS_GNOME1, "sys-gnome1");
	load_icon(ICON_SYS_GNOME2, "sys-gnome2");
	load_icon(ICON_SYS_LINUX, "sys-linux");
	load_icon(ICON_SYS_MOZILLA, "sys-mozilla");
	load_icon(ICON_SYS_WINDOWS, "sys-windows");
	
	load_icon(ICON_ED2K_GTK_GUI, "ed2k-gtk-gui");
	load_icon(ICON_ED2K_GTK_GUI_MINI, "ed2k-logo-mini");

	load_icon(ICON_FILETYPE_AUDIO,     "filetype-audio");
	load_icon(ICON_FILETYPE_IMAGE,     "filetype-image");
	load_icon(ICON_FILETYPE_VIDEO,     "filetype-video");
	load_icon(ICON_FILETYPE_PACKAGE,   "filetype-package");
	load_icon(ICON_FILETYPE_DOCUMENT,  "filetype-document");
	load_icon(ICON_FILETYPE_PROGRAM,   "filetype-program");
	load_icon(ICON_FILETYPE_SUBTITLES, "filetype-subtitles");
	load_icon(ICON_FILETYPE_CDIMAGE,   "filetype-cdimage");

	load_icon (ICON_DL_STATUS_ERROR,      "dl-status-error");
	load_icon (ICON_DL_STATUS_AVAIL_LOW,  "dl-status-avail-low");
	load_icon (ICON_DL_STATUS_AVAIL_HIGH, "dl-status-avail-high");
	load_icon (ICON_DL_STATUS_AVAIL_FULL, "dl-status-avail-full");
	load_icon (ICON_DL_STATUS_PAUSED,     "dl-status-paused");

	
	icon_alias(ICON_FILETYPE_FOLDER, ICON_TAB_SHARED);

}


GdkPixbuf *
get_icon (IconName icon_num)
{

	g_return_val_if_fail ( icon_num < ICON_NUM_ICONS, icon_pool[ICON_NONE]);

	return icon_pool[icon_num];


}


GdkPixbuf *
get_icon_for_file (const gchar *name)
{

	gchar	*ext = NULL;
	gchar	*type = NULL;

	g_return_val_if_fail (name != NULL, icon_pool[ICON_NONE]);

	ext = (char *)misc_get_extension_from_filename ((char *)name);

	if (!ext)

		return icon_pool[ICON_FILETYPE_DOCUMENT];
	
	type = (char *)misc_get_filetype_from_extension (ext);

	if (type)
	{

		if (strcmp (type, FILETYPE_AUDIO) == 0)

			return icon_pool[ICON_FILETYPE_AUDIO];

		else if (strcmp (type, FILETYPE_VIDEO) == 0)

			return icon_pool[ICON_FILETYPE_VIDEO];

		else if (strcmp (type, FILETYPE_PROGRAM) == 0)

			return icon_pool[ICON_FILETYPE_PROGRAM];

		else if (strcmp (type, FILETYPE_IMAGE) == 0)

			return icon_pool[ICON_FILETYPE_IMAGE];

		else if (strcmp (type, FILETYPE_ARCHIVE) == 0)

			return icon_pool[ICON_FILETYPE_PACKAGE];

		else if (strcmp (type, FILETYPE_DOCUMENT) == 0)

			return icon_pool[ICON_FILETYPE_DOCUMENT];

		else if (strcmp (type, FILETYPE_CDIMAGE) == 0)

			return icon_pool[ICON_FILETYPE_CDIMAGE];

		else if (strcmp (type, FILETYPE_SUBTITLES) == 0)

			return icon_pool[ICON_FILETYPE_SUBTITLES];

	}

	return icon_pool[ICON_FILETYPE_DOCUMENT];

}



