/* shared_page.c
*
*  populates shared page in notepad and deals with its functions
*
*  (c) 2001 Tim-Philipp Muller <t.i.m@orange.net>
*/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "colors.h"
#include "core-conn.h"
#include "global.h"
#include "misc_strings.h"
#include "options.h"

#include "shared_dirs.h"
#include "shared_files.h"
#include "shared_page.h"

#include <gtk/gtkhbox.h>
#include <gtk/gtkhpaned.h>
#include <gtk/gtklabel.h>
#include <gtk/gtkvbox.h>

/* variables */

static GtkWidget	*spanedwin = NULL;


/******************************************************************************
 *
 *   shared_page_onLoadingSharedFiles
 *
 ******************************************************************************/

static void
shared_page_onStatusMessage (GtkLabel *label, const gchar *text, gpointer data)
{
	static gint lastmsghash; /* 0 */

	g_return_if_fail ( GTK_IS_LABEL(label) );
	g_return_if_fail ( text != NULL );

	if (g_ascii_strncasecmp(text, "Loading shared files", 20) == 0)
	{
		if (lastmsghash != g_str_hash(text))
			gtk_label_set_text (GTK_LABEL(label), UTF8_SHORT_PRINTF(_(" Shared - Loading shared files ...")));
	}
	else if (g_ascii_strncasecmp(text, "Done loading shared files", 25) == 0)
	{
		gtk_label_set_text (GTK_LABEL(label), UTF8_SHORT_PRINTF(_(" Shared - Done loading shared files.")));
		gui_core_conn_send_get_shared_files(core);
		gui_core_conn_send_get_shared_dirs(core);
	}

	lastmsghash = g_str_hash(text);
}

/******************************************************************************
 *
 *   shared_page_onSharedFileHashing
 *
 ******************************************************************************/

static void
shared_page_onSharedFileHashing (GtkLabel *label, const gchar *fn, gpointer data)
{
	g_return_if_fail ( GTK_IS_LABEL(label) );
	g_return_if_fail ( fn != NULL );

	gtk_label_set_text (GTK_LABEL(label), UTF8_SHORT_PRINTF(_(" Shared - Hashing %s"), fn));
}


/******************************************************************************
 *
 *   shared_page_onPanedHandlePositionChanged
 *
 *   Called when an object property changes (here we are interested in
 *    the "position" one).
 *
 ******************************************************************************/

static void
shared_page_onPanedHandlePositionChanged (GObject *obj, GParamSpec *pspec, gpointer data)
{
	opt_set_int (OPT_GUI_PANE_POS_SHARED_PAGE, gtk_paned_get_position(GTK_PANED(obj)));
}


/******************************************************************************
 *
 *   shared_page_create_stats_label
 *
 ******************************************************************************/

static GtkWidget *
shared_page_create_stats_label (void)
{
	GtkWidget	*hbox, *emptylabel, *stats_label;

	hbox = gtk_hbox_new(FALSE,0);
	gtk_widget_show(hbox);

	emptylabel = gtk_label_new(" ");
	gtk_widget_show(emptylabel);

	stats_label = gtk_label_new (UTF8_SHORT_PRINTF("%s",_(" Shared ")));
	gtk_label_set_justify (GTK_LABEL(stats_label), GTK_JUSTIFY_LEFT);
	gtk_widget_show(stats_label);

	gtk_box_pack_start (GTK_BOX(hbox), stats_label, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX(hbox), emptylabel, TRUE, TRUE, 0);

	if (!opt_get_bool(OPT_GUI_NO_GREEN))
		set_style_recursively (hbox, style_notebook_label);

	g_signal_connect_swapped(core, "shared-file-hashing", (GCallback) shared_page_onSharedFileHashing, stats_label);
	g_signal_connect_swapped(core, "status-message", (GCallback) shared_page_onStatusMessage, stats_label);

	return hbox;
}



/******************************************************************************
 *
 *  shared_page_create
 *
 ******************************************************************************/

GtkWidget *
shared_page_create (void)
{
	GtkWidget *vbox, *statslabel;

	vbox = gtk_vbox_new(FALSE,0);

	gtk_widget_show(vbox);

	statslabel = shared_page_create_stats_label();

	gtk_box_pack_start (GTK_BOX(vbox), statslabel, FALSE, FALSE, 0);

	spanedwin = gtk_hpaned_new ();

	gtk_widget_show (spanedwin);

	gtk_paned_add2 (GTK_PANED(spanedwin), shared_files_create_view  () );

	gtk_paned_add1 (GTK_PANED(spanedwin), shared_dirs_create_view () );

	gtk_paned_set_position (GTK_PANED(spanedwin), opt_get_int(OPT_GUI_PANE_POS_SHARED_PAGE));

	gtk_box_pack_start (GTK_BOX(vbox), spanedwin, TRUE, TRUE, 0);

	g_signal_connect(spanedwin, "notify::position", (GCallback) shared_page_onPanedHandlePositionChanged, NULL);

	return vbox;
}


