/***************************************************************************
 *   Copyright (C) 2005-2006 by Conrad Hoffmann                            *
 *   conrausch@gmx.de                                                      *
 *   http://conrausch.doesntexist.org                                      *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

/* KTTS PLUGIN FOR THE KOPETE KDE INSTANT MESSANGER 
 *
 * base class implementation
 *
 */

#include <qregexp.h>
#include <qtimer.h>

#include <kdebug.h>
#include <kapplication.h>
#include <dcopclient.h>
#include <kgenericfactory.h>
#include <kaction.h>
#include <kaboutdata.h>
#include <kmessagebox.h>

#include <kopetemetacontact.h>
#include <kopetecontactlist.h>
#include <kopeteview.h>
#include <kopeteaccount.h>
#include <kopeteaccountmanager.h>
#include <kopetechatsessionmanager.h>

#include "kopetekttsuserprefs.h"
#include "kopete_ktts.h"
#include "kopetekttsplugin.h"

typedef KGenericFactory<KopeteKttsPlugin> KopeteKttsPluginFactory;

static const KAboutData aboutdata( "kopete_ktts", I18N_NOOP( "KopeteKtts" ) , "1.3.0" );
K_EXPORT_COMPONENT_FACTORY( kopete_ktts, KopeteKttsPluginFactory( &aboutdata ) )


KopeteKttsPlugin::KopeteKttsPlugin( QObject *parent, const char *name, const QStringList & )
		: DCOPStub( "kttsd", "KSpeech" ),
		Kopete::Plugin( KopeteKttsPluginFactory::instance(), parent, name )
		
{
	kdDebug() << k_funcinfo << "Starting KTTS Plugin" << endl;
	
	if ( ! pluginStatic_ )
	{
		pluginStatic_ = this;
	}

	// load configuration
	slotSettingsChanged();

	// connect the statusChanged slots
	initialize();
	
	// add item to (contact) menu
	KAction *showUserPrefsDialog = new KAction( i18n( "&Text-to-Speech settings" ), QString::fromLatin1( "kmouth" ), 0, this, SLOT( slotShowUserPrefsDialog() ), actionCollection() , "showUserPrefsDialog" );
	QObject::connect ( Kopete::ContactList::self() , SIGNAL( metaContactSelected( bool ) ) , showUserPrefsDialog , SLOT( setEnabled( bool ) ) );
	showUserPrefsDialog->setEnabled( Kopete::ContactList::self() ->selectedMetaContacts().count() == 1 );
	setXMLFile( "kttsui.rc" );

	// look out for new meta contacts
	QObject::connect( Kopete::ContactList::self(), SIGNAL( metaContactAdded( Kopete::MetaContact* ) ), this, SLOT( slotMetaContactAdded( Kopete::MetaContact* ) ) );
	// listen for incoming messages
	QObject::connect( Kopete::ChatSessionManager::self(), SIGNAL( aboutToReceive( Kopete::Message& ) ), this, SLOT( slotIncomingMessage( Kopete::Message& ) ) );
	// listen for outgoing messages
	QObject::connect( Kopete::ChatSessionManager::self(), SIGNAL( aboutToSend( Kopete::Message& ) ), this, SLOT( slotOutgoingMessage( Kopete::Message& ) ) );
	// listen for config changes
	QObject::connect( this, SIGNAL( settingsChanged() ), this, SLOT( slotSettingsChanged() ) );
	// track newly created chat sessions
	QObject::connect( Kopete::ChatSessionManager::self(), SIGNAL( chatSessionCreated(Kopete::ChatSession*) ), this, SLOT( slotSessionCreated( Kopete::ChatSession* ) ) );

	// see if kttsd is running, if not try to start it...
	DCOPClient *client = KApplication::dcopClient();
	if ( ! client->isAttached() )
	{
		client->attach();
	}
	if ( ! client->isApplicationRegistered("kttsd") )
	{
		QString error;
		if ( KApplication::startServiceByDesktopName( "kttsd", QStringList(), &error ) )
		{
			KMessageBox::error( 0, i18n( "Starting KTTSD failed:\n %1\n Speech output will not work!").arg(error), i18n( "Kopete KTTSD Plugin") );
		}
	}
	// test 1,2,3
	if ( KopeteKttsKcfg::readTestMessage() )
	{
		testSpeechOutput();
	}

}


KopeteKttsPlugin::~KopeteKttsPlugin()
{
	pluginStatic_ = 0L;
}


KopeteKttsPlugin* KopeteKttsPlugin::pluginStatic_ = 0L;


KopeteKttsPlugin* KopeteKttsPlugin::self()
{
	return pluginStatic_ ;
}


void KopeteKttsPlugin::initialize()
{
	QPtrList<Kopete::MetaContact> metalist = Kopete::ContactList::self()->metaContacts();
	QPtrList<Kopete::Contact> list;
	Kopete::MetaContact* mc;
	for ( mc = metalist.first(); mc != 0; mc = metalist.next() )
	{
		QObject::connect( mc, SIGNAL( onlineStatusChanged( Kopete::MetaContact*, Kopete::OnlineStatus::StatusType ) ), this, SLOT( slotOnlineStatusChanged( Kopete::MetaContact*, Kopete::OnlineStatus::StatusType ) ) );
	}
}


KopeteKttsUserPrefs* KopeteKttsPlugin::getUserPrefs( Kopete::MetaContact* mc )
{
	// return user specific preferences. create them if necessary
	KopeteKttsUserPrefs* prefs = this->userPrefs.find( mc->metaContactId() );
	if ( prefs == 0 )
	{
		prefs = new KopeteKttsUserPrefs( mc );
		this->userPrefs.insert( mc->metaContactId(), prefs );
	}
	return prefs;
}


//
// SLOTS
//

void KopeteKttsPlugin::slotSettingsChanged()
{
	kdDebug() << k_funcinfo << "slotSettingsChanged() triggered" << endl;
	KopeteKttsKcfg::self()->readConfig();
}


void KopeteKttsPlugin::slotSessionCreated( Kopete::ChatSession* session )
{
	freshSessions.append( session );
}


void KopeteKttsPlugin::slotIncomingMessage( Kopete::Message& msg )
{
	// notify only on newly created chats management
	bool isFreshSession = true;
	Kopete::Contact* c = const_cast<Kopete::Contact*>( msg.from() );
	if (  freshSessions.find( c->manager( Kopete::Contact::CannotCreate ) ) < 0 )
	{
		isFreshSession = false;
	}
	else
	{
		freshSessions.remove( c->manager( Kopete::Contact::CannotCreate ) );
	}
	
	// check contact settings and initiate the appropiate action
	KopeteKttsUserPrefs* prefs = getUserPrefs( msg.from()->metaContact() );
	if ( prefs->getUseSpeech() )
	{
		if ( prefs->getAnnounceIncomingMessages() )
		{
			if ( isFreshSession || ( ! prefs->getAnnounceIncomingMessagesOnlyOnce() ) )
			{
				announceIncomingMessage( prefs );
			}
		}
		if ( prefs->getReadIncomingMessages() )
		{
			if ( isFreshSession || ( ! prefs->getReadIncomingMessagesOnlyOnce() ) )
			{
				readMessage( msg.plainBody() );
			}
		}
	}
}


void KopeteKttsPlugin::slotOutgoingMessage( Kopete::Message& msg )
{
	// check contact settings and initiate the appropiate action
	KopeteKttsUserPrefs* prefs = getUserPrefs( msg.from()->metaContact() );
	if ( prefs->getUseSpeech() && prefs->getReadOutgoingMessages() )
	{
		readMessage( msg.plainBody() );
	}
}


void KopeteKttsPlugin::slotMetaContactAdded( Kopete::MetaContact* mc )
{
	QObject::connect( mc, SIGNAL( onlineStatusChanged( Kopete::MetaContact*, Kopete::OnlineStatus::StatusType ) ), this, SLOT( slotOnlineStatusChanged( Kopete::MetaContact*, Kopete::OnlineStatus::StatusType ) ) );
}


void KopeteKttsPlugin::slotOnlineStatusChanged( Kopete::MetaContact* mc, Kopete::OnlineStatus::StatusType status )
{
	// disable status announcements while any account has the suppressStatusNotification flag set
	QPtrList<Kopete::Account> accounts = Kopete::AccountManager::self()->accounts();
	Kopete::Account* a;
	for ( a = accounts.first(); a != 0; a = accounts.next() )
	{
		if ( a->suppressStatusNotification() )
		{
			return;
		}
	}

	KopeteKttsUserPrefs* prefs = getUserPrefs( mc );
	if ( prefs->getUseSpeech() && prefs->getAnnounceStatusChanges() )
	{
		if ( status != Kopete::OnlineStatus::Unknown )
		{
			announceStatusChange( prefs, Kopete::OnlineStatus::statusTypeToString( status ) );
		}
	}
}


void KopeteKttsPlugin::slotShowUserPrefsDialog()
{
	// get the clicked contact
	Kopete::MetaContact* mc = Kopete::ContactList::self()->selectedMetaContacts().first();
	if ( mc )
	{
		// show his settings
		KopeteKttsUserPrefs* userPrefs = getUserPrefs( mc );
		userPrefs->showContactSettingsDialog( );
	}
}


//
// SPEECH ACTIONS
//

void KopeteKttsPlugin::testSpeechOutput( )
{
	kdDebug() << k_funcinfo << "Now testing the text-to-speech plugin" << endl;
	// If you really read through all this, you really deserve a cooler startup message (see below) ;)
	// QString text = "lets play some tetris motherfucker";
	QString text = "your speech pluck in is now enabled";
	sayMessage( text, 0 );
}


void KopeteKttsPlugin::announceStatusChange( KopeteKttsUserPrefs* prefs, QString statusDescription )
{
	kdDebug() << k_funcinfo << "Now reading online alert" << endl;
	QString name = prefs->getName();
 	QString text = name + " " + i18n("is now") + " " + statusDescription;
	sayMessage( text, 0 );
}


void KopeteKttsPlugin::announceIncomingMessage( KopeteKttsUserPrefs* prefs )
{
	kdDebug() << k_funcinfo << "Now announcing incoming message" << endl;
	QString text = prefs->getAnnouncementText();
	QString name = prefs->getName();
	text.replace( QRegExp::QRegExp( QString::QString( "%name%" ), false, false ), name );
	sayMessage( text, 0 );
}

void KopeteKttsPlugin::readMessage( QString message )
{
	kdDebug() << k_funcinfo << "Now reading message" << endl;
	uint job = setText( message, 0 );
	startText( job );
}


#include "kopetekttsplugin.moc"
