/*   **********************************************************************  **
 **   Copyright notice                                                       **
 **                                                                          **
 **   (c) 2003-2006 RSSOwl Development Team                                  **
 **   http://www.rssowl.org/                                                 **
 **                                                                          **
 **   All rights reserved                                                    **
 **                                                                          **
 **   This program and the accompanying materials are made available under   **
 **   the terms of the Eclipse Public License 1.0 which accompanies this     **
 **   distribution, and is available at:                                     **
 **   http://www.rssowl.org/legal/epl-v10.html                               **
 **                                                                          **
 **   A copy is found in the file epl-v10.html and important notices to the  **
 **   license from the team is found in the textfile LICENSE.txt distributed **
 **   in this package.                                                       **
 **                                                                          **
 **   This copyright notice MUST APPEAR in all copies of the file!           **
 **                                                                          **
 **   Contributors:                                                          **
 **     RSSOwl - initial API and implementation (bpasero@rssowl.org)         **
 **                                                                          **
 **  **********************************************************************  */

package net.sourceforge.rssowl.controller;

import net.sourceforge.rssowl.model.TreeItemData;
import net.sourceforge.rssowl.util.GlobalSettings;
import net.sourceforge.rssowl.util.shop.WidgetShop;

import org.eclipse.swt.custom.StyledText;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.MenuItem;
import org.eclipse.swt.widgets.Text;
import org.eclipse.swt.widgets.ToolItem;
import org.eclipse.swt.widgets.Tree;

import java.util.ArrayList;

/**
 * The MenuManager manages the disabled / enabled state of some MenuItems and
 * ToolItems. For example, if no news is opened in the TabFolder, MenuItems like
 * Export get disabled.
 * 
 * @author <a href="mailto:bpasero@rssowl.org">Benjamin Pasero </a>
 * @version 1.2.3
 */
public class MenuManager {
  private static MenuItem addFavoriteMenu;
  private static MenuItem blogNews;
  private static MenuItem closeAllTabs;
  private static ToolItem closeAllTabsToolItem;
  private static MenuItem closeCurrentTab;
  private static ToolItem closeOtherTabsToolItem;
  private static ToolItem closeTabToolItem;
  private static MenuItem copy;
  private static MenuItem copyUrlItem;
  private static MenuItem cut;
  private static MenuItem delete;
  private static MenuItem exportHTML;
  private static MenuItem exportMenuItem;
  private static MenuItem exportPDF;
  private static MenuItem exportRTF;
  private static ToolItem exportToolItem;
  private static MenuItem gotoNextTab;
  private static ToolItem gotoNextTabToolItem;
  private static MenuItem gotoPreviousTab;
  private static ToolItem gotoPreviousTabToolItem;
  private static boolean isCloseAllTabsToolItemEnabled = true;
  private static boolean isCloseOtherTabsToolItemEnabled = true;
  private static boolean isCloseTabToolItemEnabled = true;
  private static boolean isExportToolItemEnabled = true;
  private static boolean isMarkAllReadToolItemEnabled = true;
  private static boolean isNewCategorySubMenuItemEnabled = true;
  private static boolean isNewsTipToolItemEnabled = true;
  private static boolean isNewSubCategorySubMenuItemEnabled = true;
  private static boolean isNextTabToolItemEnabled = true;
  private static boolean isPreviousTabToolItemEnabled = true;
  private static boolean isReloadToolItemEnabled = true;
  private static boolean isSearchToolItemEnabled = true;
  private static Text liveSearch;
  private static MenuItem mailLinkToFriend;
  private static MenuItem markAllReadItem;
  private static ToolItem markAllReadToolItem;
  private static MenuItem markUnreadItem;
  private static MenuItem newCategory;
  private static MenuItem newCategorySubMenuItem;
  private static ToolItem newsTipToolItem;
  private static MenuItem newSubCategory;
  private static MenuItem newSubCategorySubMenuItem;
  private static MenuItem paste;
  private static MenuItem print;
  private static MenuItem properties;
  private static MenuItem readNewsInHTML;
  private static MenuItem readNewsInPDF;
  private static MenuItem readNewsInRTF;
  private static MenuItem reload;
  private static ToolItem reloadToolItem;
  private static MenuItem saveRSS;
  private static MenuItem search;
  private static ToolItem searchToolItem;
  private static MenuItem selectAll;

  /** A search did not brought any results */
  static final int EMPTY_SEARCH_TAB = 9;

  /** More than one tab */
  static final int MORE_THAN_ONE_TAB_OPENED = 8;

  /** State: Tab is displaying news and a header is selected */
  static final int NEWS_HEADER_SELECTED = 6;

  /** State: Newstext view is empty */
  static final int NEWS_TEXT_EMPTY = 4;

  /** State: Newstext view is filled with text */
  static final int NEWS_TEXT_FILLED = 3;

  /** Only one tab or less opened */
  static final int ONE_ZERO_TAB_OPENED = 7;

  /** State: Tab is not displaying any tab */
  static final int TAB_EMPTY = 5;

  /** State: Tab is not displaying an warning */
  static final int TAB_ERROR = 10;

  /** State: Tab is displaying a newsfeed */
  static final int TAB_FILLED_WITH_NEWS = 2;

  /** State: Tab is not displaying a newsfeed */
  static final int TAB_NO_NEWS = 1;

  /** State: Tree Selection is a Blogroll Category */
  static final int TREE_SELECTION_BLOGROLL_CAT = 13;

  /** State: Tree Selection is a Blogroll Favorite */
  static final int TREE_SELECTION_BLOGROLL_FAV = 14;

  /** State: Tree Selection is a Category */
  static final int TREE_SELECTION_CATEGORY = 12;

  /** State: Tree Selection is Empty */
  static final int TREE_SELECTION_EMPTY = 15;

  /** State: Tree Selection is a Favorite */
  static final int TREE_SELECTION_FAVORITE = 11;

  /** This utility class constructor is hidden */
  private MenuManager() {
  // Protect default constructor
  }

  /**
   * Init the mnemonics for the given Menu
   * 
   * @param menu The Menu to init the Mnemonics
   */
  public static void initMnemonics(Menu menu) {
    initMnemonics(menu.getItems());
  }

  /**
   * These method disables general menuitems of the main menu in RSSOwl.
   */
  private static void disableGeneralItems() {

    /** Disable other items */
    if (GlobalSettings.useIText()) {
      exportPDF.setEnabled(false);
      exportRTF.setEnabled(false);
      readNewsInPDF.setEnabled(false);
      readNewsInRTF.setEnabled(false);
    }

    exportHTML.setEnabled(false);
    readNewsInHTML.setEnabled(false);

    saveRSS.setEnabled(false);
    search.setEnabled(false);

    if (WidgetShop.isset(searchToolItem))
      searchToolItem.setEnabled(false);
    isSearchToolItemEnabled = false;

    reload.setEnabled(false);

    if (WidgetShop.isset(reloadToolItem))
      reloadToolItem.setEnabled(false);
    isReloadToolItemEnabled = false;

    addFavoriteMenu.setEnabled(false);
    markAllReadItem.setEnabled(false);

    if (WidgetShop.isset(markAllReadToolItem))
      markAllReadToolItem.setEnabled(false);
    isMarkAllReadToolItemEnabled = false;

    closeAllTabs.setEnabled(false);
    closeCurrentTab.setEnabled(false);

    if (WidgetShop.isset(closeAllTabsToolItem))
      closeAllTabsToolItem.setEnabled(false);
    isCloseAllTabsToolItemEnabled = false;

    if (WidgetShop.isset(closeTabToolItem))
      closeTabToolItem.setEnabled(false);
    isCloseTabToolItemEnabled = false;

    mailLinkToFriend.setEnabled(false);

    if (WidgetShop.isset(newsTipToolItem))
      newsTipToolItem.setEnabled(false);
    isNewsTipToolItemEnabled = false;

    blogNews.setEnabled(false);
    copyUrlItem.setEnabled(false);
    markUnreadItem.setEnabled(false);
  }

  /**
   * Disable all of the MenuItems in the Edit Menu
   */
  private static void setEditMenuDisabled() {
    setEditMenuEnabled(false, false, false, false, false, false);
  }

  /**
   * Set the state of the MenuItems in the Edit Menu
   * 
   * @param cut Enabled if TRUE
   * @param copy Enabled if TRUE
   * @param delete Enabled if TRUE
   * @param paste Enabled if TRUE
   * @param selectAll Enabled if TRUE
   * @param properties Enabled if TRUE
   */
  private static void setEditMenuEnabled(boolean cut, boolean copy, boolean delete, boolean paste, boolean selectAll, boolean properties) {
    MenuManager.cut.setEnabled(cut);
    MenuManager.copy.setEnabled(copy);
    MenuManager.delete.setEnabled(delete);
    MenuManager.paste.setEnabled(paste);
    MenuManager.selectAll.setEnabled(selectAll);
    MenuManager.properties.setEnabled(properties);
  }

  /**
   * Check the selected Control and enable / disable MenuItems from the edit
   * menu in dependance of the selected Control.
   */
  public static void handleEditMenuState() {

    /** Retrieve the Focus Control */
    Control control = GUI.display.getFocusControl();

    /** No focus control available, disable Edit Menu and return */
    if (!WidgetShop.isset(control)) {
      setEditMenuDisabled();
      return;
    }

    /** Control is a Text widget */
    if (control instanceof Text) {
      boolean isEditable = ((Text) control).getEditable();
      boolean isTextSelected = ((Text) control).getSelectionCount() > 0;

      setEditMenuEnabled(isEditable && isTextSelected, isTextSelected, false, isEditable, true, false);
      return;
    }

    /** Control is a StyledText widget */
    if (control instanceof StyledText) {
      boolean isEditable = ((StyledText) control).getEditable();
      boolean isTextSelected = ((StyledText) control).getSelectionCount() > 0;

      setEditMenuEnabled(isEditable && isTextSelected, isTextSelected, false, isEditable, true, false);
      return;
    }

    /** Control is a Combo widget */
    if (control instanceof Combo) {
      setEditMenuEnabled(true, true, false, true, true, false);
      return;
    }

    /** Control is a Tree */
    if (control instanceof Tree) {
      Tree tree = (Tree) control;

      /** Check if Tree is favorite Tree with a selection */
      if (tree.getSelectionCount() > 0 && tree.getSelection()[0].getData() != null) {
        Object data = tree.getSelection()[0].getData();

        /** Check if a favorite, category or blogroll category is selected */
        if (data instanceof TreeItemData && (((TreeItemData) data).isFavorite() || ((TreeItemData) data).isCategory() || ((TreeItemData) data).isBlogroll()))
          setEditMenuEnabled(false, false, true, false, false, true);

        /** Blogroll Favorite is selected */
        else
          setEditMenuEnabled(false, false, true, false, false, false);

        return;
      }
    }

    /** Control is any other widget, disable Edit Menu in that case */
    setEditMenuDisabled();
  }

  /**
   * Recursivly sets the mnemonic for each Menu and MenuItem. The method trys to
   * set a mnemonic that has not yet been set (trys all chars of the name).
   * 
   * @param items The MenuItems
   */
  static void initMnemonics(MenuItem items[]) {

    /** First check if mnemonics should be displayed */
    if (!GlobalSettings.shouldShowMnemonics())
      return;

    /** Store chars that have been used as mnemonic */
    ArrayList chars = new ArrayList();

    /** For each MenuItem */
    for (int a = 0; a < items.length; a++) {
      String name = items[a].getText();

      /** Replace any & that are existing */
      name = name.replaceAll("&", "");

      /** For each char in the name */
      for (int b = 0; b < name.length(); b++) {

        /** Check if char is available and no whitespace */
        if (name.substring(b, b + 1) != null && !name.substring(b, b + 1).equals(" ")) {

          /** Check if char has been used as mnemonic before */
          if (!chars.contains(name.substring(b, b + 1).toLowerCase())) {

            /** Create mnemonic */
            StringBuffer itemText = new StringBuffer(name.substring(0, b));
            itemText.append("&").append(name.substring(b, name.length()));

            /** Set mnemonic */
            items[a].setText(itemText.toString());

            /** Add char as used mnemonic */
            chars.add(name.substring(b, b + 1).toLowerCase());
            break;
          }
        }
      }

      /** Also check MenuItems ob possible Sub-Menus */
      if (items[a].getMenu() != null)
        initMnemonics(items[a].getMenu().getItems());
    }
  }

  /**
   * Notify about a new State.
   * 
   * @param state One of the state constants
   */
  static void notifyState(int state) {

    /** React on the new state */
    switch (state) {

      /** Tab is not displaying a newsfeed */
      case TAB_NO_NEWS:
        disableGeneralItems();
        liveSearch.setEnabled(false);
        closeAllTabs.setEnabled(true);
        closeCurrentTab.setEnabled(true);

        if (WidgetShop.isset(closeTabToolItem))
          closeTabToolItem.setEnabled(true);
        isCloseTabToolItemEnabled = true;

        if (WidgetShop.isset(closeAllTabsToolItem))
          closeAllTabsToolItem.setEnabled(true);
        isCloseAllTabsToolItemEnabled = true;

        break;

      /** Tab is displaying an error */
      case TAB_ERROR:
        disableGeneralItems();
        liveSearch.setEnabled(false);
        addFavoriteMenu.setEnabled(true);
        reload.setEnabled(true);

        if (WidgetShop.isset(reloadToolItem))
          reloadToolItem.setEnabled(true);
        isReloadToolItemEnabled = true;

        closeAllTabs.setEnabled(true);
        closeCurrentTab.setEnabled(true);

        if (WidgetShop.isset(closeTabToolItem))
          closeTabToolItem.setEnabled(true);
        isCloseTabToolItemEnabled = true;

        if (WidgetShop.isset(closeAllTabsToolItem))
          closeAllTabsToolItem.setEnabled(true);
        isCloseAllTabsToolItemEnabled = true;

        break;

      /** Tab is not containing Tabs */
      case TAB_EMPTY:
        disableGeneralItems();
        liveSearch.setEnabled(false);
        break;

      /** A search did not brought any results */
      case EMPTY_SEARCH_TAB:
        disableGeneralItems();
        search.setEnabled(true);

        if (WidgetShop.isset(searchToolItem))
          searchToolItem.setEnabled(true);
        isSearchToolItemEnabled = true;

        liveSearch.setEnabled(true);
        reload.setEnabled(true);

        if (WidgetShop.isset(reloadToolItem))
          reloadToolItem.setEnabled(true);
        isReloadToolItemEnabled = true;

        addFavoriteMenu.setEnabled(true);
        closeAllTabs.setEnabled(true);
        closeCurrentTab.setEnabled(true);

        if (WidgetShop.isset(closeTabToolItem))
          closeTabToolItem.setEnabled(true);
        isCloseTabToolItemEnabled = true;

        if (WidgetShop.isset(closeAllTabsToolItem))
          closeAllTabsToolItem.setEnabled(true);
        isCloseAllTabsToolItemEnabled = true;

        break;

      /** Tab is displaying a newsfeed */
      case TAB_FILLED_WITH_NEWS:
        if (GlobalSettings.useIText()) {
          exportPDF.setEnabled(true);
          exportRTF.setEnabled(true);
          readNewsInPDF.setEnabled(true);
          readNewsInRTF.setEnabled(true);
        }

        exportHTML.setEnabled(true);
        readNewsInHTML.setEnabled(true);

        saveRSS.setEnabled(true);
        search.setEnabled(true);

        if (WidgetShop.isset(searchToolItem))
          searchToolItem.setEnabled(true);
        isSearchToolItemEnabled = true;

        liveSearch.setEnabled(true);
        reload.setEnabled(true);

        if (WidgetShop.isset(reloadToolItem))
          reloadToolItem.setEnabled(true);
        isReloadToolItemEnabled = true;

        addFavoriteMenu.setEnabled(true);
        markAllReadItem.setEnabled(true);

        if (WidgetShop.isset(markAllReadToolItem))
          markAllReadToolItem.setEnabled(true);
        isMarkAllReadToolItemEnabled = true;

        closeAllTabs.setEnabled(true);
        closeCurrentTab.setEnabled(true);

        if (WidgetShop.isset(closeTabToolItem))
          closeTabToolItem.setEnabled(true);
        isCloseTabToolItemEnabled = true;

        if (WidgetShop.isset(closeAllTabsToolItem))
          closeAllTabsToolItem.setEnabled(true);
        isCloseAllTabsToolItemEnabled = true;

        mailLinkToFriend.setEnabled(false);

        if (WidgetShop.isset(newsTipToolItem))
          newsTipToolItem.setEnabled(false);
        isNewsTipToolItemEnabled = false;

        blogNews.setEnabled(false);
        copyUrlItem.setEnabled(false);
        markUnreadItem.setEnabled(false);

        break;

      /** News header selected in tab */
      case NEWS_HEADER_SELECTED:
        mailLinkToFriend.setEnabled(true);

        if (WidgetShop.isset(newsTipToolItem))
          newsTipToolItem.setEnabled(true);
        isNewsTipToolItemEnabled = true;

        blogNews.setEnabled(true);
        copyUrlItem.setEnabled(true);
        markUnreadItem.setEnabled(true);

        break;

      /** Newstext view is filled with text */
      case NEWS_TEXT_FILLED:
        if (GlobalSettings.usePrinting())
          print.setEnabled(true);

        break;

      /** Newstext view is empty */
      case NEWS_TEXT_EMPTY:
        if (GlobalSettings.usePrinting())
          print.setEnabled(false);

        break;

      /** Only one tab or less opened */
      case ONE_ZERO_TAB_OPENED:
        gotoPreviousTab.setEnabled(false);
        gotoNextTab.setEnabled(false);

        if (WidgetShop.isset(closeOtherTabsToolItem))
          closeOtherTabsToolItem.setEnabled(false);
        isCloseOtherTabsToolItemEnabled = false;

        if (WidgetShop.isset(gotoPreviousTabToolItem))
          gotoPreviousTabToolItem.setEnabled(false);
        isPreviousTabToolItemEnabled = false;

        if (WidgetShop.isset(gotoNextTabToolItem))
          gotoNextTabToolItem.setEnabled(false);
        isNextTabToolItemEnabled = false;

        break;

      /** More than one tab */
      case MORE_THAN_ONE_TAB_OPENED:
        gotoPreviousTab.setEnabled(true);
        gotoNextTab.setEnabled(true);

        if (WidgetShop.isset(closeOtherTabsToolItem))
          closeOtherTabsToolItem.setEnabled(true);
        isCloseOtherTabsToolItemEnabled = true;

        if (WidgetShop.isset(gotoPreviousTabToolItem))
          gotoPreviousTabToolItem.setEnabled(true);
        isPreviousTabToolItemEnabled = true;

        if (WidgetShop.isset(gotoNextTabToolItem))
          gotoNextTabToolItem.setEnabled(true);
        isNextTabToolItemEnabled = true;

        break;

      /** Tree Selection is a Favorite */
      case TREE_SELECTION_FAVORITE:
        newCategory.setEnabled(true);
        newSubCategory.setEnabled(false);
        exportMenuItem.setEnabled(true);

        if (WidgetShop.isset(newCategorySubMenuItem))
          newCategorySubMenuItem.setEnabled(true);
        isNewCategorySubMenuItemEnabled = true;

        if (WidgetShop.isset(newSubCategorySubMenuItem))
          newSubCategorySubMenuItem.setEnabled(false);
        isNewSubCategorySubMenuItemEnabled = false;

        if (WidgetShop.isset(exportToolItem))
          exportToolItem.setEnabled(true);
        isExportToolItemEnabled = true;

        break;

      /** Tree Selection is a Category */
      case TREE_SELECTION_CATEGORY:
        newCategory.setEnabled(true);
        newSubCategory.setEnabled(true);
        exportMenuItem.setEnabled(true);

        if (WidgetShop.isset(newCategorySubMenuItem))
          newCategorySubMenuItem.setEnabled(true);
        isNewCategorySubMenuItemEnabled = true;

        if (WidgetShop.isset(newSubCategorySubMenuItem))
          newSubCategorySubMenuItem.setEnabled(true);
        isNewSubCategorySubMenuItemEnabled = true;

        if (WidgetShop.isset(exportToolItem))
          exportToolItem.setEnabled(true);
        isExportToolItemEnabled = true;

        break;

      /** Tree Selection is a Blogroll Category */
      case TREE_SELECTION_BLOGROLL_CAT:
        newCategory.setEnabled(true);
        newSubCategory.setEnabled(false);
        exportMenuItem.setEnabled(true);

        if (WidgetShop.isset(newCategorySubMenuItem))
          newCategorySubMenuItem.setEnabled(true);
        isNewCategorySubMenuItemEnabled = true;

        if (WidgetShop.isset(newSubCategorySubMenuItem))
          newSubCategorySubMenuItem.setEnabled(false);
        isNewSubCategorySubMenuItemEnabled = false;

        if (WidgetShop.isset(exportToolItem))
          exportToolItem.setEnabled(true);
        isExportToolItemEnabled = true;
        break;

      /** Tree Selection is a Blogroll Favorite */
      case TREE_SELECTION_BLOGROLL_FAV:
        newCategory.setEnabled(false);
        newSubCategory.setEnabled(false);
        exportMenuItem.setEnabled(true);

        if (WidgetShop.isset(newCategorySubMenuItem))
          newCategorySubMenuItem.setEnabled(false);
        isNewCategorySubMenuItemEnabled = false;

        if (WidgetShop.isset(newSubCategorySubMenuItem))
          newSubCategorySubMenuItem.setEnabled(false);
        isNewSubCategorySubMenuItemEnabled = false;

        if (WidgetShop.isset(exportToolItem))
          exportToolItem.setEnabled(true);
        isExportToolItemEnabled = true;

        break;

      /** Tree Selection is Empty */
      case TREE_SELECTION_EMPTY:
        newCategory.setEnabled(true);
        newSubCategory.setEnabled(false);
        exportMenuItem.setEnabled(false);

        if (WidgetShop.isset(newCategorySubMenuItem))
          newCategorySubMenuItem.setEnabled(true);
        isNewCategorySubMenuItemEnabled = true;

        if (WidgetShop.isset(newSubCategorySubMenuItem))
          newSubCategorySubMenuItem.setEnabled(false);
        isNewSubCategorySubMenuItemEnabled = false;

        if (WidgetShop.isset(exportToolItem))
          exportToolItem.setEnabled(false);
        isExportToolItemEnabled = false;
        break;
    }
  }

  /**
   * Register a menuitem to the MenuManager
   * 
   * @param blogNews The MenuItem "Blog News"
   */
  static void regiserBlogNews(MenuItem blogNews) {
    MenuManager.blogNews = blogNews;
  }

  /**
   * Register a Text input field to the MenuManager
   * 
   * @param liveSearch The live search Text field
   */
  static void regiserLiveSearch(Text liveSearch) {
    MenuManager.liveSearch = liveSearch;
  }

  /**
   * Register a menuitem to the MenuManager
   * 
   * @param print The MenuItem "Print"
   */
  static void regiserPrint(MenuItem print) {
    MenuManager.print = print;
  }

  /**
   * Register a menuitem to the MenuManager
   * 
   * @param addFavoriteMenu The MenuItem "Add to favorits"
   */
  static void registerAddFavoriteMenu(MenuItem addFavoriteMenu) {
    MenuManager.addFavoriteMenu = addFavoriteMenu;
  }

  /**
   * Register a toolitem to the MenuManager
   * 
   * @param closeAllTabsToolItem The ToolItem "Close All"
   */
  static void registerCloseAllTabsToolItem(ToolItem closeAllTabsToolItem) {
    MenuManager.closeAllTabsToolItem = closeAllTabsToolItem;
    if (!isCloseAllTabsToolItemEnabled)
      closeAllTabsToolItem.setEnabled(false);
  }

  /**
   * Register a toolitem to the MenuManager
   * 
   * @param closeOtherTabsToolItem The ToolItem "Close Others"
   */
  static void registerCloseOtherTabsToolItem(ToolItem closeOtherTabsToolItem) {
    MenuManager.closeOtherTabsToolItem = closeOtherTabsToolItem;
    if (!isCloseOtherTabsToolItemEnabled)
      closeOtherTabsToolItem.setEnabled(false);
  }

  /**
   * Register menuitems to the MenuManager
   * 
   * @param closeCurrentTab The MenuItem "Close"
   * @param closeAllTabs The MenuItem "Close all"
   */
  static void registerCloseTab(MenuItem closeCurrentTab, MenuItem closeAllTabs) {
    MenuManager.closeCurrentTab = closeCurrentTab;
    MenuManager.closeAllTabs = closeAllTabs;
  }

  /**
   * Register a toolitem to the MenuManager
   * 
   * @param closeTabToolItem The ToolItem "Close"
   */
  static void registerCloseTabToolItem(ToolItem closeTabToolItem) {
    MenuManager.closeTabToolItem = closeTabToolItem;
    if (!isCloseTabToolItemEnabled)
      closeTabToolItem.setEnabled(false);
  }

  /**
   * Register a menuitem to the MenuManager
   * 
   * @param copyUrlItem The MenuItem "Copy URL"
   */
  static void registerCopyLink(MenuItem copyUrlItem) {
    MenuManager.copyUrlItem = copyUrlItem;
  }

  /**
   * Register the "Edit" Menu to the MenuManager
   * 
   * @param copy The MenuItem "Copy"
   * @param paste The MenuItem "Paste"
   * @param cut The MenuItem "Cut"
   * @param selectAll The MenuItem "Select All"
   * @param delete The MenuItem "Delete"
   * @param properties The MenuItem "Properties"
   */
  static void registerEditMenu(MenuItem copy, MenuItem paste, MenuItem cut, MenuItem selectAll, MenuItem delete, MenuItem properties) {
    MenuManager.copy = copy;
    MenuManager.paste = paste;
    MenuManager.cut = cut;
    MenuManager.selectAll = selectAll;
    MenuManager.delete = delete;
    MenuManager.properties = properties;
  }

  /**
   * Register menuitems to the MenuManager
   * 
   * @param exportPDF The MenuItem "Export to PDF"
   * @param exportRTF The MenuItem "Export to RTF"
   * @param exportHTML The MenuItem "Export to HTML"
   */
  static void registerExport(MenuItem exportPDF, MenuItem exportRTF, MenuItem exportHTML) {
    MenuManager.exportPDF = exportPDF;
    MenuManager.exportRTF = exportRTF;
    MenuManager.exportHTML = exportHTML;
  }

  /**
   * Register menuitems to the MenuManager
   * 
   * @param exportMenuItem The MenuItem "Export"
   */
  static void registerExportMenuItem(MenuItem exportMenuItem) {
    MenuManager.exportMenuItem = exportMenuItem;
  }

  /**
   * Register toolitems to the MenuManager
   * 
   * @param exportToolItem The ToolItem "Export"
   */
  static void registerExportToolItem(ToolItem exportToolItem) {
    MenuManager.exportToolItem = exportToolItem;

    if (!isExportToolItemEnabled)
      exportToolItem.setEnabled(false);
  }

  /**
   * Register a toolitem to the MenuManager
   * 
   * @param gotoNextTabToolItem The ToolItem "Next Tab"
   */
  static void registerGotoNextTabToolItem(ToolItem gotoNextTabToolItem) {
    MenuManager.gotoNextTabToolItem = gotoNextTabToolItem;
    if (!isNextTabToolItemEnabled)
      gotoNextTabToolItem.setEnabled(false);
  }

  /**
   * Register a toolitem to the MenuManager
   * 
   * @param gotoPreviousTabToolItem The ToolItem "Previous Tab"
   */
  static void registerGotoPreviousTabToolItem(ToolItem gotoPreviousTabToolItem) {
    MenuManager.gotoPreviousTabToolItem = gotoPreviousTabToolItem;
    if (!isPreviousTabToolItemEnabled)
      gotoPreviousTabToolItem.setEnabled(false);
  }

  /**
   * Register menuitems to the MenuManager
   * 
   * @param gotoPreviousTab The MenuItem "Previous tab"
   * @param gotoNextUnreadTab The MenuItem "Next tab"
   */
  static void registerGotoTab(MenuItem gotoPreviousTab, MenuItem gotoNextUnreadTab) {
    MenuManager.gotoPreviousTab = gotoPreviousTab;
    MenuManager.gotoNextTab = gotoNextUnreadTab;
  }

  /**
   * Register a menuitem to the MenuManager
   * 
   * @param markAllReadItem The MenuItem "Mark All Read"
   */
  static void registerMarkAllReadItem(MenuItem markAllReadItem) {
    MenuManager.markAllReadItem = markAllReadItem;
  }

  /**
   * Register a toolitem to the MenuManager
   * 
   * @param markAllReadToolItem The ToolItem "Mark All Read"
   */
  static void registerMarkAllReadToolItem(ToolItem markAllReadToolItem) {
    MenuManager.markAllReadToolItem = markAllReadToolItem;
    if (!isMarkAllReadToolItemEnabled)
      markAllReadToolItem.setEnabled(false);
  }

  /**
   * Register a menuitem to the MenuManager
   * 
   * @param markUnreadItem The MenuItem "Mark unread"
   */
  static void registerMarkUnreadItem(MenuItem markUnreadItem) {
    MenuManager.markUnreadItem = markUnreadItem;
  }

  /**
   * Register menuitems to the MenuManager
   * 
   * @param newCategory The MenuItem "New Category"
   * @param newSubCategory The MenuItem "New Sub-Category"
   */
  static void registerNewItems(MenuItem newCategory, MenuItem newSubCategory) {
    MenuManager.newCategory = newCategory;
    MenuManager.newSubCategory = newSubCategory;
  }

  /**
   * Register a menuitem to the MenuManager
   * 
   * @param mailLinkToFriend The MenuItem "Mail NewsTip"
   */
  static void registerNewsTipMail(MenuItem mailLinkToFriend) {
    MenuManager.mailLinkToFriend = mailLinkToFriend;
  }

  /**
   * Register a toolitem to the MenuManager
   * 
   * @param newsTipToolItem The ToolItem "Mail NewsTip"
   */
  static void registerNewsTipToolItem(ToolItem newsTipToolItem) {
    MenuManager.newsTipToolItem = newsTipToolItem;
    if (!isNewsTipToolItemEnabled)
      newsTipToolItem.setEnabled(false);
  }

  /**
   * Register menuitems to the MenuManager. These come from the "New"-Dropdown
   * of the ToolBar.
   * 
   * @param newCategorySubMenuItem The Sub-Menu "New Category"
   * @param newSubCategorySubMenuItem The Sub-Menu "New Sub-Category"
   */
  static void registerNewSubMenuItems(MenuItem newCategorySubMenuItem, MenuItem newSubCategorySubMenuItem) {
    MenuManager.newCategorySubMenuItem = newCategorySubMenuItem;
    MenuManager.newSubCategorySubMenuItem = newSubCategorySubMenuItem;

    if (!isNewCategorySubMenuItemEnabled)
      newCategorySubMenuItem.setEnabled(false);

    if (!isNewSubCategorySubMenuItemEnabled)
      newSubCategorySubMenuItem.setEnabled(false);
  }

  /**
   * Register menuitems to the MenuManager
   * 
   * @param readNewsInPDF The MenuItem "Display Newsfeed in PDF"
   * @param readNewsInRTF The MenuItem "Display Newsfeed in RTF"
   * @param readNewsInHTML The MenuItem "Display Newsfeed in HTML"
   */
  static void registerReadNewsInDocument(MenuItem readNewsInPDF, MenuItem readNewsInRTF, MenuItem readNewsInHTML) {
    MenuManager.readNewsInPDF = readNewsInPDF;
    MenuManager.readNewsInRTF = readNewsInRTF;
    MenuManager.readNewsInHTML = readNewsInHTML;
  }

  /**
   * Register a menuitem to the MenuManager
   * 
   * @param reload The MenuItem "Reload"
   */
  static void registerReload(MenuItem reload) {
    MenuManager.reload = reload;
  }

  /**
   * Register a toolitem to the MenuManager
   * 
   * @param reloadToolItem The ToolItem "Reload"
   */
  static void registerReloadToolItem(ToolItem reloadToolItem) {
    MenuManager.reloadToolItem = reloadToolItem;
    if (!isReloadToolItemEnabled)
      reloadToolItem.setEnabled(false);
  }

  /**
   * Register a menuitem to the MenuManager
   * 
   * @param saveRSS The MenuItem "Save RSS"
   */
  static void registerSaveRSS(MenuItem saveRSS) {
    MenuManager.saveRSS = saveRSS;
  }

  /**
   * Register a menuitem to the MenuManager
   * 
   * @param search The MenuItem "Search"
   */
  static void registerSearch(MenuItem search) {
    MenuManager.search = search;
  }

  /**
   * Register a toolitem to the MenuManager
   * 
   * @param searchToolItem The ToolItem "Search"
   */
  static void registerSearchToolItem(ToolItem searchToolItem) {
    MenuManager.searchToolItem = searchToolItem;
    if (!isSearchToolItemEnabled)
      searchToolItem.setEnabled(false);
  }
}