/*   **********************************************************************  **
 **   Copyright notice                                                       **
 **                                                                          **
 **   (c) 2003-2006 RSSOwl Development Team                                  **
 **   http://www.rssowl.org/                                                 **
 **                                                                          **
 **   All rights reserved                                                    **
 **                                                                          **
 **   This program and the accompanying materials are made available under   **
 **   the terms of the Eclipse Public License 1.0 which accompanies this     **
 **   distribution, and is available at:                                     **
 **   http://www.rssowl.org/legal/epl-v10.html                               **
 **                                                                          **
 **   A copy is found in the file epl-v10.html and important notices to the  **
 **   license from the team is found in the textfile LICENSE.txt distributed **
 **   in this package.                                                       **
 **                                                                          **
 **   This copyright notice MUST APPEAR in all copies of the file!           **
 **                                                                          **
 **   Contributors:                                                          **
 **     RSSOwl - initial API and implementation (bpasero@rssowl.org)         **
 **                                                                          **
 **  **********************************************************************  */

package net.sourceforge.rssowl.controller.dialog;

import net.sourceforge.rssowl.controller.GUI;
import net.sourceforge.rssowl.util.GlobalSettings;
import net.sourceforge.rssowl.util.shop.FontShop;
import net.sourceforge.rssowl.util.shop.LayoutDataShop;
import net.sourceforge.rssowl.util.shop.LayoutShop;
import net.sourceforge.rssowl.util.shop.PaintShop;
import net.sourceforge.rssowl.util.shop.WidgetShop;

import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.IMessageProvider;
import org.eclipse.jface.dialogs.TitleAreaDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;

/**
 * The ConfirmDeleteDialog asks the user if he wants to proceed with the
 * deletion of a favorite, category or blogroll. A checkbutton allows to never
 * show the dialog again.
 * 
 * @author <a href="mailto:bpasero@rssowl.org">Benjamin Pasero </a>
 * @version 1.2.3 IconAndMessageDialog
 */
public class ConfirmDeleteDialog extends TitleAreaDialog {

  /** Confirm dialog for the deletion of a blogroll */
  public static final int TYPE_BLOGROLL = 2;

  /** Confirm dialog for the deletion of a category */
  public static final int TYPE_CATEGORY = 1;

  /** Confirm dialog for the deletion of a favorite */
  public static final int TYPE_FAVORITE = 0;

  /** Min. width of the dialog in DLUs */
  private static final int dialogMinWidth = 320;

  private String dialogHeaderMessage;
  private String dialogMessage;
  private Button neverAskAgainCheck;
  private String title;
  private int type;

  /**
   * Instantiate a new ConfirmDeleteDialog
   * 
   * @param parentShell The parent shell
   * @param title The title of the dialog
   * @param dialogHeaderMessage The info message
   * @param dialogMessage The dialog message
   * @param type The type of the deletion
   */
  public ConfirmDeleteDialog(Shell parentShell, String title, String dialogHeaderMessage, String dialogMessage, int type) {
    super(parentShell);
    this.title = title;
    this.dialogMessage = dialogMessage;
    this.dialogHeaderMessage = dialogHeaderMessage;
    this.type = type;
  }

  /**
   * @see org.eclipse.jface.dialogs.Dialog#close()
   */
  public boolean close() {

    /** Dispose title image */
    getTitleImageLabel().getImage().dispose();

    return super.close();
  }

  /**
   * @see org.eclipse.jface.dialogs.Dialog#buttonPressed(int)
   */
  protected void buttonPressed(int buttonId) {

    /** Update Global settings if check was checked */
    if (buttonId == IDialogConstants.OK_ID && neverAskAgainCheck.getSelection()) {
      switch (type) {
        case TYPE_FAVORITE:
          GlobalSettings.confirmFavoriteDeletion = false;
          break;

        case TYPE_CATEGORY:
          GlobalSettings.confirmCategoryDeletion = false;
          break;

        case TYPE_BLOGROLL:
          GlobalSettings.confirmBlogrollDeletion = false;
          break;
      }
    }

    super.buttonPressed(buttonId);
  }

  /**
   * @see org.eclipse.jface.window.Window#configureShell(org.eclipse.swt.widgets.Shell)
   */
  protected void configureShell(Shell shell) {
    super.configureShell(shell);

    /** On Mac do not set Shell Image since it will change the Dock Image */
    if (!GlobalSettings.isMac())
      shell.setImages(PaintShop.iconOwl);

    shell.setText(title);
  }

  /**
   * @see org.eclipse.jface.dialogs.Dialog#createButtonsForButtonBar(org.eclipse.swt.widgets.Composite)
   */
  protected void createButtonsForButtonBar(Composite parent) {

    /** Override parent, DLU dependant margins */
    ((GridLayout) parent.getLayout()).marginHeight = 10;
    ((GridLayout) parent.getLayout()).marginWidth = 10;

    /** Button order on Mac is different */
    if (GUI.display.getDismissalAlignment() == SWT.RIGHT) {

      /** Create Buttons */
      Button noButton = createButton(parent, IDialogConstants.CANCEL_ID, GUI.i18n.getTranslation("BUTTON_CANCLE"), false);
      noButton.setFont(FontShop.dialogFont);

      Button yesButton = createButton(parent, IDialogConstants.OK_ID, GUI.i18n.getTranslation("MENU_EDIT_DELETE"), true);
      yesButton.setFont(FontShop.dialogFont);
      yesButton.setFocus();
    } else {

      /** Create Buttons */
      Button yesButton = createButton(parent, IDialogConstants.OK_ID, GUI.i18n.getTranslation("MENU_EDIT_DELETE"), true);
      yesButton.setFont(FontShop.dialogFont);
      yesButton.setFocus();

      Button noButton = createButton(parent, IDialogConstants.CANCEL_ID, GUI.i18n.getTranslation("BUTTON_CANCLE"), false);
      noButton.setFont(FontShop.dialogFont);
    }
  }

  /**
   * @see org.eclipse.jface.dialogs.Dialog#createDialogArea(org.eclipse.swt.widgets.Composite)
   */
  protected Control createDialogArea(Composite parent) {

    /** Composite to hold all components */
    Composite composite = new Composite((Composite) super.createDialogArea(parent), SWT.NONE);
    composite.setLayout(new GridLayout(1, false));
    composite.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));

    /** Title Image */
    setTitleImage(PaintShop.loadImage("/img/icons/trash_big.gif"));

    /** Title Message */
    setMessage(dialogHeaderMessage, IMessageProvider.INFORMATION);

    /** Dialog Message */
    Label dialogMessageLabel = new Label(composite, SWT.WRAP);
    dialogMessageLabel.setText(dialogMessage);
    dialogMessageLabel.setLayoutData(new GridData(SWT.FILL, SWT.BEGINNING, true, false));
    dialogMessageLabel.setFont(FontShop.dialogFont);

    /** Spacer */
    new Label(composite, SWT.NONE);

    /** Checkbox to disable confirm dialog */
    neverAskAgainCheck = new Button(composite, SWT.CHECK);
    neverAskAgainCheck.setFont(FontShop.dialogFont);
    neverAskAgainCheck.setText(GUI.i18n.getTranslation("BUTTON_NEVER_ASK_AGAIN"));

    /** Holder for the separator to the OK and Cancel buttons */
    Composite sepHolder = new Composite(parent, SWT.NONE);
    sepHolder.setLayoutData(LayoutDataShop.createGridData(GridData.FILL_HORIZONTAL, 1));
    sepHolder.setLayout(LayoutShop.createGridLayout(1, 0, 0));

    /** Separator */
    Label separator = new Label(sepHolder, SWT.SEPARATOR | SWT.HORIZONTAL);
    separator.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));

    /** Init Mnemonics */
    WidgetShop.initMnemonics(new Button[] { neverAskAgainCheck });

    return composite;
  }

  /**
   * @see org.eclipse.jface.window.Window#getShellStyle()
   */
  protected int getShellStyle() {
    int style = SWT.TITLE | SWT.BORDER | SWT.APPLICATION_MODAL | getDefaultOrientation();

    /** Follow Apple's Human Interface Guidelines for Application Modal Dialogs */
    if (!GlobalSettings.isMac())
      style |= SWT.CLOSE;

    return style;
  }

  /**
   * @see org.eclipse.jface.dialogs.Dialog#initializeBounds()
   */
  protected void initializeBounds() {
    super.initializeBounds();
    Point bestSize = getShell().computeSize(convertHorizontalDLUsToPixels(dialogMinWidth), SWT.DEFAULT);
    Point location = getInitialLocation(bestSize);
    getShell().setBounds(location.x, location.y, bestSize.x, bestSize.y);
  }

  /**
   * Set the layout data of the button to a GridData with appropriate widths
   * This method was slightly modified so that it is not setting a heightHint.
   * 
   * @param button The button to layout
   */
  protected void setButtonLayoutData(Button button) {
    GridData data = new GridData(GridData.HORIZONTAL_ALIGN_FILL);
    int widthHint = convertHorizontalDLUsToPixels(IDialogConstants.BUTTON_WIDTH);
    data.widthHint = Math.max(widthHint, button.computeSize(SWT.DEFAULT, SWT.DEFAULT, true).x);
    button.setLayoutData(data);
  }
}