/*   **********************************************************************  **
 **   Copyright notice                                                       **
 **                                                                          **
 **   (c) 2003-2006 RSSOwl Development Team                                  **
 **   http://www.rssowl.org/                                                 **
 **                                                                          **
 **   All rights reserved                                                    **
 **                                                                          **
 **   This program and the accompanying materials are made available under   **
 **   the terms of the Eclipse Public License 1.0 which accompanies this     **
 **   distribution, and is available at:                                     **
 **   http://www.rssowl.org/legal/epl-v10.html                               **
 **                                                                          **
 **   A copy is found in the file epl-v10.html and important notices to the  **
 **   license from the team is found in the textfile LICENSE.txt distributed **
 **   in this package.                                                       **
 **                                                                          **
 **   This copyright notice MUST APPEAR in all copies of the file!           **
 **                                                                          **
 **   Contributors:                                                          **
 **     RSSOwl - initial API and implementation (bpasero@rssowl.org)         **
 **                                                                          **
 **  **********************************************************************  */

package net.sourceforge.rssowl.controller.dialog;

import net.sourceforge.rssowl.controller.GUI;
import net.sourceforge.rssowl.util.GlobalSettings;
import net.sourceforge.rssowl.util.shop.FileShop;
import net.sourceforge.rssowl.util.shop.FontShop;
import net.sourceforge.rssowl.util.shop.LayoutDataShop;
import net.sourceforge.rssowl.util.shop.LayoutShop;
import net.sourceforge.rssowl.util.shop.PaintShop;
import net.sourceforge.rssowl.util.shop.WidgetShop;

import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.IMessageProvider;
import org.eclipse.jface.dialogs.TitleAreaDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;

/**
 * Dialog prompts for the path to a newsfeed to display it in RSSOwl
 * 
 * @author <a href="mailto:bpasero@rssowl.org">Benjamin Pasero </a>
 * @version 1.2.3
 */
public class OpenFeedDialog extends TitleAreaDialog {

  /** Min. width of the dialog in DLUs */
  private static final int dialogMinWidth = 320;

  private String dialogMessage;
  private String dialogTitle;
  private String feedPath;
  Text feedPathInput;
  Button okButton;

  /**
   * Creates an input dialog with OK and Cancel buttons. Prompts for a valid
   * path to a feed. Note that the dialog will have no visual representation (no
   * widgets) until it is told to open.
   * <p>
   * Note that the <code>open</code> method blocks for input dialogs.
   * </p>
   * 
   * @param parentShell the parent shell
   * @param dialogTitle the dialog dialogTitle, or <code>null</code> if none
   * @param dialogMessage the dialog dialogMessage, or <code>null</code> if
   * none
   */
  public OpenFeedDialog(Shell parentShell, String dialogTitle, String dialogMessage) {
    super(parentShell);
    this.dialogTitle = dialogTitle;
    this.dialogMessage = dialogMessage;
  }

  /**
   * @see org.eclipse.jface.dialogs.Dialog#close()
   */
  public boolean close() {

    /** Dispose dialogTitle image */
    getTitleImageLabel().getImage().dispose();

    return super.close();
  }

  /**
   * Get the path to the newsfeed
   * 
   * @return String The path to the newsfeed
   */
  public String getFeedPath() {
    return feedPath;
  }

  /**
   * @see org.eclipse.jface.dialogs.Dialog#buttonPressed(int)
   */
  protected void buttonPressed(int buttonId) {
    feedPath = feedPathInput.getText().trim();
    super.buttonPressed(buttonId);
  }

  /**
   * @see org.eclipse.jface.window.Window#configureShell(org.eclipse.swt.widgets.Shell)
   */
  protected void configureShell(Shell shell) {
    super.configureShell(shell);

    /** On Mac do not set Shell Image since it will change the Dock Image */
    if (!GlobalSettings.isMac())
      shell.setImages(PaintShop.iconOwl);

    shell.setText(dialogTitle);
  }

  /**
   * @see org.eclipse.jface.dialogs.Dialog#createButtonsForButtonBar(org.eclipse.swt.widgets.Composite)
   */
  protected void createButtonsForButtonBar(Composite parent) {

    /** Button order on Mac is different */
    if (GUI.display.getDismissalAlignment() == SWT.RIGHT) {

      /** Create Cancel Button */
      createButton(parent, IDialogConstants.CANCEL_ID, GUI.i18n.getTranslation("BUTTON_CANCLE"), false).setFont(FontShop.dialogFont);

      /** Create OK Button */
      okButton = createButton(parent, IDialogConstants.OK_ID, GUI.i18n.getTranslation("BUTTON_OK"), true);
    } else {

      /** Create OK Button */
      okButton = createButton(parent, IDialogConstants.OK_ID, GUI.i18n.getTranslation("BUTTON_OK"), true);

      /** Create Cancel Button */
      createButton(parent, IDialogConstants.CANCEL_ID, GUI.i18n.getTranslation("BUTTON_CANCLE"), false).setFont(FontShop.dialogFont);
    }

    /** Give Focus to input text field */
    feedPathInput.setFocus();

    /** Ok Button is disabled on first open since input is empty */
    okButton.setEnabled(false);
    okButton.setFont(FontShop.dialogFont);
  }

  /**
   * @see org.eclipse.jface.dialogs.Dialog#createDialogArea(org.eclipse.swt.widgets.Composite)
   */
  protected Control createDialogArea(Composite parent) {

    /** Composite to hold all components */
    Composite composite = new Composite((Composite) super.createDialogArea(parent), SWT.NONE);
    composite.setLayout(LayoutShop.createGridLayout(3, 5, 20));
    composite.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));

    /** Title Image */
    setTitleImage(PaintShop.loadImage("/img/icons/file_big.gif"));

    /** Title Message */
    setMessage(dialogMessage, IMessageProvider.INFORMATION);

    /** Adress Label */
    Label adressLabel = new Label(composite, SWT.LEFT);
    adressLabel.setText(GUI.i18n.getTranslation("LABEL_ADDRESS"));
    adressLabel.setLayoutData(new GridData(GridData.HORIZONTAL_ALIGN_BEGINNING));
    adressLabel.setFont(FontShop.dialogFont);

    /** Feed path input field */
    feedPathInput = new Text(composite, SWT.SINGLE | SWT.BORDER);
    feedPathInput.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
    feedPathInput.setFont(FontShop.dialogFont);
    feedPathInput.addModifyListener(new ModifyListener() {
      public void modifyText(ModifyEvent e) {
        okButton.setEnabled(!feedPathInput.getText().trim().equals(""));
      }
    });

    /** Tweak Text Widget */
    WidgetShop.tweakTextWidget(feedPathInput);

    /** Open the file dialog to select a file to open */
    Button openFromFileButton = new Button(composite, SWT.PUSH);
    openFromFileButton.setText(GUI.i18n.getTranslation("BUTTON_FILE") + "...");
    openFromFileButton.setFont(FontShop.dialogFont);
    openFromFileButton.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(SelectionEvent e) {
        String file = FileShop.getFilePath(getShell(), null, null, SWT.OPEN, null, GUI.i18n.getTranslation("BUTTON_OPEN"));

        if (file != null)
          feedPathInput.setText(file);
      }
    });

    /** Holder for the separator to the OK and Cancel buttons */
    Composite sepHolder = new Composite(parent, SWT.NONE);
    sepHolder.setLayoutData(LayoutDataShop.createGridData(GridData.FILL_HORIZONTAL, 2));
    sepHolder.setLayout(LayoutShop.createGridLayout(1, 0, 0));

    /** Separator */
    Label separator = new Label(sepHolder, SWT.SEPARATOR | SWT.HORIZONTAL);
    separator.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));

    /** Set Mnemonics to Button */
    WidgetShop.initMnemonics(new Button[] { openFromFileButton });

    return composite;
  }

  /**
   * @see org.eclipse.jface.window.Window#getShellStyle()
   */
  protected int getShellStyle() {
    int style = SWT.TITLE | SWT.BORDER | SWT.APPLICATION_MODAL | getDefaultOrientation();

    /** Follow Apple's Human Interface Guidelines for Application Modal Dialogs */
    if (!GlobalSettings.isMac())
      style |= SWT.CLOSE;

    return style;
  }

  /**
   * @see org.eclipse.jface.dialogs.Dialog#initializeBounds()
   */
  protected void initializeBounds() {
    super.initializeBounds();
    Point bestSize = getShell().computeSize(convertHorizontalDLUsToPixels(dialogMinWidth), SWT.DEFAULT);
    Point location = getInitialLocation(bestSize);
    getShell().setBounds(location.x, location.y, bestSize.x, bestSize.y);
  }

  /**
   * Set the layout data of the button to a GridData with appropriate widths
   * This method was slightly modified so that it is not setting a heightHint.
   * 
   * @param button The button to layout
   */
  protected void setButtonLayoutData(Button button) {
    GridData data = new GridData(GridData.HORIZONTAL_ALIGN_FILL);
    int widthHint = convertHorizontalDLUsToPixels(IDialogConstants.BUTTON_WIDTH);
    data.widthHint = Math.max(widthHint, button.computeSize(SWT.DEFAULT, SWT.DEFAULT, true).x);
    button.setLayoutData(data);
  }
}