/*   **********************************************************************  **
 **   Copyright notice                                                       **
 **                                                                          **
 **   (c) 2003-2006 RSSOwl Development Team                                  **
 **   http://www.rssowl.org/                                                 **
 **                                                                          **
 **   All rights reserved                                                    **
 **                                                                          **
 **   This program and the accompanying materials are made available under   **
 **   the terms of the Eclipse Public License 1.0 which accompanies this     **
 **   distribution, and is available at:                                     **
 **   http://www.rssowl.org/legal/epl-v10.html                               **
 **                                                                          **
 **   A copy is found in the file epl-v10.html and important notices to the  **
 **   license from the team is found in the textfile LICENSE.txt distributed **
 **   in this package.                                                       **
 **                                                                          **
 **   This copyright notice MUST APPEAR in all copies of the file!           **
 **                                                                          **
 **   Contributors:                                                          **
 **     RSSOwl - initial API and implementation (bpasero@rssowl.org)         **
 **                                                                          **
 **  **********************************************************************  */

package net.sourceforge.rssowl.controller.popup;

import net.sourceforge.rssowl.controller.EventManager;
import net.sourceforge.rssowl.controller.GUI;
import net.sourceforge.rssowl.model.Category;
import net.sourceforge.rssowl.model.TreeItemData;
import net.sourceforge.rssowl.util.GlobalSettings;
import net.sourceforge.rssowl.util.shop.PaintShop;
import net.sourceforge.rssowl.util.shop.ProxyShop;

import org.eclipse.swt.SWT;
import org.eclipse.swt.events.MenuAdapter;
import org.eclipse.swt.events.MenuEvent;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.MenuItem;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.TreeItem;

/**
 * The category popup appears when right-clicking on any category in the tree.
 * 
 * @author <a href="mailto:bpasero@rssowl.org">Benjamin Pasero </a>
 * @version 1.2.3
 */
public class CategoryPopup {
  private Menu popUpMenu;
  private Shell shell;
  EventManager eventManager;
  GUI rssOwlGui;
  TreeItem selectedTreeItem;

  /**
   * Instantiate a new CategoryPopup
   * 
   * @param shell The shell
   * @param rssOwlGui The maincontroller
   * @param popUpMenu The menu to add this menu to
   * @param eventManager The event manager
   * @param selectedTreeItem The treeitem that gets this menu
   */
  public CategoryPopup(Shell shell, GUI rssOwlGui, Menu popUpMenu, EventManager eventManager, TreeItem selectedTreeItem) {
    this.shell = shell;
    this.rssOwlGui = rssOwlGui;
    this.popUpMenu = popUpMenu;
    this.eventManager = eventManager;
    this.selectedTreeItem = selectedTreeItem;

    /** Init the Popup */
    initCatMenu();
  }

  /** Init the popup menuStructure for a category */
  private void initCatMenu() {

    /** Aggregate selected category */
    MenuItem aggregateCategory = new MenuItem(popUpMenu, SWT.PUSH);
    aggregateCategory.setText(GUI.i18n.getTranslation("POP_AGGREGATE_FAV"));
    aggregateCategory.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(SelectionEvent e) {
        eventManager.actionAggregateCategory(false);
      }
    });

    /** Disable this menuitem if treeitem has no childs */
    if (selectedTreeItem.getItemCount() <= 0)
      aggregateCategory.setEnabled(false);

    /** Open selected categorie's direct Favorites */
    if (!GlobalSettings.displaySingleTab) {
      boolean enabled = false;

      MenuItem openCategory = new MenuItem(popUpMenu, SWT.PUSH);
      openCategory.setText(GUI.i18n.getTranslation("BUTTON_OPEN"));
      openCategory.addSelectionListener(new SelectionAdapter() {
        public void widgetSelected(SelectionEvent e) {
          eventManager.actionOpenSelectedCategory();
        }
      });

      /** Disable this menuitem if treeitem has no direct Favorites */
      TreeItem childs[] = selectedTreeItem.getItems();
      for (int i = 0; i < childs.length; i++) {
        TreeItem item = childs[i];
        if (item.getData() != null && ((TreeItemData) item.getData()).isFavorite()) {
          enabled = true;
          break;
        }
      }

      if (!enabled)
        openCategory.setEnabled(false);
    }

    /** Separator */
    new MenuItem(popUpMenu, SWT.SEPARATOR);

    /** Mark selected category read */
    MenuItem markRead = new MenuItem(popUpMenu, SWT.PUSH);
    markRead.setText(GUI.i18n.getTranslation("POP_MARK_CATEGORY_READ"));
    markRead.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(SelectionEvent e) {
        eventManager.actionMarkCategoryRead();
      }
    });

    /** Disable this menuitem if treeitem has no childs */
    if (selectedTreeItem.getItemCount() <= 0)
      markRead.setEnabled(false);

    /** Separator */
    new MenuItem(popUpMenu, SWT.SEPARATOR);

    /** Reload favorites of the selected category */
    MenuItem reload = new MenuItem(popUpMenu, SWT.PUSH);
    reload.setText(GUI.i18n.getTranslation("BUTTON_RELOAD_CAT"));
    if (!GlobalSettings.isMac())
      reload.setImage(PaintShop.iconReload);
    reload.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(SelectionEvent e) {
        eventManager.actionAggregateCategory(true);
      }
    });

    /** Disable this menuitem if treeitem has no childs */
    if (selectedTreeItem.getItemCount() <= 0)
      reload.setEnabled(false);

    /** Separator */
    new MenuItem(popUpMenu, SWT.SEPARATOR);

    /** Search all favorites of the selected category */
    MenuItem search = new MenuItem(popUpMenu, SWT.PUSH);
    search.setText(GUI.i18n.getTranslation("SEARCH_DIALOG_TITLE") + "...");
    if (!GlobalSettings.isMac())
      search.setImage(PaintShop.iconFind);
    search.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(SelectionEvent e) {
        eventManager.actionSearchInCategory();
      }
    });

    /** Disable this menuitem if treeitem has no childs */
    if (selectedTreeItem.getItemCount() <= 0)
      search.setEnabled(false);

    /** Separator */
    new MenuItem(popUpMenu, SWT.SEPARATOR);

    /** New Menu */
    MenuItem newCatOrFav = new MenuItem(popUpMenu, SWT.CASCADE);
    newCatOrFav.setText(GUI.i18n.getTranslation("POP_NEW"));
    if (!GlobalSettings.isMac())
      newCatOrFav.setImage(PaintShop.iconNew);

    Menu selectNewCatOrFav = new Menu(shell, SWT.DROP_DOWN);
    newCatOrFav.setMenu(selectNewCatOrFav);

    /** Create a new favorite */
    MenuItem cascadeItemNewFav = new MenuItem(selectNewCatOrFav, SWT.PUSH);
    cascadeItemNewFav.setText(GUI.i18n.getTranslation("LABEL_FAVORITE") + "...");
    if (!GlobalSettings.isMac())
      cascadeItemNewFav.setImage(PaintShop.iconAddToFavorites);
    cascadeItemNewFav.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(SelectionEvent e) {
        eventManager.actionNewFavorite();
      }
    });

    /** Separator */
    new MenuItem(selectNewCatOrFav, SWT.SEPARATOR);

    /** Create a new category */
    MenuItem cascadeItemNewCat = new MenuItem(selectNewCatOrFav, SWT.PUSH);
    cascadeItemNewCat.setText(GUI.i18n.getTranslation("LABEL_CATEGORY") + "...");
    if (!GlobalSettings.isMac())
      cascadeItemNewCat.setImage(PaintShop.iconAddCategory);
    cascadeItemNewCat.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(SelectionEvent e) {
        eventManager.actionNewCategory(false);
      }
    });

    /** Create a new sub category */
    MenuItem cascadeItemNewSubCat = new MenuItem(selectNewCatOrFav, SWT.PUSH);
    cascadeItemNewSubCat.setText(GUI.i18n.getTranslation("POP_SUB_CATEGORY") + "...");
    if (!GlobalSettings.isMac())
      cascadeItemNewSubCat.setImage(PaintShop.iconSubCategory);
    cascadeItemNewSubCat.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(SelectionEvent e) {
        eventManager.actionNewCategory(true);
      }
    });

    /** Rename selected category */
    MenuItem rename = new MenuItem(popUpMenu, SWT.PUSH);
    rename.setText(GUI.i18n.getTranslation("MENU_EDIT_RENAME"));
    rename.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(SelectionEvent e) {
        eventManager.actionRenameSelectedItem();
      }
    });

    /** Delete selected category */
    MenuItem delete = new MenuItem(popUpMenu, SWT.PUSH);
    delete.setText(GUI.i18n.getTranslation("MENU_EDIT_DELETE"));
    if (!GlobalSettings.isMac())
      delete.setImage(PaintShop.iconDelete);
    delete.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(SelectionEvent e) {
        eventManager.actionDeleteCategory();
      }
    });

    /** Separator */
    new MenuItem(popUpMenu, SWT.SEPARATOR);

    /** Import OPML files */
    MenuItem importMenu = new MenuItem(popUpMenu, SWT.CASCADE);
    importMenu.setText(GUI.i18n.getTranslation("POP_IMPORT"));
    if (!GlobalSettings.isMac())
      importMenu.setImage(PaintShop.iconImport);

    Menu selectImport = new Menu(shell, SWT.DROP_DOWN);
    importMenu.setMenu(selectImport);

    /** Import OPML file as category on same level */
    MenuItem cascadeItemImportOPML = new MenuItem(selectImport, SWT.PUSH);
    cascadeItemImportOPML.setText(GUI.i18n.getTranslation("POP_FROM_OPML") + "...");
    if (!GlobalSettings.isMac())
      cascadeItemImportOPML.setImage(PaintShop.iconImportOpml);
    cascadeItemImportOPML.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(SelectionEvent e) {
        eventManager.actionImportOPML();
      }
    });

    /** Import Synchronized Blogroll */
    MenuItem cascadeItemImportBlogroll = new MenuItem(selectImport, SWT.PUSH);
    cascadeItemImportBlogroll.setText(GUI.i18n.getTranslation("POP_IMPORT_BLOGROLL") + "...");
    if (!GlobalSettings.isMac())
      cascadeItemImportBlogroll.setImage(PaintShop.iconAddBlogroll);
    cascadeItemImportBlogroll.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(SelectionEvent e) {
        eventManager.actionImportBlogroll();
      }
    });

    /** Export in OPML or Blogroll */
    MenuItem export = new MenuItem(popUpMenu, SWT.CASCADE);
    export.setText(GUI.i18n.getTranslation("BUTTON_EXPORT"));
    if (!GlobalSettings.isMac())
      export.setImage(PaintShop.iconExport);

    /** Disable this menuitem if treeitem has no childs */
    if (selectedTreeItem.getItemCount() <= 0)
      export.setEnabled(false);

    Menu selectexport = new Menu(shell, SWT.DROP_DOWN);
    export.setMenu(selectexport);

    /** Export selected category into OPML file */
    MenuItem cascadeItemCreateOPML = new MenuItem(selectexport, SWT.PUSH);
    cascadeItemCreateOPML.setText(GUI.i18n.getTranslation("POP_EXPORT_OPML") + "...");
    if (!GlobalSettings.isMac())
      cascadeItemCreateOPML.setImage(PaintShop.iconExportOpml);
    cascadeItemCreateOPML.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(SelectionEvent e) {
        eventManager.actionCategoryToOPML();
      }
    });

    /** In case Proxy is activated */
    if (ProxyShop.isUseProxy()) {

      /** Separator */
      new MenuItem(popUpMenu, SWT.SEPARATOR);

      /** Enable / Disable use of proxy for the favorites in this category */
      MenuItem proxy = new MenuItem(popUpMenu, SWT.CASCADE);
      proxy.setText(GUI.i18n.getTranslation("MENU_PROXY"));

      Menu selectProxy = new Menu(shell, SWT.DROP_DOWN);
      proxy.setMenu(selectProxy);

      final MenuItem cascadeItemProxy = new MenuItem(selectProxy, SWT.CHECK);
      cascadeItemProxy.setText(GUI.i18n.getTranslation("POP_USEPROXY"));
      cascadeItemProxy.addSelectionListener(new SelectionAdapter() {
        public void widgetSelected(SelectionEvent e) {
          if (cascadeItemProxy.isEnabled())
            eventManager.actionSetProxyOnTreeItem(cascadeItemProxy.getSelection());
        }
      });

      /** Dynamically react on the Proxy Settings */
      selectProxy.addMenuListener(new MenuAdapter() {
        public void menuShown(MenuEvent e) {
          Category selectedCat = rssOwlGui.getRSSOwlFavoritesTree().getSelectedCat();

          if (selectedCat != null)
            cascadeItemProxy.setSelection(selectedCat.isUseProxy());
        }
      });
    }

    /** Separator */
    new MenuItem(popUpMenu, SWT.SEPARATOR);

    /** Edit selected category */
    MenuItem properties = new MenuItem(popUpMenu, SWT.PUSH);
    properties.setText(GUI.i18n.getTranslation("POP_PROPERTIES"));
    properties.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(SelectionEvent e) {
        eventManager.actionEditCategory();
      }
    });
  }
}