/*   **********************************************************************  **
 **   Copyright notice                                                       **
 **                                                                          **
 **   (c) 2003-2006 RSSOwl Development Team                                  **
 **   http://www.rssowl.org/                                                 **
 **                                                                          **
 **   All rights reserved                                                    **
 **                                                                          **
 **   This program and the accompanying materials are made available under   **
 **   the terms of the Eclipse Public License 1.0 which accompanies this     **
 **   distribution, and is available at:                                     **
 **   http://www.rssowl.org/legal/epl-v10.html                               **
 **                                                                          **
 **   A copy is found in the file epl-v10.html and important notices to the  **
 **   license from the team is found in the textfile LICENSE.txt distributed **
 **   in this package.                                                       **
 **                                                                          **
 **   This copyright notice MUST APPEAR in all copies of the file!           **
 **                                                                          **
 **   Contributors:                                                          **
 **     RSSOwl - initial API and implementation (bpasero@rssowl.org)         **
 **                                                                          **
 **  **********************************************************************  */

package net.sourceforge.rssowl.controller.properties;

import net.sourceforge.rssowl.controller.GUI;
import net.sourceforge.rssowl.util.GlobalSettings;
import net.sourceforge.rssowl.util.shop.FontShop;
import net.sourceforge.rssowl.util.shop.LayoutDataShop;
import net.sourceforge.rssowl.util.shop.LayoutShop;
import net.sourceforge.rssowl.util.shop.WidgetShop;

import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.CLabel;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Group;

/**
 * Class displays a PropertyPage to set System-Tray-releated properties in
 * RSSOwl.
 * 
 * @author <a href="mailto:bpasero@rssowl.org">Benjamin Pasero </a>
 * @version 1.2.3
 */
public class SystemTrayProperties extends PropertyPage {
  private Button trayOnExitCheck;
  private Button trayOnStartupCheck;
  Button trayOwlCheck;
  Button trayPopupAnimate;
  Button trayPopupCheck;
  Button trayPopupFade;

  /**
   * Instatiate a new SystemTrayProperties PropertyPage
   * 
   * @param parent Composite's parent
   * @param rssOwlGui The MainController
   */
  public SystemTrayProperties(Composite parent, GUI rssOwlGui) {
    super(parent, rssOwlGui);
    setApplyButtonState(GlobalSettings.useSystemTray());
    setRestoreButtonState(GlobalSettings.useSystemTray());
  }

  /**
   * @see net.sourceforge.rssowl.controller.properties.PropertyPage#applyButtonPressed()
   */
  public void applyButtonPressed() {

    /** Place RSSOwl in Tray when minimized */
    if (GlobalSettings.useSystemTray() && GlobalSettings.showSystrayIcon != trayOwlCheck.getSelection()) {
      GlobalSettings.showSystrayIcon = trayOwlCheck.getSelection();

      /** Update in GUI */
      rssOwlGui.enableSystrayIcon(GlobalSettings.showSystrayIcon);
    }

    /** Other Tray GlobalSettings */
    if (GlobalSettings.useSystemTray()) {
      GlobalSettings.trayOnStartup = trayOnStartupCheck.getSelection();
      GlobalSettings.trayOnExit = trayOnExitCheck.getSelection();
      GlobalSettings.showTrayPopup = trayPopupCheck.getSelection();
      GlobalSettings.autoCloseNewsPopup = trayPopupFade.getSelection();
      GlobalSettings.animateNewsPopup = trayPopupAnimate.getSelection();
    }

    updatePropertiesChangeManager();
  }

  /**
   * @see net.sourceforge.rssowl.controller.properties.PropertyPage#updatePropertiesChangeManager()
   */
  public void updatePropertiesChangeManager() {
    if (GlobalSettings.useSystemTray()) {
      propertyChangeManager.setShowSystrayIcon(trayOwlCheck.getSelection());
      propertyChangeManager.setTrayOnStartup(trayOnStartupCheck.getSelection());
      propertyChangeManager.setTrayOnExit(trayOnExitCheck.getSelection());
      propertyChangeManager.setShowTrayPopup(trayPopupCheck.getSelection());
      propertyChangeManager.setAutoCloseNewsPopup(trayPopupFade.getSelection());
      propertyChangeManager.setAnimateNewsPopup(trayPopupAnimate.getSelection());
    }
  }

  /**
   * @see net.sourceforge.rssowl.controller.properties.PropertyPage#initComponents()
   */
  protected void initComponents() {

    /** Tray related settings */
    Group trayGroup = new Group(composite, SWT.NONE);
    trayGroup.setLayoutData(LayoutDataShop.createGridData(GridData.VERTICAL_ALIGN_BEGINNING | GridData.FILL_HORIZONTAL, 2));
    trayGroup.setText(GUI.i18n.getTranslation("GROUP_TRAY"));
    trayGroup.setLayout(new GridLayout(2, false));
    trayGroup.setFont(FontShop.dialogFont);

    /** Enable Tray on minimize */
    trayOwlCheck = new Button(trayGroup, SWT.CHECK);
    trayOwlCheck.setLayoutData(LayoutDataShop.createGridData(GridData.FILL_HORIZONTAL, 2));
    trayOwlCheck.setSelection(propertyChangeManager.isShowSystrayIcon());
    trayOwlCheck.setText(GUI.i18n.getTranslation("MENU_SYSTRAY"));
    trayOwlCheck.setFont(dialogFont);
    trayOwlCheck.addSelectionListener(new SelectionAdapter() {

      /** Disable tray controls if tray is disabled */
      public void widgetSelected(SelectionEvent e) {
        setTrayControlsEnabled(trayOwlCheck.getSelection());
      }
    });

    /** Place RSSOwl in tray on startup */
    trayOnStartupCheck = new Button(trayGroup, SWT.CHECK);
    trayOnStartupCheck.setLayoutData(LayoutDataShop.createGridData(GridData.FILL_HORIZONTAL, 2));
    trayOnStartupCheck.setSelection(propertyChangeManager.isTrayOnStartup());
    trayOnStartupCheck.setText(GUI.i18n.getTranslation("BUTTON_TRAY_STARTUP"));
    trayOnStartupCheck.setFont(dialogFont);

    /** Place RSSOwl in tray on exit */
    trayOnExitCheck = new Button(trayGroup, SWT.CHECK);
    trayOnExitCheck.setLayoutData(LayoutDataShop.createGridData(GridData.FILL_HORIZONTAL, 2));
    trayOnExitCheck.setSelection(propertyChangeManager.isTrayOnExit());
    trayOnExitCheck.setText(GUI.i18n.getTranslation("BUTTON_TRAY_EXIT"));
    trayOnExitCheck.setFont(dialogFont);

    /** News popup related settings */
    Group newsPopupGroup = new Group(composite, SWT.NONE);
    newsPopupGroup.setLayoutData(LayoutDataShop.createGridData(GridData.VERTICAL_ALIGN_BEGINNING | GridData.FILL_HORIZONTAL, 2));
    newsPopupGroup.setText(GUI.i18n.getTranslation("GROUP_NEWS_POPUP"));
    newsPopupGroup.setLayout(new GridLayout(2, false));
    newsPopupGroup.setFont(FontShop.dialogFont);

    /** Show tray popup when unread news are available */
    trayPopupCheck = new Button(newsPopupGroup, SWT.CHECK);
    trayPopupCheck.setLayoutData(LayoutDataShop.createGridData(GridData.FILL_HORIZONTAL, 2));
    trayPopupCheck.setSelection(propertyChangeManager.isShowTrayPopup());
    trayPopupCheck.setText(GUI.i18n.getTranslation("BUTTON_TRAY_POPUP"));
    trayPopupCheck.setFont(dialogFont);
    trayPopupCheck.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(SelectionEvent e) {
        trayPopupFade.setEnabled(trayPopupCheck.getSelection());
        trayPopupAnimate.setEnabled(trayPopupCheck.getSelection());
      }
    });

    /** Animate Popup into View */
    trayPopupAnimate = new Button(newsPopupGroup, SWT.CHECK);
    trayPopupAnimate.setLayoutData(LayoutDataShop.createGridData(GridData.FILL_HORIZONTAL, 2));
    trayPopupAnimate.setSelection(propertyChangeManager.isAnimateNewsPopup());
    trayPopupAnimate.setText(GUI.i18n.getTranslation("BUTTON_ANIMATE_POPUP"));
    trayPopupAnimate.setFont(dialogFont);

    /** Close tray popup automatically after some time */
    trayPopupFade = new Button(newsPopupGroup, SWT.CHECK);
    trayPopupFade.setLayoutData(LayoutDataShop.createGridData(GridData.FILL_HORIZONTAL, 2));
    trayPopupFade.setSelection(propertyChangeManager.isAutoCloseNewsPopup());
    trayPopupFade.setText(GUI.i18n.getTranslation("BUTTON_AUTOCLOSE_POPUP"));
    trayPopupFade.setFont(dialogFont);

    /** Error label in case System Tray is not yet supported */
    if (!GlobalSettings.useSystemTray()) {
      errorMessageLabel = new CLabel(composite, SWT.NONE);
      errorMessageLabel.setFont(dialogFont);
      errorMessageLabel.setLayoutData(LayoutDataShop.createGridData(GridData.HORIZONTAL_ALIGN_BEGINNING, 2));
      setErrorMessage(GUI.i18n.getTranslation("LABEL_NOT_UNSUPPORTED"));
    }

    /** Fill with some spacer */
    LayoutShop.setDialogSpacer(composite, 2, 4);

    /** Set state of tray controls */
    if (GlobalSettings.useSystemTray()) {
      setTrayControlsEnabled(trayOwlCheck.getSelection());
    }

    /** System Tray not supported on OS, disable */
    else {
      trayOwlCheck.setSelection(false);
      trayOnStartupCheck.setSelection(false);
      trayOnExitCheck.setSelection(false);
      trayPopupCheck.setSelection(false);
      trayPopupFade.setSelection(false);
      trayPopupAnimate.setSelection(false);
      trayOwlCheck.setEnabled(false);
      setTrayControlsEnabled(false);
    }

    /** Set Mnemonics to Buttons */
    WidgetShop.initMnemonics(new Button[] { trayOwlCheck, trayOnStartupCheck, trayOnExitCheck, trayPopupCheck, trayPopupAnimate, trayPopupFade });
  }

  /**
   * @see net.sourceforge.rssowl.controller.properties.PropertyPage#restoreButtonPressed()
   */
  protected void restoreButtonPressed() {
    if (GlobalSettings.useSystemTray()) {
      trayOwlCheck.setSelection(false);
      trayOnStartupCheck.setSelection(false);
      trayOnExitCheck.setSelection(false);
      trayPopupCheck.setSelection(true);
      trayPopupFade.setSelection(true);
      trayPopupFade.setEnabled(true);
      trayPopupAnimate.setSelection(true);

      /** Set state of tray controls */
      setTrayControlsEnabled(trayOwlCheck.getSelection());
    }
  }

  /**
   * Enable / Disable tray controls
   * 
   * @param enabled TRUE if enabled
   */
  void setTrayControlsEnabled(boolean enabled) {
    trayOnStartupCheck.setEnabled(enabled);
    trayOnExitCheck.setEnabled(enabled);
    trayPopupCheck.setEnabled(enabled);
    trayPopupAnimate.setEnabled(trayPopupCheck.getSelection() && enabled);
    trayPopupFade.setEnabled(trayPopupCheck.getSelection() && enabled);
  }
}