/*   **********************************************************************  **
 **   Copyright notice                                                       **
 **                                                                          **
 **   (c) 2003-2006 RSSOwl Development Team                                  **
 **   http://www.rssowl.org/                                                 **
 **                                                                          **
 **   All rights reserved                                                    **
 **                                                                          **
 **   This program and the accompanying materials are made available under   **
 **   the terms of the Eclipse Public License 1.0 which accompanies this     **
 **   distribution, and is available at:                                     **
 **   http://www.rssowl.org/legal/epl-v10.html                               **
 **                                                                          **
 **   A copy is found in the file epl-v10.html and important notices to the  **
 **   license from the team is found in the textfile LICENSE.txt distributed **
 **   in this package.                                                       **
 **                                                                          **
 **   This copyright notice MUST APPEAR in all copies of the file!           **
 **                                                                          **
 **   Contributors:                                                          **
 **     RSSOwl - initial API and implementation (bpasero@rssowl.org)         **
 **                                                                          **
 **  **********************************************************************  */

package net.sourceforge.rssowl.controller.statusline;

import net.sourceforge.rssowl.controller.GUI;
import net.sourceforge.rssowl.util.shop.PaintShop;

import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;

/**
 * This Thread will animate the status line.
 * 
 * @author <a href="mailto:bpasero@rssowl.org">Benjamin Pasero </a>
 * @version 1.2.3
 */
public class StatusLineAnimator extends Thread {

  /** Interval to update the progress bar */
  private static final int PROGRESS_UPDATE_INTERVAL = 500;

  private Display display;
  int progress;
  GUI rssOwlGui;
  Label statusProgressBar;

  /**
   * Instantiate a new StatusLineAnimator
   * 
   * @param display The display
   * @param rssOwlGui The main controller
   * @param statusProgressBar The widget to set the progress image
   * @param initProgress The initial progress value (0-100)
   */
  public StatusLineAnimator(Display display, GUI rssOwlGui, Label statusProgressBar, int initProgress) {
    this.display = display;
    this.rssOwlGui = rssOwlGui;
    this.statusProgressBar = statusProgressBar;
    this.progress = initProgress;
    setName("Status Line Animator Thread");
    setDaemon(true);
  }

  /**
   * @see java.lang.Thread#run()
   */
  public void run() {

    /** Status Animator Thread loop */
    while (!isInterrupted()) {

      /** Break the threadloop if device is disposed */
      if (!GUI.isAlive() || statusProgressBar.isDisposed())
        break;

      /** Update the progress bar if still running */
      if (rssOwlGui.isBusyLoading())
        updateProgressBar(progress);

      /** Sleep some time before updating the progress bar */
      try {
        sleep(PROGRESS_UPDATE_INTERVAL);
      } catch (InterruptedException e) {
        break;
      }

      /** Increment count value for the progress bar */
      progress += (progress == 100) ? -80 : 20;
    }
  }

  /**
   * Update the progress bar value
   * 
   * @param value The int value
   */
  void updateProgressBar(final int value) {

    /** Only perform this Runnable if RSSOwl was not closed */
    if (GUI.isAlive() && !isInterrupted()) {
      display.asyncExec(new Runnable() {
        public void run() {

          /** Status Bar must not be disposed, and RSSOwl still loading */
          if (!statusProgressBar.isDisposed() && rssOwlGui.isBusyLoading()) {
            statusProgressBar.setImage(PaintShop.getProgressIcon(value));

            /** Parent of Progress Bar must not be disposed */
            if (!statusProgressBar.getParent().isDisposed())
              statusProgressBar.getParent().layout();
          }
        }
      });
    }
  }
}