/*   **********************************************************************  **
 **   Copyright notice                                                       **
 **                                                                          **
 **   (c) 2003-2006 RSSOwl Development Team                                  **
 **   http://www.rssowl.org/                                                 **
 **                                                                          **
 **   All rights reserved                                                    **
 **                                                                          **
 **   This program and the accompanying materials are made available under   **
 **   the terms of the Eclipse Public License 1.0 which accompanies this     **
 **   distribution, and is available at:                                     **
 **   http://www.rssowl.org/legal/epl-v10.html                               **
 **                                                                          **
 **   A copy is found in the file epl-v10.html and important notices to the  **
 **   license from the team is found in the textfile LICENSE.txt distributed **
 **   in this package.                                                       **
 **                                                                          **
 **   This copyright notice MUST APPEAR in all copies of the file!           **
 **                                                                          **
 **   Contributors:                                                          **
 **     RSSOwl - initial API and implementation (bpasero@rssowl.org)         **
 **                                                                          **
 **  **********************************************************************  */

package net.sourceforge.rssowl.model;

import net.sourceforge.rssowl.controller.NewsTable;
import net.sourceforge.rssowl.controller.panel.BrowserPanel;

import org.eclipse.swt.widgets.Table;

/**
 * Model class to store data in a TabItem. The setData() Method of any TabItem
 * in RSSOwl is used with a TabItemData Object.
 * 
 * @author <a href="mailto:bpasero@rssowl.org">Benjamin Pasero </a>
 * @version 1.2.3
 */
public class TabItemData {

  /** Tab is displaying a browser */
  private static final int TYPE_BROWSER = 0;

  /** Tab is displaying a warning */
  private static final int TYPE_ERROR = 3;

  /** Tab is displaying a feed */
  private static final int TYPE_FEED = 2;

  /** Tab is displaying a message */
  private static final int TYPE_MESSAGE = 1;

  private Category aggregatedCategory;
  private Channel channel;
  private Table newsHeaderTableWidget;
  private NewsTable newsTable;
  private BrowserPanel rssOwlBrowserPanel;
  private boolean showsUnreadNews;
  private String title;
  private int type;
  private String url;

  /**
   * Private constructor which is used by the factory methods.
   * 
   * @param url The URL of the feed
   * @param title The title of the feed
   * @param aggregatedCategory If this is an aggregation, the aggregated
   * category
   * @param newsHeaderTableWidget The table holding the newsheader of the feed
   * @param channel The displayed channel if this tab is displaying a feed
   * @param rssOwlBrowserPanel The browser panel if a browser is displayed
   * @param type The type of the tab
   */
  private TabItemData(String url, String title, Category aggregatedCategory, NewsTable newsTable, Table newsHeaderTableWidget, Channel channel, BrowserPanel rssOwlBrowserPanel, int type) {
    this.url = url;
    this.title = title;
    this.aggregatedCategory = aggregatedCategory;
    this.channel = channel;
    this.newsTable = newsTable;
    this.newsHeaderTableWidget = newsHeaderTableWidget;
    this.type = type;
    this.rssOwlBrowserPanel = rssOwlBrowserPanel;
    showsUnreadNews = (channel != null) ? channel.containsUnreadNews() : false;
  }

  /**
   * Create a new TabItemData with the type: Browser
   * 
   * @param rssOwlBrowserPanel The browserpanel used by the browser
   * @return TabItemData An instance of TabItemData
   */
  public static TabItemData createBrowserData(BrowserPanel rssOwlBrowserPanel) {
    return new TabItemData(null, null, null, null, null, null, rssOwlBrowserPanel, TYPE_BROWSER);
  }

  /**
   * Create a new TabItemData with the type: Error
   * 
   * @param urlOrTitle URL or Title of the Error
   * @return TabItemData An instance of TabItemData
   */
  public static TabItemData createErrorData(String urlOrTitle) {
    return new TabItemData(urlOrTitle, null, null, null, null, null, null, TYPE_ERROR);
  }

  /**
   * Create a new TabItemData with the type: Feed
   * 
   * @param url The URL of the feed
   * @param title The title of the feed
   * @param channel The displayed channel if this tab is displaying a feed
   * @param newsTable The NewsTable proxy
   * @param newsHeaderTable The table holding the newsheader of the feed
   * @return TabItemData An instance of TabItemData
   */
  public static TabItemData createFeedData(String url, String title, Channel channel, NewsTable newsTable, Table newsHeaderTable) {
    return new TabItemData(url, title, channel.getAggregatedCategory(), newsTable, newsHeaderTable, channel, null, TYPE_FEED);
  }

  /**
   * Create a new TabItemData with the type: Message
   * 
   * @param urlOrTitle URL or Title of the Message / Error
   * @return TabItemData An instance of TabItemData
   */
  public static TabItemData createMessageData(String urlOrTitle) {
    return new TabItemData(urlOrTitle, null, null, null, null, null, null, TYPE_MESSAGE);
  }

  /**
   * Dispose this TabItemData Object
   */
  public void dispose() {
    channel = null;
    newsTable = null;
    newsHeaderTableWidget = null;
    if (rssOwlBrowserPanel != null)
      rssOwlBrowserPanel.dispose();
    rssOwlBrowserPanel = null;
  }

  /**
   * @return Category If this is an aggregation, the aggregated category
   */
  public Category getAggregatedCategory() {
    return aggregatedCategory;
  }

  /**
   * @return Returns the channel that is displayed
   */
  public Channel getChannel() {
    return channel;
  }

  /**
   * @return Returns the newsHeaderTable.
   */
  public Table getNewsHeaderTable() {
    return newsHeaderTableWidget;
  }

  /**
   * Gets the NewsTable Proxy.
   * 
   * @return NewsTable The NewsTable Proxy.
   */
  public NewsTable getNewsTable() {
    return newsTable;
  }

  /**
   * @return Returns the rssOwlBrowserPanel.
   */
  public BrowserPanel getRSSOwlBrowserPanel() {
    return rssOwlBrowserPanel;
  }

  /**
   * @return Returns the title.
   */
  public String getTitle() {
    return title;
  }

  /**
   * @return Returns the url.
   */
  public String getUrl() {
    return url;
  }

  /**
   * @return boolean TRUE if the Feed is an aggregated category
   */
  public boolean isAggregatedCat() {
    return aggregatedCategory != null;
  }

  /**
   * @return boolean TRUE if the Type is Browser
   */
  public boolean isBrowser() {
    return (type == TYPE_BROWSER);
  }

  /**
   * @return boolean TRUE if the Type is Message
   */
  public boolean isError() {
    return (type == TYPE_ERROR);
  }

  /**
   * @return boolean TRUE if the Type is Feed
   */
  public boolean isFeed() {
    return (type == TYPE_FEED);
  }

  /**
   * @return boolean TRUE if the Type is Message
   */
  public boolean isMessage() {
    return (type == TYPE_MESSAGE);
  }

  /**
   * Status flag to remember the TabItems state
   * 
   * @return boolean Returns the showsUnreadNews.
   */
  public boolean isShowsUnreadNews() {
    return showsUnreadNews;
  }

  /**
   * Set the status flag to remember the TabItems state
   * 
   * @param showsUnreadNews The showsUnreadNews to set.
   */
  public void setShowsUnreadNews(boolean showsUnreadNews) {
    this.showsUnreadNews = showsUnreadNews;
  }

  /**
   * Check if the given Category is aggregated in this TabItem
   * 
   * @param category The category to check if shown
   * @return boolean TRUE in case this tabitem is showing an aggregation of the
   * given category
   */
  public boolean showsCategory(Category category) {
    if (aggregatedCategory == null || category == null)
      return false;
    return aggregatedCategory.toCatPath().equals(category.toCatPath());
  }
}