/*   **********************************************************************  **
 **   Copyright notice                                                       **
 **                                                                          **
 **   (c) 2003-2006 RSSOwl Development Team                                  **
 **   http://www.rssowl.org/                                                 **
 **                                                                          **
 **   All rights reserved                                                    **
 **                                                                          **
 **   This program and the accompanying materials are made available under   **
 **   the terms of the Eclipse Public License 1.0 which accompanies this     **
 **   distribution, and is available at:                                     **
 **   http://www.rssowl.org/legal/epl-v10.html                               **
 **                                                                          **
 **   A copy is found in the file epl-v10.html and important notices to the  **
 **   license from the team is found in the textfile LICENSE.txt distributed **
 **   in this package.                                                       **
 **                                                                          **
 **   This copyright notice MUST APPEAR in all copies of the file!           **
 **                                                                          **
 **   Contributors:                                                          **
 **     RSSOwl - initial API and implementation (bpasero@rssowl.org)         **
 **                                                                          **
 **  **********************************************************************  */

package net.sourceforge.rssowl.model;

/**
 * Model class to store data in a TreeItem. The setData() Method of any TreeItem
 * in RSSOwl is used with a TreeItemData Object.
 * 
 * @author <a href="mailto:bpasero@rssowl.org">Benjamin Pasero </a>
 * @version 1.2.3
 */
public class TreeItemData {

  /** TreeItem is a blogroll */
  private static final int TYPE_BLOGROLL = 2;

  /** TreeItem is a blogroll favorite */
  private static final int TYPE_BLOGROLL_FAVORITE = 3;

  /** TreeItem is a category */
  private static final int TYPE_CATEGORY = 1;

  /** TreeItem is a favorite */
  private static final int TYPE_FAVORITE = 0;
  
  private Favorite favorite;
  private String name;
  private boolean statusUnread;
  private int type;

  /**
   * Private constructor that is used by the factory methods
   * 
   * @param type The type of the TreeItem
   * @param name Name of the given TreeItem
   * @param statusUnread TRUE indicates that Favorite contains unread news
   */
  private TreeItemData(int type, String name, boolean statusUnread) {
    this.type = type;
    this.name = name;
    this.statusUnread = statusUnread;
  }

  /**
   * Create a new TreeItem with the type: Blogroll
   * 
   * @param name Name of the given TreeItem
   * @param statusUnread TRUE indicates that Favorite contains unread news
   * @return TreeItemData An instance of TreeItemData
   */
  public static TreeItemData createBlogroll(String name, boolean statusUnread) {
    return new TreeItemData(TYPE_BLOGROLL, name, statusUnread);
  }

  /**
   * Create a new TreeItem with the type: Blogroll Favorite
   * 
   * @param name Name of the given TreeItem
   * @param statusUnread TRUE indicates that Favorite contains unread news
   * @return TreeItemData An instance of TreeItemData
   */
  public static TreeItemData createBlogrollFavorite(String name, boolean statusUnread) {
    return new TreeItemData(TYPE_BLOGROLL_FAVORITE, name, statusUnread);
  }

  /**
   * Create a new TreeItem with the type: Category
   * 
   * @param name Name of the given TreeItem
   * @param statusUnread TRUE indicates that Favorite contains unread news
   * @return TreeItemData An instance of TreeItemData
   */
  public static TreeItemData createCategory(String name, boolean statusUnread) {
    return new TreeItemData(TYPE_CATEGORY, name, statusUnread);
  }

  /**
   * Create a new TreeItem with the type: Favorite
   * 
   * @param name Name of the given TreeItem
   * @param statusUnread TRUE indicates that Favorite contains unread news
   * @return TreeItemData An instance of TreeItemData
   */
  public static TreeItemData createFavorite(String name, boolean statusUnread) {
    return new TreeItemData(TYPE_FAVORITE, name, statusUnread);
  }

  /**
   * If this Data Object is from a Favorite, store a reference to the connected
   * model object.
   * 
   * @return Favorite The Favorite Model Object.
   */
  public Favorite getFavorite() {
    return favorite;
  }

  /**
   * Get the name for this TreeItem
   * 
   * @return Returns the name.
   */
  public String getName() {
    return name;
  }

  /**
   * @return boolean TRUE if the TreeItem is a blogroll
   */
  public boolean isBlogroll() {
    return (type == TYPE_BLOGROLL);
  }

  /**
   * @return boolean TRUE if the TreeItem is a blogroll favorite
   */
  public boolean isBlogrollFavorite() {
    return (type == TYPE_BLOGROLL_FAVORITE);
  }

  /**
   * @return boolean TRUE if the TreeItem is a category
   */
  public boolean isCategory() {
    return (type == TYPE_CATEGORY);
  }

  /**
   * @return boolean TRUE if the TreeItem is a favorite
   */
  public boolean isFavorite() {
    return (type == TYPE_FAVORITE);
  }

  /**
   * Get the status of this TreeItem. It shows either read or unread
   * 
   * @return Returns the statusUnread.
   */
  public boolean isStatusUnread() {
    return statusUnread;
  }

  /**
   * If this Data Object is from a Favorite, store a reference to the connected
   * model object.
   * 
   * @param favorite The Favorite Model Object to set.
   */
  public void setFavorite(Favorite favorite) {
    this.favorite = favorite;
  }

  /**
   * Set the status of this TreeItem. Either set read or unread
   * 
   * @param statusUnread The statusUnread to set.
   */
  public void setStatusUnread(boolean statusUnread) {
    this.statusUnread = statusUnread;
  }
}