/*   **********************************************************************  **
 **   Copyright notice                                                       **
 **                                                                          **
 **   (c) 2003-2006 RSSOwl Development Team                                  **
 **   http://www.rssowl.org/                                                 **
 **                                                                          **
 **   All rights reserved                                                    **
 **                                                                          **
 **   This program and the accompanying materials are made available under   **
 **   the terms of the Eclipse Public License 1.0 which accompanies this     **
 **   distribution, and is available at:                                     **
 **   http://www.rssowl.org/legal/epl-v10.html                               **
 **                                                                          **
 **   A copy is found in the file epl-v10.html and important notices to the  **
 **   license from the team is found in the textfile LICENSE.txt distributed **
 **   in this package.                                                       **
 **                                                                          **
 **   This copyright notice MUST APPEAR in all copies of the file!           **
 **                                                                          **
 **   Contributors:                                                          **
 **     RSSOwl - initial API and implementation (bpasero@rssowl.org)         **
 **                                                                          **
 **  **********************************************************************  */

package net.sourceforge.rssowl.util.archive;

import net.sourceforge.rssowl.controller.GUI;
import net.sourceforge.rssowl.util.GlobalSettings;

import java.io.File;
import java.io.IOException;
import java.util.Enumeration;
import java.util.Hashtable;

/**
 * The index of the archive contains of a simple Hashtable that holds keys and
 * values: - Key: Unique feedurl - Value: Unique filename Any feedurl from the
 * index points to the file that stores the archive item of the feedurl.
 * 
 * @author <a href="mailto:bpasero@rssowl.org">Benjamin Pasero </a>
 * @version 1.2.3
 */
public class ArchiveIndex {
  private ArchiveManager archiveManager;
  private boolean dirty;
  private Hashtable index;

  /**
   * Instantiate a new ArchiveIndex
   * 
   * @param archiveManager The archiv manager
   */
  public ArchiveIndex(ArchiveManager archiveManager) {
    this.archiveManager = archiveManager;
    index = new Hashtable();

    /** Mark as in need of being saved on flush */
    dirty = true;
  }

  /**
   * Add a new item to the index
   * 
   * @param key The feedurl
   * @param value The filename of the archive item
   */
  public void addIndex(String key, String value) {
    index.put(key, value);

    /** Mark as in need of being saved on flush */
    dirty = true;
  }

  /**
   * @return Hashtable The actual index Hashtable.
   */
  public Hashtable getIndex() {
    return index;
  }

  /**
   * Get the value for the given key
   * 
   * @param key Any existing feedurl of the index
   * @return String The filename of the archive item
   */
  public String getValue(String key) {
    return (String) index.get(key);
  }

  /**
   * Check wether this Index has changed since it has been loaded from the File.
   * 
   * @return boolean TRUE if this Index has changed since it has been loaded and
   * requires to be saved to the File again, FALSE otherwise.
   */
  public boolean isDirty() {
    return dirty;
  }

  /**
   * Remove an item from the index
   * 
   * @param key The key identifier for the item
   */
  public void removeIndex(Object key) {
    index.remove(key);

    /** Mark as in need of being saved on flush */
    dirty = true;
  }

  /**
   * Set wether this Index has changed since it has been loaded from the File
   * and needs to be saved to the File again on shutdown.
   * 
   * @param dirty Set to TRUE if this Index has changed since it has been loaded
   * and requires to be saved to the File again, FALSE otherwise.
   */
  public void setDirty(boolean dirty) {
    this.dirty = dirty;
  }

  /**
   * Synchronise the filename for the archiv item with the index. If the archiv
   * item is new to the index, create a new unique file and tell the index.
   * 
   * @param archiveItem An archive item
   */
  public void syncFileName(ArchiveItem archiveItem) {

    /** ArchivItem is new to the Index */
    if (!index.containsKey(archiveItem.getFeedurl())) {

      /** Create a new unique file */
      File file;
      try {
        file = archiveManager.createUniqueArchiveFile();

        /** Tell Index about filename */
        addIndex(archiveItem.getFeedurl(), file.getName());

        /** Tell Archiv Item about filename */
        archiveItem.setFile(file);
      } catch (IOException e) {
        GUI.logger.log("syncFileName()", e);
      }
    }

    /** ArchivItem is known to the Index */
    else {

      /** Tell Archiv Item about filename */
      archiveItem.setFile(new File(GlobalSettings.ARCHIVE_DIR + GlobalSettings.PATH_SEPARATOR + getValue(archiveItem.getFeedurl())));
    }
  }

  /**
   * @see java.lang.Object#toString()
   */
  public String toString() {
    StringBuffer strBuf = new StringBuffer("");
    Enumeration keys = index.keys();
    while (keys.hasMoreElements()) {
      String feedurl = (String) keys.nextElement();
      String fileName = (String) index.get(feedurl);
      strBuf.append("FeedURL: " + feedurl + "   FileName: " + fileName);
    }
    return new String(strBuf);
  }
}