/*   **********************************************************************  **
 **   Copyright notice                                                       **
 **                                                                          **
 **   (c) 2003-2006 RSSOwl Development Team                                  **
 **   http://www.rssowl.org/                                                 **
 **                                                                          **
 **   All rights reserved                                                    **
 **                                                                          **
 **   This program and the accompanying materials are made available under   **
 **   the terms of the Eclipse Public License 1.0 which accompanies this     **
 **   distribution, and is available at:                                     **
 **   http://www.rssowl.org/legal/epl-v10.html                               **
 **                                                                          **
 **   A copy is found in the file epl-v10.html and important notices to the  **
 **   license from the team is found in the textfile LICENSE.txt distributed **
 **   in this package.                                                       **
 **                                                                          **
 **   This copyright notice MUST APPEAR in all copies of the file!           **
 **                                                                          **
 **   Contributors:                                                          **
 **     RSSOwl - initial API and implementation (bpasero@rssowl.org)         **
 **                                                                          **
 **  **********************************************************************  */

package net.sourceforge.rssowl.util.document;

import net.sourceforge.rssowl.controller.GUI;

import com.lowagie.text.DocumentException;
import com.lowagie.text.Font;
import com.lowagie.text.FontFactory;
import com.lowagie.text.pdf.BaseFont;

import java.awt.Color;
import java.io.IOException;
import java.util.Locale;

/**
 * Class that offers fonts for the PDF / RTF / HTML export of a Newsfeed
 * 
 * @author <a href="mailto:bpasero@rssowl.org">Benjamin Pasero </a>
 * @version 1.2.3
 */
public class DocumentConfig {

  /** Default font for non Asia locale */
  private static final String DEFAULT_FONT = FontFactory.HELVETICA;

  /** Instance of the DocumentConfig */
  private static DocumentConfig documentConfig;

  /** Constant for the Color of a Link */
  private static final Color LINK_COLOR = new Color(0, 0, 153);

  private BaseFont baseFont;
  private Font boldFont;
  private Font italicFont;
  private Font linkFont;
  private Font normalFont;
  private Font smallItalicFont;
  private Font subTitleFont;

  /**
   * Instantiates a new DocumentConfig and inits all needed Fonts
   */
  private DocumentConfig() {

    /** Init a Base Font in case the OS locale is Asia */
    try {
      initBaseFont();
    }

    /** Log any error and use default Fonts then */
    catch (DocumentException e) {
      GUI.logger.log("initBaseFont()", e);
    }

    /** Log any error and use default Fonts then */
    catch (IOException e) {
      GUI.logger.log("initBaseFont()", e);
    }

    /** A baseFont was set, use it for the fonts */
    if (baseFont != null)
      initAsiaFonts();

    /** No baseFont was set, use default fonts */
    else
      initDefaultFonts();
  }

  /**
   * Get an Instance of DocumentConfig
   * 
   * @return DocumentConfig An instance of DocumentConfig
   */
  public static DocumentConfig getInstance() {
    if (documentConfig != null)
      return documentConfig;

    documentConfig = new DocumentConfig();
    return documentConfig;
  }

  /**
   * Get the Bold Font
   * 
   * @return Font The bold Font
   */
  public Font getBoldFont() {
    return boldFont;
  }

  /**
   * Get the Italic Font
   * 
   * @return Font The italic Font
   */
  public Font getItalicFont() {
    return italicFont;
  }

  /**
   * Get the Font to use for Links
   * 
   * @return Font The link Font
   */
  public Font getLinkFont() {
    return linkFont;
  }

  /**
   * Get the Normal Font
   * 
   * @return Font The normal Font
   */
  public Font getNormalFont() {
    return normalFont;
  }

  /**
   * Get the Small Italic Font
   * 
   * @return Font the small italic Font
   */
  public Font getSmallItalicFont() {
    return smallItalicFont;
  }

  /**
   * Get the Font to use for the Sub Title
   * 
   * @return Font the Sub Title font
   */
  public Font getSubTitleFont() {
    return subTitleFont;
  }

  /**
   * Init asia Fonts in case the OS has an Asia locale
   */
  private void initAsiaFonts() {
    normalFont = new Font(baseFont, 12, Font.NORMAL);
    italicFont = new Font(baseFont, 12, Font.ITALIC);
    smallItalicFont = new Font(baseFont, 10, Font.ITALIC);
    boldFont = new Font(baseFont, 12, Font.BOLD);
    subTitleFont = new Font(baseFont, 10, Font.NORMAL, Color.GRAY);
    linkFont = new Font(baseFont, 12, Font.BOLD, LINK_COLOR);
  }

  /**
   * Init the base Font. The base font will only be set in case the OS' locale
   * is Asian.
   * 
   * @throws DocumentException In case an error occurs
   * @throws IOException In case an error occurs
   */
  private void initBaseFont() throws DocumentException, IOException {
    Locale locale = Locale.getDefault();

    /** Simplified Chinese */
    if (locale.equals(Locale.SIMPLIFIED_CHINESE))
      baseFont = BaseFont.createFont("STSong-Light", "UniGB-UCS2-H", BaseFont.NOT_EMBEDDED);

    /** Traditional Chinese */
    else if (locale.equals(Locale.TRADITIONAL_CHINESE))
      baseFont = BaseFont.createFont("MSung-Light", "UniCNS-UCS2-H", BaseFont.NOT_EMBEDDED);

    /** Japan */
    else if (locale.getLanguage().equalsIgnoreCase(Locale.JAPAN.getLanguage()))
      baseFont = BaseFont.createFont("HeiseiKakuGo-W5", "UniJIS-UCS2-H", BaseFont.NOT_EMBEDDED);

    /** Korean */
    else if (locale.getLanguage().equalsIgnoreCase(Locale.KOREAN.getLanguage()))
      baseFont = BaseFont.createFont("HYGoThic-Medium", "UniKS-UCS2-H", BaseFont.NOT_EMBEDDED);
  }

  /**
   * Init default Fonts in case the OS has not an Asian locale
   */
  private void initDefaultFonts() {
    normalFont = FontFactory.getFont(DEFAULT_FONT, 12, Font.NORMAL);
    italicFont = FontFactory.getFont(DEFAULT_FONT, 12, Font.ITALIC);
    smallItalicFont = FontFactory.getFont(DEFAULT_FONT, 10, Font.ITALIC);
    boldFont = FontFactory.getFont(DEFAULT_FONT, 12, Font.BOLD);
    subTitleFont = FontFactory.getFont(DEFAULT_FONT, 10, Font.NORMAL, Color.GRAY);
    linkFont = FontFactory.getFont(DEFAULT_FONT, 12, Font.BOLD, LINK_COLOR);
  }
}