/*   **********************************************************************  **
 **   Copyright notice                                                       **
 **                                                                          **
 **   (c) 2003-2006 RSSOwl Development Team                                  **
 **   http://www.rssowl.org/                                                 **
 **                                                                          **
 **   All rights reserved                                                    **
 **                                                                          **
 **   This program and the accompanying materials are made available under   **
 **   the terms of the Eclipse Public License 1.0 which accompanies this     **
 **   distribution, and is available at:                                     **
 **   http://www.rssowl.org/legal/epl-v10.html                               **
 **                                                                          **
 **   A copy is found in the file epl-v10.html and important notices to the  **
 **   license from the team is found in the textfile LICENSE.txt distributed **
 **   in this package.                                                       **
 **                                                                          **
 **   This copyright notice MUST APPEAR in all copies of the file!           **
 **                                                                          **
 **   Contributors:                                                          **
 **     RSSOwl - initial API and implementation (bpasero@rssowl.org)         **
 **                                                                          **
 **  **********************************************************************  */

package net.sourceforge.rssowl.util.i18n;

import net.sourceforge.rssowl.util.shop.StringShop;

import java.util.Hashtable;
import java.util.Locale;

/**
 * Class to parse a language from a Locale that is supported by RSSOwl. This is
 * usefull if the locale of a user holds a dialect of a language that is
 * supported.
 * 
 * @author <a href="mailto:bpasero@rssowl.org">Benjamin Pasero </a>
 * @version 1.2.3
 */
public class LanguageDetector {

  /** Hashtable with ISO countrycodes */
  private static Hashtable countryMap = new Hashtable();

  /** This utility class constructor is hidden */
  private LanguageDetector() {
  // Protect default constructor
  }

  /**
   * Try parse a language from the given locale that is supported in RSSOwl.
   * Otherwise return default language english.
   * 
   * @param locale Selected locale
   * @return String The language to use
   */
  public static String parseLanguage(Locale locale) {
    String language = locale.getLanguage().toLowerCase();
    String country = locale.getCountry();

    /** Case: Language given and translation is existing */
    if (StringShop.isset(language) && Dictionary.languages.contains(language))
      return language;

    /** Case: Language not given but country */
    if (StringShop.isset(country) && countryMap.containsKey(country))
      return (String) countryMap.get(country);

    /** User is not from ZH or TW but speaks Chinese */
    if (language != null && language.equals("zh"))
      return "zhcn";

    /** Return default language */
    return "en";
  }

  /** Map countrys to languages that are supported in RSSOwl */
  static {

    /** English mapping */
    countryMap.put("UK", "en"); // Map united kingdom to english
    countryMap.put("US", "en"); // Map united states to english

    /** Danish mapping */
    countryMap.put("DK", "da"); // Map danmark to danish

    /** German mapping */
    countryMap.put("CH", "de"); // Map switzerland to german
    countryMap.put("AT", "de"); // Map austria to german
    countryMap.put("LI", "de"); // Map liechtenstein to german

    /** Greek mapping */
    countryMap.put("GR", "el"); // Map greece to greek

    /** Spanish mapping */
    countryMap.put("AR", "es"); // Map argentina to spanish
    countryMap.put("BO", "es"); // Map bolivia to spanish
    countryMap.put("CL", "es"); // Map chile to spanish
    countryMap.put("CO", "es"); // Map colombia to spanish
    countryMap.put("CU", "es"); // Map cuba to spanish
    countryMap.put("MX", "es"); // Map mexico to spanish
    countryMap.put("PA", "es"); // Map panama to spanish
    countryMap.put("PY", "es"); // Map paraguay to spanish
    countryMap.put("UY", "es"); // Map uruguay to spanish
    countryMap.put("VE", "es"); // Map venezuela to spanish

    /** French mapping */
    countryMap.put("BJ", "fr"); // Map benin to french
    countryMap.put("BF", "fr"); // Map burkina faso to french
    countryMap.put("CF", "fr"); // Map central afr. rep. to french
    countryMap.put("GN", "fr"); // Map guinea to french
    countryMap.put("ML", "fr"); // Map mali to french
    countryMap.put("MC", "fr"); // Map monaco to french
    countryMap.put("SN", "fr"); // Map senegal to french
    countryMap.put("LU", "fr"); // Map luxembourg to french

    /** Italian mapping */
    countryMap.put("VA", "it"); // Map vatican city to italian

    /** Dutch mapping */
    countryMap.put("BE", "nl"); // Map belgium to dutch

    /** Portuguese mapping */
    countryMap.put("BR", "pt"); // Map brazil to portuguese
    countryMap.put("CV", "pt"); // Map cape verde to portuguese
    countryMap.put("MZ", "pt"); // Map mozambique to portuguese
    countryMap.put("AO", "pt"); // Map angola to portuguese

    /** Russian mapping */
    countryMap.put("BY", "ru"); // Map belarus to russian
    countryMap.put("UZ", "ru"); // Map uzbekistan to russian
    countryMap.put("GE", "ru"); // Map georgia to russian
    countryMap.put("KZ", "ru"); // Map kazakhstan to russian
    countryMap.put("TM", "ru"); // Map turkmenistan to russian

    /** Chinese (simplified) mapping */
    countryMap.put("CN", "zhcn"); // Map cn to chinese (simplified)

    /** Chinese (traditional) mapping */
    countryMap.put("TW", "zhtw"); // Map tw to chinese (traditional)

    /** Japanese mapping */
    countryMap.put("JP", "ja"); // Map japan to japanese

    /** Korean mapping */
    countryMap.put("KR", "ko"); // Map north korea to korean
    countryMap.put("KP", "ko"); // Map south korea to korean

    /** Bangladesh mapping */
    countryMap.put("BD", "bn"); // Map bangladesh to bengali

    /** Swedish mapping */
    countryMap.put("SE", "sv"); // Map sweden to swedish

    /** Ukrania mapping */
    countryMap.put("UA", "uk"); // Map ukrania to ukrainian

    /** Czech mapping */
    countryMap.put("CZ", "cs"); // Map czech republic to czech

    /** Slovenian mapping */
    countryMap.put("SI", "sl"); // Map Slovenia to Slovenian

    /** Turkish mapping */
    countryMap.put("TR", "tr"); // Map Turkey to Turkish

    /** Hungarian mapping */
    countryMap.put("HU", "hu"); // Map Hungary to Hungarian

    /** Thai mapping */
    countryMap.put("TH", "th"); // Map Thailand to Thai

    /** Serbian Cyrillic mapping */
    countryMap.put("CS", "sr"); // Map Serbia to Serbian
    countryMap.put("MK", "sr"); // Map Macedonia to Serbian

    /** Serbian Latin mapping */
    countryMap.put("BA", "sh"); // Map Bosnia-Herzegovina to Serbian Latin
    countryMap.put("YU", "sh"); // Map Jugoslavia to Serbian Latin
    countryMap.put("CS@Latn", "sh"); // Map Serbia Latin to Serbian Latin
  }
}