/*   **********************************************************************  **
 **   Copyright notice                                                       **
 **                                                                          **
 **   (c) 2003-2006 RSSOwl Development Team                                  **
 **   http://www.rssowl.org/                                                 **
 **                                                                          **
 **   All rights reserved                                                    **
 **                                                                          **
 **   This program and the accompanying materials are made available under   **
 **   the terms of the Eclipse Public License 1.0 which accompanies this     **
 **   distribution, and is available at:                                     **
 **   http://www.rssowl.org/legal/epl-v10.html                               **
 **                                                                          **
 **   A copy is found in the file epl-v10.html and important notices to the  **
 **   license from the team is found in the textfile LICENSE.txt distributed **
 **   in this package.                                                       **
 **                                                                          **
 **   This copyright notice MUST APPEAR in all copies of the file!           **
 **                                                                          **
 **   Contributors:                                                          **
 **     RSSOwl - initial API and implementation (bpasero@rssowl.org)         **
 **                                                                          **
 **  **********************************************************************  */

package net.sourceforge.rssowl.util.i18n;

import net.sourceforge.rssowl.controller.GUI;

import java.lang.reflect.Constructor;
import java.util.HashMap;
import java.util.Locale;

/**
 * Internationalization for RSSOwl (i18n). <br />
 * Supported Languages <br />
 * ------------------- <br />
 * <br />
 * o German - by Benjamin P. RSSOwl Administrator / Developer <br />
 * o English - by Benjamin P. RSSOwl Administrator / Developer <br />
 * o French - by Christophe Dumez RSSOwl Doc Translator (French) <br />
 * o Spanish - by José Domínguez and Ruben <br />
 * o Galician - by José Domínguez <br />
 * o Danish - by Tonny Bredsgaard RSSOwl Doc Translator (Danish) <br />
 * o Italian - by Claudio Fontana <br />
 * o Dutch - by Joris Kluivers and Toon Geens <br />
 * o Greek - by Jacaranda Bill <br />
 * o Russian - by Sergey Rozenblat and Alexandr <br />
 * o Portugues (Brasil) - by Marcelo Fenoll Ramal Tradutor <br />
 * o Bulgarian - by Valeri Damianov <br />
 * o Norwegian (Bokmål) - by Eivind Syverts <br />
 * o Chinese Simplified - by Merlin Ran <br />
 * o Japanese - by Takashi Komatsubara <br />
 * o Korean - by Geon Goo <br />
 * o Polish - by Ryszard Sierotnik <br />
 * o Bangla (Bengali) - by Omi Azad <br />
 * o Swedish - by Patrik Johansson <br />
 * o Chinese Traditional - by Jerry Giant <br />
 * o Finnish - by Samu Reinikainen <br />
 * o Ukrainian - by Alexandr <br />
 * o Czech - by Lukas Petrovicky <br />
 * o Slovenian - by Jure Zemljič <br />
 * o Turkish - by Engin Erenturk <br />
 * o Hungarian - by Balázs <br />
 * o Thai - by Patipat Susumpow <br />
 * o Croatian - by Neven Zitek<br />
 * o Slovak - by Vlado Jasaň<br />
 * o Armenian - by Vardan Movsisyan<br />
 * <br />
 * If you would like to add another language to RSSOwl, feel free to contact me
 * (http://www.rssowl.org).
 * 
 * @author <a href="mailto:bpasero@rssowl.org">Benjamin Pasero </a>
 * @version 1.2.3
 */
public class RSSOwlI18n extends Dictionary {

  /** Default translation for RSSOwl is english */
  private static RSSOwlI18nEN defaultTranslation;

  /** Hashtable with: KEY => Translated Value */
  private HashMap i18n;

  /**
   * Initialize i18n with given language
   * 
   * @param locale Selected locale
   */
  public RSSOwlI18n(Locale locale) {
    String language = locale.getLanguage();

    /** Skip "en" as it is the default language and already instantiated */
    if (language != null && !language.equals("en") && translations.containsKey(language)) {
      try {
        String country = "";

        /** Get the country from the locales */
        if (locales.containsKey(language))
          country = ((Locale) locales.get(language)).getCountry();

        /** Create a new I18N class from the given language */
        Class languageClass = Class.forName((String) translations.get(language));
        Constructor constructor = languageClass.getConstructor(new Class[] { String.class, String.class });
        i18n = ((Translation) constructor.newInstance(new Object[] { language, country })).getTranslation();

        /** Set dictionary to new language */
        Dictionary.selectedLanguage = language;

        /** Set selected locale to given locale */
        selectedLocale = new Locale(language, country);
      }

      /** The I18N class could not be found, use default language */
      catch (Exception e) {
        GUI.logger.log("RSSOwlI18n()", e);
        switchToDefaultTranslation();
      }
    }

    /** Language is English */
    else {
      switchToDefaultTranslation();
    }
  }

  /**
   * Get the Default Translation, which is English. The Translation is created
   * once, if not yet created before.
   * 
   * @return The default English Translation.
   */
  public static Translation getDefaultTranslation() {
    if (defaultTranslation == null)
      defaultTranslation = new RSSOwlI18nEN("en", "EN");
    return defaultTranslation;
  }

  /**
   * Return the translated word for the given key or the english translation if
   * no translation was found. If the english translation is missing too, return
   * the key.
   * 
   * @param key unique key
   * @return String translated word in the selected language
   */
  public String getTranslation(String key) {
    Object translation = i18n.get(key);

    if (translation != null)
      return (String) translation;
    else if (getDefaultTranslation().getTranslation().get(key) != null)
      return (String) getDefaultTranslation().getTranslation().get(key);
    else
      return key;
  }

  /**
   * Use english as default translation
   */
  private void switchToDefaultTranslation() {
    Dictionary.selectedLanguage = "en";
    i18n = getDefaultTranslation().getTranslation();
    selectedLocale = new Locale("en");
  }

  /** Register supported languages to the Dictionary */
  static {
    String packageName = "net.sourceforge.rssowl.util.i18n.";

    /** German */
    Dictionary.register("de", "DE", true, packageName + "RSSOwlI18nDE");

    /** English */
    Dictionary.register("en", "EN", false, packageName + "RSSOwlI18nEN");

    /** French */
    Dictionary.register("fr", "FR", true, packageName + "RSSOwlI18nFR");

    /** Italian */
    Dictionary.register("it", "IT", true, packageName + "RSSOwlI18nIT");

    /** Danish */
    Dictionary.register("da", "DK", true, packageName + "RSSOwlI18nDA");

    /** Norwegian */
    Dictionary.register("no", "NO", true, packageName + "RSSOwlI18nNO");

    /** Dutch */
    Dictionary.register("nl", "NL", true, packageName + "RSSOwlI18nNL");

    /** Swedish */
    Dictionary.register("sv", "SE", true, packageName + "RSSOwlI18nSV");

    /** Finnish */
    Dictionary.register("fi", "FI", true, packageName + "RSSOwlI18nFI");

    /** Spanish */
    Dictionary.register("es", "ES", true, packageName + "RSSOwlI18nES");

    /** Galician */
    Dictionary.register("gl", "GL", false, packageName + "RSSOwlI18nGL");

    /** Polish */
    Dictionary.register("pl", "PL", true, packageName + "RSSOwlI18nPL");

    /** Greek */
    Dictionary.register("el", "GR", true, packageName + "RSSOwlI18nEL");

    /** Russian */
    Dictionary.register("ru", "RU", true, packageName + "RSSOwlI18nRU");

    /** Ukrainian */
    Dictionary.register("uk", "UA", false, packageName + "RSSOwlI18nUK");

    /** Bengali */
    Dictionary.register("bn", "BD", false, packageName + "RSSOwlI18nBN");

    /** Portuguese (Brazil) */
    Dictionary.register("pt", "BR", false, packageName + "RSSOwlI18nPT");

    /** Bulgarian */
    Dictionary.register("bg", "BG", false, packageName + "RSSOwlI18nBG");

    /** Simplified Chinese */
    Dictionary.register("zhcn", "CN", false, packageName + "RSSOwlI18nZHcn");

    /** Japanese */
    Dictionary.register("ja", "JP", true, packageName + "RSSOwlI18nJA");

    /** Traditional Chinese */
    Dictionary.register("zhtw", "TW", false, packageName + "RSSOwlI18nZHtw");

    /** Korean */
    Dictionary.register("ko", "KR", false, packageName + "RSSOwlI18nKO");

    /** Thai */
    Dictionary.register("th", "TH", false, packageName + "RSSOwlI18nTH");

    /** Turkish */
    Dictionary.register("tr", "TR", false, packageName + "RSSOwlI18nTR");

    /** Armenian */
    Dictionary.register("hy", "AM", true, packageName + "RSSOwlI18nHY");

    /** Czech */
    Dictionary.register("cs", "CZ", false, packageName + "RSSOwlI18nCS");

    /** Serbian (Cyrillic) */
    Dictionary.register("sr", "CS", false, packageName + "RSSOwlI18nSR");

    /** Serbian (Latin) */
    Dictionary.register("sh", "CS@Latn", false, packageName + "RSSOwlI18nSH");

    /** Croatian */
    Dictionary.register("hr", "HR", true, packageName + "RSSOwlI18nHR");

    /** Hungarian */
    Dictionary.register("hu", "HU", false, packageName + "RSSOwlI18nHU");

    /** Slovak */
    Dictionary.register("sk", "SK", true, packageName + "RSSOwlI18nSK");

    /** Slovenian */
    Dictionary.register("sl", "SI", false, packageName + "RSSOwlI18nSL");
  }
}