/*   **********************************************************************  **
 **   Copyright notice                                                       **
 **                                                                          **
 **   (c) 2003-2006 RSSOwl Development Team                                  **
 **   http://www.rssowl.org/                                                 **
 **                                                                          **
 **   All rights reserved                                                    **
 **                                                                          **
 **   This program and the accompanying materials are made available under   **
 **   the terms of the Eclipse Public License 1.0 which accompanies this     **
 **   distribution, and is available at:                                     **
 **   http://www.rssowl.org/legal/epl-v10.html                               **
 **                                                                          **
 **   A copy is found in the file epl-v10.html and important notices to the  **
 **   license from the team is found in the textfile LICENSE.txt distributed **
 **   in this package.                                                       **
 **                                                                          **
 **   This copyright notice MUST APPEAR in all copies of the file!           **
 **                                                                          **
 **   Contributors:                                                          **
 **     RSSOwl - initial API and implementation (bpasero@rssowl.org)         **
 **                                                                          **
 **  **********************************************************************  */

package net.sourceforge.rssowl.util.i18n;

import java.util.HashMap;
import java.util.Locale;

/**
 * Translation is the Super-Class for all i18n translation classes in RSSOwl. It
 * provides the HashMap, to store the translations. A locale indicates for what
 * locales the translation should be used.
 * 
 * @author <a href="mailto:bpasero@rssowl.org">Benjamin Pasero </a>
 * @version 1.2.3
 */
public abstract class Translation {

  /**
   * The initial capacity of the Dictionaries HashMap. Since the number of words
   * is known, it is possible to calculate the optimal capacity.<br>
   * <br>
   * In order to avoid calls to rehash() while adding Elements, the initial
   * capacity must be set to a value bigger than Entries divided by Load Factor.<br>
   * <br>
   * Current Wording Count: 454<br>
   * Default Load Factor: 0.75<br>
   * Optimal Initial Cap: 454/0.75 ~= 606
   */
  private static final int HASHMAP_INITIAL_CAP = 606;

  /** Locale of the translation */
  private Locale locale;

  /**
   * The HashMap maps i18n-keys to translated values.
   */
  protected HashMap translation;

  /**
   * Instantiate a new translation for a locale
   * 
   * @param language two-letter ISO-639 code
   * @param country two-letter ISO-3166 code
   */
  protected Translation(String language, String country) {
    translation = new HashMap(HASHMAP_INITIAL_CAP);
    locale = new Locale(language, country);
    initTranslation();
  }

  /**
   * Get the Hashmap with the translation
   * 
   * @return Returns the translation.
   */
  public HashMap getTranslation() {
    return translation;
  }

  /**
   * @see java.lang.Object#toString()
   */
  public String toString() {
    return locale.toString();
  }

  /**
   * Fill the HashMap with translations
   */
  protected abstract void initTranslation();
}