/*   **********************************************************************  **
 **   Copyright notice                                                       **
 **                                                                          **
 **   (c) 2003-2006 RSSOwl Development Team                                  **
 **   http://www.rssowl.org/                                                 **
 **                                                                          **
 **   All rights reserved                                                    **
 **                                                                          **
 **   This program and the accompanying materials are made available under   **
 **   the terms of the Eclipse Public License 1.0 which accompanies this     **
 **   distribution, and is available at:                                     **
 **   http://www.rssowl.org/legal/epl-v10.html                               **
 **                                                                          **
 **   A copy is found in the file epl-v10.html and important notices to the  **
 **   license from the team is found in the textfile LICENSE.txt distributed **
 **   in this package.                                                       **
 **                                                                          **
 **   This copyright notice MUST APPEAR in all copies of the file!           **
 **                                                                          **
 **   Contributors:                                                          **
 **     RSSOwl - initial API and implementation (bpasero@rssowl.org)         **
 **                                                                          **
 **  **********************************************************************  */

package net.sourceforge.rssowl.util.search;

/**
 * A SearchDefinition describes a Pattern to search in a certain Scope.
 * 
 * @author <a href="mailto:bpasero@rssowl.org">Benjamin Pasero </a>
 * @version 1.2.3
 */
public class SearchDefinition {

  /** Default indicator for no search */
  public static final SearchDefinition NO_SEARCH = new SearchDefinition(null, -1);

  /** Scope: Author */
  public static final int SCOPE_AUTHOR = 5;

  /** Scope: Category */
  public static final int SCOPE_CATEGORY = 4;

  /** Scope: Description */
  public static final int SCOPE_DESCRIPTION = 2;

  /** Scope: Enclosure */
  public static final int SCOPE_ENCLOSURE = 6;

  /** Human Readable Scope-Names */
  public static final String[] SCOPE_NAMES = new String[] { "ENTIRE_NEWS", "TABLE_HEADER_NEWSTITLE", "LABEL_DESCRIPTION", "TABLE_HEADER_PUBDATE", "TABLE_HEADER_CATEGORY", "TABLE_HEADER_AUTHOR", "NEWS_ITEM_INFO_ENCLOSURE", "TABLE_HEADER_FEEDTITLE" };

  /** Scope: Entire News */
  public static final int SCOPE_NEWS = 0;

  /** Scope: Newsfeed Title */
  public static final int SCOPE_NEWSFEED = 7;

  /** Scope: Publish Date */
  public static final int SCOPE_PUBDATE = 3;

  /** Scope: Title */
  public static final int SCOPE_TITLE = 1;

  private boolean isCaseSensitive;
  private boolean isPatternRegEx;
  private boolean isWholeWord;
  private String pattern;
  private int scope;

  /**
   * Instantiate a new SearchDefinition.
   * 
   * @param pattern The Pattern to search.
   * @param scope The Scope of the search.
   */
  public SearchDefinition(String pattern, int scope) {
    this.pattern = pattern;
    this.scope = scope;
    isPatternRegEx = false;
  }

  /**
   * <p>
   * This implementation of equals() is not checking all fields, because only
   * the Pattern is currently relevent in the SearchDialog, where a List of
   * searches is remembered.
   * </p>
   * 
   * @see java.lang.Object#equals(java.lang.Object)
   */
  public boolean equals(Object obj) {

    /** Not a SearchDefinition instance */
    if (obj == null || !(obj instanceof SearchDefinition))
      return false;

    /** Consider equal in case the pattern is equal */
    return getPattern().equals(((SearchDefinition) obj).getPattern());
  }

  /**
   * Get the Pattern to use for the search.
   * 
   * @return The Pattern to search.
   */
  public String getPattern() {
    return pattern;
  }

  /**
   * Get the Scope to use for the search.
   * 
   * @return The Scope of the search.
   */
  public int getScope() {
    return scope;
  }

  /**
   * @see java.lang.Object#hashCode()
   */
  public int hashCode() {
    return getPattern() != null ? getPattern().hashCode() : super.hashCode();
  }

  /**
   * Check wether this Search is Case Sensitive.
   * 
   * @return TRUE if the Search is Case Sensitive.
   */
  public boolean isCaseSensitive() {
    return isCaseSensitive;
  }

  /**
   * Get wether the Pattern is a Regular Expression.
   * 
   * @return boolean TRUE if the Pattern should be used as Regular Expression
   * directly.
   */
  public boolean isPatternRegEx() {
    return isPatternRegEx;
  }

  /**
   * Check if Search should only consider Whole Words.
   * 
   * @return TRUE if Search should only consider Whole Words.
   */
  public boolean isWholeWord() {
    return isWholeWord;
  }

  /**
   * Set wether this Search is Case Sensitive
   * 
   * @param isCaseSensitive TRUE for Case Sensitive Search.
   */
  public void setIsCaseSensitive(boolean isCaseSensitive) {
    this.isCaseSensitive = isCaseSensitive;
  }

  /**
   * Set wether Search should only consider Whole Words.
   * 
   * @param isWholeWord TRUE for Whole Words Search.
   */
  public void setIsWholeWord(boolean isWholeWord) {
    this.isWholeWord = isWholeWord;
  }

  /**
   * Set wether the Pattern is a Regular Expression.
   * 
   * @param isPatternRegEx Set TRUE if the Pattern should be used as Regular
   * Expression directly.
   */
  public void setPatternIsRegEx(boolean isPatternRegEx) {
    this.isPatternRegEx = isPatternRegEx;
  }
}