/*   **********************************************************************  **
 **   Copyright notice                                                       **
 **                                                                          **
 **   (c) 2003-2006 RSSOwl Development Team                                  **
 **   http://www.rssowl.org/                                                 **
 **                                                                          **
 **   All rights reserved                                                    **
 **                                                                          **
 **   This program and the accompanying materials are made available under   **
 **   the terms of the Eclipse Public License 1.0 which accompanies this     **
 **   distribution, and is available at:                                     **
 **   http://www.rssowl.org/legal/epl-v10.html                               **
 **                                                                          **
 **   A copy is found in the file epl-v10.html and important notices to the  **
 **   license from the team is found in the textfile LICENSE.txt distributed **
 **   in this package.                                                       **
 **                                                                          **
 **   This copyright notice MUST APPEAR in all copies of the file!           **
 **                                                                          **
 **   Contributors:                                                          **
 **     RSSOwl - initial API and implementation (bpasero@rssowl.org)         **
 **                                                                          **
 **  **********************************************************************  */

package net.sourceforge.rssowl.util.shop;

import net.sourceforge.rssowl.controller.GUI;
import net.sourceforge.rssowl.util.i18n.Dictionary;

import org.eclipse.jface.resource.JFaceResources;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.graphics.FontData;

/**
 * Factory class for some Font / -style concerns in RSSOwl
 * 
 * @author <a href="mailto:bpasero@rssowl.org">Benjamin Pasero </a>
 * @version 1.2.3
 */
public class FontShop {

  /** Dialog bold font */
  public static Font dialogBoldFont = null;

  /** Font for the Dialogs */
  public static Font dialogFont = createFont();

  /** Bold Font for the ViewForm headers and TabItem titles */
  public static Font headerBoldFont = null;

  /** Font for the ViewForm headers and TabItem titles */
  public static Font headerFont = createFont(getFontData()[0].getHeight() + 1);

  /** Bold Font for the table holding news */
  public static Font tableBoldFont = null;

  /** Font for the table holding news */
  public static Font tableFont = createFont();

  /** Bold font for the Text */
  public static Font textBoldFont = null;

  /** Font for the newstext */
  public static Font textFont = createFont(getFontData()[0].getHeight() + 1);

  /** Font for the tree holding favorites (Bold) */
  public static Font treeBoldFont = null;

  /** Font for the tree holding favorites */
  public static Font treeFont = createFont();

  /** This utility class constructor is hidden */
  private FontShop() {
  // Protect default constructor
  }

  /**
   * Create a Font using the system font
   * 
   * @return Font The new Font from the system font
   */
  public static Font createFont() {
    FontData[] fontData = getFontData();
    return createFont(fontData[0].getName(), fontData[0].getHeight(), 0);
  }

  /**
   * Create a Font from the given height
   * 
   * @param height The height of the new Font
   * @return Font The new Font from the given Font information
   */
  public static Font createFont(int height) {
    FontData[] fontData = getFontData();
    return createFont(fontData[0].getName(), height, 0);
  }

  /**
   * Create a Font from the given height and style
   * 
   * @param height The height of the new Font
   * @param style The style of the new Font
   * @return Font The new Font from the given Font information
   */
  public static Font createFont(int height, int style) {
    FontData[] fontData = getFontData();
    return createFont(fontData[0].getName(), height, style);
  }

  /**
   * Create a Font from the given name, height and style
   * 
   * @param name The name of the new Font
   * @param height The height of the new Font
   * @param style The style of the new Font
   * @return Font The new Font from the given Font information
   */
  public static Font createFont(String name, int height, int style) {
    FontData[] fontData = getFontData();
    for (int a = 0; a < fontData.length; a++) {
      fontData[a].setName(name);
      fontData[a].setHeight(height);
      fontData[a].setLocale(Dictionary.selectedLocale.toString());

      /** Normal is already default */
      if (style != SWT.NORMAL)
        fontData[a].setStyle(style);
    }
    return new Font(GUI.display, fontData);
  }

  /** Dispose fonts */
  public static void disposeFonts() {
    if (isset(dialogBoldFont))
      dialogBoldFont.dispose();

    if (isset(dialogFont))
      dialogFont.dispose();

    if (isset(headerFont))
      headerFont.dispose();

    if (isset(tableFont))
      tableFont.dispose();

    if (isset(tableBoldFont))
      tableBoldFont.dispose();

    if (isset(textBoldFont))
      textBoldFont.dispose();

    if (isset(textFont))
      textFont.dispose();

    if (isset(treeFont))
      treeFont.dispose();

    if (isset(treeBoldFont))
      treeBoldFont.dispose();

    if (isset(headerBoldFont))
      headerBoldFont.dispose();
  }

  /**
   * Get the FontData array using the given Font information on the system font
   * fontdata
   * 
   * @return FontData The FontData from the system font
   */
  public static FontData[] getFontData() {
    return GUI.display.getSystemFont().getFontData();
  }

  /**
   * Get the FontData array using the given Font information on the system font
   * fontdata
   * 
   * @param height The height of the Font
   * @return FontData The FontData array using the given Font information
   */
  public static FontData[] getFontData(int height) {
    FontData[] fontData = getFontData();
    return getFontData(fontData[0].getName(), height, 0);
  }

  /**
   * Get the FontData array using the given Font information on the system font
   * fontdata
   * 
   * @param name The name of the Font
   * @param height The height of the Font
   * @param style The style of the Font
   * @return FontData The FontData array using the given Font information
   */
  public static FontData[] getFontData(String name, int height, int style) {
    FontData[] fontData = getFontData();
    for (int a = 0; a < fontData.length; a++) {
      fontData[a].setName(name);
      fontData[a].setHeight(height);
      fontData[a].setLocale(Dictionary.selectedLocale.toString());

      /** Normal is already default */
      if (style != SWT.NORMAL)
        fontData[a].setStyle(style);
    }
    return fontData;
  }

  /**
   * Check the given Font for being NULL or disposed. Return false in that case.
   * 
   * @param font The font to check
   * @return boolean TRUE if the font is available
   */
  public static boolean isset(Font font) {
    return (font != null && !font.isDisposed());
  }

  /**
   * Update fonts for all controls in RSSOwl
   * 
   * @param rssOwlGui The maincontroller
   */
  public static void setFontForAll(GUI rssOwlGui) {
    rssOwlGui.getRSSOwlQuickview().updateFonts();
    rssOwlGui.getRSSOwlFavoritesTree().updateFonts();
    rssOwlGui.getRSSOwlNewsTabFolder().updateFonts();
    rssOwlGui.getRSSOwlNewsText().updateFonts();
    rssOwlGui.getRSSOwlStatusLine().updateFonts();

    /** Update Fonts for all UI components */
    LayoutShop.setLayoutForAll(GUI.shell);
  }

  /**
   * Update styled fonts and the dialog font used by JFace dialogs
   */
  public static void updateFonts() {
    initStyledFonts();
    JFaceResources.getFontRegistry().put(JFaceResources.DIALOG_FONT, FontShop.dialogFont.getFontData());
  }

  /**
   * Create a new Font using the given one and making it Bold
   * 
   * @param sourceFont The source Font object
   * @return Font The Font in Bold
   */
  private static Font createBoldFont(Font sourceFont) {
    FontData[] fontData = sourceFont.getFontData();

    for (int a = 0; a < fontData.length; a++) {
      fontData[a].setStyle(SWT.BOLD);
      fontData[a].setLocale(Dictionary.selectedLocale.toString());
    }
    return new Font(GUI.display, fontData);
  }

  /**
   * Init all styled (bold) fonts
   */
  private static void initStyledFonts() {
    dialogBoldFont = createBoldFont(dialogFont);
    tableBoldFont = createBoldFont(tableFont);
    textBoldFont = createBoldFont(textFont);
    treeBoldFont = createBoldFont(treeFont);
    headerBoldFont = createBoldFont(headerFont);
  }
}