/*   **********************************************************************  **
 **   Copyright notice                                                       **
 **                                                                          **
 **   (c) 2003-2006 RSSOwl Development Team                                  **
 **   http://www.rssowl.org/                                                 **
 **                                                                          **
 **   All rights reserved                                                    **
 **                                                                          **
 **   This program and the accompanying materials are made available under   **
 **   the terms of the Eclipse Public License 1.0 which accompanies this     **
 **   distribution, and is available at:                                     **
 **   http://www.rssowl.org/legal/epl-v10.html                               **
 **                                                                          **
 **   A copy is found in the file epl-v10.html and important notices to the  **
 **   license from the team is found in the textfile LICENSE.txt distributed **
 **   in this package.                                                       **
 **                                                                          **
 **   This copyright notice MUST APPEAR in all copies of the file!           **
 **                                                                          **
 **   Contributors:                                                          **
 **     RSSOwl - initial API and implementation (bpasero@rssowl.org)         **
 **                                                                          **
 **  **********************************************************************  */

package net.sourceforge.rssowl.util.shop;

import net.sourceforge.rssowl.controller.GUI;
import net.sourceforge.rssowl.controller.NewsText;
import net.sourceforge.rssowl.controller.panel.BrowserPanel;

import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.StyleRange;
import org.eclipse.swt.custom.StyledText;
import org.eclipse.swt.custom.StyledTextPrintOptions;
import org.eclipse.swt.printing.PrintDialog;
import org.eclipse.swt.printing.Printer;
import org.eclipse.swt.printing.PrinterData;
import org.eclipse.swt.widgets.Shell;

/**
 * Factory class for some Print concerns in RSSOwl
 * 
 * @author <a href="mailto:bpasero@rssowl.org">Benjamin Pasero </a>
 * @version 1.2.3
 */
public class PrintShop {

  /** This utility class constructor is hidden */
  private PrintShop() {
  // Protect default constructor
  }

  /**
   * Method for calling a PrintDialog and printing a news (title, url, text) to
   * the selected printer.
   * 
   * @param title Title of the news
   * @param text Text / Description of the news
   */
  public static void printNews(String title, String text) {

    /** StyledText to print from */
    StyledText printText = getPrintPane(title, text);

    /** Set print options */
    StyledTextPrintOptions printOptions = new StyledTextPrintOptions();
    printOptions.header = "\t\t" + GUI.i18n.getTranslation("PRINTED_FROM_RSSOWL");
    printOptions.printTextFontStyle = true;
    printOptions.jobName = GUI.i18n.getTranslation("PRINTJOB_NAME");

    /** Print Dialog */
    PrintDialog printDialog = new PrintDialog(GUI.shell);
    PrinterData printerData = printDialog.open();

    /** Printer */
    Printer printer = new Printer(printerData);

    /** Do the printing if the user did not click on cancel */
    if (printerData != null)
      printText.print(printer, printOptions).run();

    /** Cleanup */
    printer.dispose();
  }

  /**
   * Print news out of the browser widget
   * 
   * @return boolean TRUE on success, FALSE otherwise
   */
  public static boolean printNewsFromBrowser() {

    /** Newstext to print */
    NewsText rssOwlNewsText = GUI.rssOwlGui.getRSSOwlNewsText();
    BrowserPanel browserPanel = rssOwlNewsText.getBrowserPanel();

    /**
     * In case printing via JavaScript out of the browser did not work, print
     * the plain newstext instead.
     */
    if (!browserPanel.print())
      return printNewsFromText();
    return true;
  }

  /**
   * Print the newstext
   * 
   * @return FALSE if there is nothing to print
   */
  public static boolean printNewsFromText() {

    /** Newstext to print */
    NewsText rssOwlNewsText = GUI.rssOwlGui.getRSSOwlNewsText();

    String title = "";

    /** Get Title */
    if (rssOwlNewsText.getNewsTextTitle().getText() != null)
      title = rssOwlNewsText.getNewsTextTitle().getText();

    /** Get Text */
    String text = rssOwlNewsText.getPrintableNewstext();

    /** Check for valid information */
    if (title.equals("") && text.equals(""))
      return false;

    /** Print */
    printNews(title, text);
    return true;
  }

  /**
   * Returns a StyledText that holds the news infos
   * 
   * @param title Title of the news
   * @param text Complete text of the news
   * @return StyledText holding news infos
   */
  private static StyledText getPrintPane(String title, String text) {

    /** StyledText used for the printing */
    StyledText printText = new StyledText(new Shell(), SWT.NONE);
    printText.setFont(FontShop.textFont);

    /** Set text to print */
    printText.setText(text);

    /** Stylerange for bold topic */
    StyleRange boldTopic = new StyleRange();
    boldTopic.start = 0;
    boldTopic.length = title.length();
    boldTopic.fontStyle = 1;
    printText.setStyleRange(boldTopic);

    return printText;
  }
}